package api

import (
	"context"
	"net/http"

	"code.justin.tv/feeds/errors"
	service_common "code.justin.tv/feeds/service-common"
)

func (s *HTTPServer) canCreateEvent(ctx context.Context, userID, ownerID string, channelID *string) error {
	// The call must be authenticated.
	err := s.requireCallerUserID(ctx, userID)
	if err != nil {
		return err
	}

	// For now, a user's event can only be broadcast on their own channel.
	if channelID != nil && *channelID != "" {
		if ownerID != *channelID {
			return newForbiddenError("user %s cannot create event to be broadcast on channel %s", userID, *channelID)
		}
	}

	authProps := s.AuthUsersClient.GetUserAuthProperties(ctx, userID)
	if authProps == nil {
		return newForbiddenError("user %s could not be found", userID)
	}

	// Suspended users are not allowed to create events.
	if authProps.IsSuspended {
		return newForbiddenError("suspended user %s is not allowed to create events", userID)
	}

	// A Twitch admin (support staff) can create events on behalf of anyone.
	if authProps.IsTwitchAdmin {
		return nil
	}

	// A user can create events for themselves.
	if userID == ownerID {
		return nil
	}

	// An admin can create events on behalf of anyone.
	if s.isEventsAdmin(ctx, userID) {
		return nil
	}

	// A user can create an event on behalf of the channel they are editors for.
	if s.isEditor(ctx, userID, ownerID) {
		return nil
	}

	return newForbiddenError("user %s cannot create event on behalf of user %s", userID, ownerID)
}

type canUpdateEventsParams struct {
	currentOwnerID   string
	currentChannelID *string

	// newOwnerID should set if the event's owner is being updated.
	newOwnerID *string

	// newChannelID should be set if the channel the event is being broadcast on is being updated.
	newChannelID *string
}

func (s *HTTPServer) canUpdateEvent(ctx context.Context, userID string, params *canUpdateEventsParams) error {
	// The call must be authenticated.
	err := s.requireCallerUserID(ctx, userID)
	if err != nil {
		return err
	}

	if params.newOwnerID != nil {
		err = s.canTransferOwnership(ctx, userID, params.currentOwnerID, *params.newOwnerID)
		if err != nil {
			return err
		}
	}
	if params.newChannelID != nil {
		err = s.canChangeChannel(ctx, userID, params.currentChannelID, *params.newChannelID)
		if err != nil {
			return err
		}
	}

	authProps := s.AuthUsersClient.GetUserAuthProperties(ctx, userID)
	if authProps == nil {
		return newForbiddenError("user %s could not be found", userID)
	}

	// Suspended users are not allowed to update events.
	if authProps.IsSuspended {
		return newForbiddenError("suspended user %s is not allowed to update events", userID)
	}

	// A Twitch admin (support staff) can update events on behalf of anyone.
	if authProps.IsTwitchAdmin {
		return nil
	}

	// A user can update their own events.
	if userID == params.currentOwnerID {
		return nil
	}

	// An admin can update any event.
	if s.isEventsAdmin(ctx, userID) {
		return nil
	}

	// A user can update an event on behalf of the channel they are editors for.
	if s.isEditor(ctx, userID, params.currentOwnerID) {
		return nil
	}

	return newForbiddenError("user %s is not allowed to update the event", userID)
}

func (s *HTTPServer) canViewManagedEvents(ctx context.Context, ownerID string, userID string) error {
	// The call must be authenticated.
	err := s.requireCallerUserID(ctx, userID)
	if err != nil {
		return err
	}

	authProps := s.AuthUsersClient.GetUserAuthProperties(ctx, userID)
	if authProps == nil {
		return newForbiddenError("user %s could not be found", userID)
	}

	// A Twitch admin (support staff) can see any events.
	if authProps.IsTwitchAdmin {
		return nil
	}

	// A user can see the events they own.
	if userID == ownerID {
		return nil
	}

	// Admins can see any events.
	if s.isEventsAdmin(ctx, userID) {
		return nil
	}

	// A user can see events that belong to channels they are editors of.
	if s.isEditor(ctx, userID, ownerID) {
		return nil
	}

	return newForbiddenError("user %s is not allowed to see user %s's events in a management context ", userID, ownerID)
}

func (s *HTTPServer) canTransferOwnership(ctx context.Context, userID, currentOwnerID string, newOwnerID string) error {
	// If we're not changing the owner, there's nothing to check.
	if currentOwnerID == newOwnerID {
		return nil
	}

	// Never allow clearing an event's owner.
	if newOwnerID == "" {
		return newForbiddenError("user %s is not allowed to clear the event's owner", userID)
	}

	authProps := s.AuthUsersClient.GetUserAuthProperties(ctx, userID)
	if authProps == nil {
		return newForbiddenError("user %s could not be found", userID)
	}

	// A Twitch admin (support staff) can transfer ownership of an event.
	if authProps.IsTwitchAdmin {
		return nil
	}

	// Someone on our whitelist can transfer ownership of an event.
	if s.isEventsAdmin(ctx, userID) {
		return nil
	}

	return newForbiddenError("user %s cannot change owner to %s", userID, newOwnerID)
}

func (s *HTTPServer) canChangeChannel(ctx context.Context, userID string, currentChannelID *string, newChannelID string) error {
	// If we're not changing the channel, there's nothing to check.
	if currentChannelID != nil && *currentChannelID == newChannelID {
		return nil
	}

	authProps := s.AuthUsersClient.GetUserAuthProperties(ctx, userID)
	if authProps == nil {
		return newForbiddenError("user %s could not be found", userID)
	}

	// A Twitch admin (support staff) can change the channel an event is broadcast on.
	if authProps.IsTwitchAdmin {
		return nil
	}

	// Someone on our whitelist can change the channel an event is broadcast on.
	if s.isEventsAdmin(ctx, userID) {
		return nil
	}

	return newForbiddenError("user %s is not allowed to change the channel the event is broadcast on", userID)
}

func (s *HTTPServer) canDeleteEvent(ctx context.Context, userID, ownerID string, channelID *string) error {
	// The call must be authenticated.
	err := s.requireCallerUserID(ctx, userID)
	if err != nil {
		return err
	}

	authProps := s.AuthUsersClient.GetUserAuthProperties(ctx, userID)
	if authProps == nil {
		return newForbiddenError("user %s could not be found", userID)
	}

	// Suspended users are not allowed to delete events.
	if authProps.IsSuspended {
		return newForbiddenError("suspended user %s is not allowed to delete events", userID)
	}

	// A Twitch admin (support staff) can delete any event.
	if authProps.IsTwitchAdmin {
		return nil
	}

	// A user can delete their own events.
	if userID == ownerID {
		return nil
	}

	// An admin can delete any event.
	if s.isEventsAdmin(ctx, userID) {
		return nil
	}

	// A user can delete events that belong to channels they are editors of.
	if s.isEditor(ctx, userID, ownerID) {
		return nil
	}

	return newForbiddenError("user %s is not allowed to delete event", userID)
}

func (s *HTTPServer) canAddLocalization(ctx context.Context, userID string) error {
	// The call must be authenticated.
	err := s.requireCallerUserID(ctx, userID)
	if err != nil {
		return err
	}

	// Localizations aren't fully implemented yet.  For now, only admins can create them.
	if s.isEventsAdmin(ctx, userID) {
		return nil
	}

	return newForbiddenError("user %s is not allowed to create localizations", userID)
}

func (s *HTTPServer) canUpdateLocalization(ctx context.Context, userID string) error {
	// The call must be authenticated.
	err := s.requireCallerUserID(ctx, userID)
	if err != nil {
		return err
	}

	// Localizations aren't fully implemented yet.  For now, only admins can update them.
	if s.isEventsAdmin(ctx, userID) {
		return nil
	}

	return newForbiddenError("user %s is not allowed to update the localization", userID)
}

func (s *HTTPServer) canDeleteLocalization(ctx context.Context, userID string) error {
	// The call must be authenticated.
	err := s.requireCallerUserID(ctx, userID)
	if err != nil {
		return err
	}

	// Localizations aren't fully implemented yet.  For now, only admins can delete them.
	if s.isEventsAdmin(ctx, userID) {
		return nil
	}

	return newForbiddenError("user %s is not allowed to delete the localization", userID)
}

func (s *HTTPServer) requireCallerUserID(ctx context.Context, userID string) error {
	if userID == "" {
		return newForbiddenError("cannot perform operation without the caller's user ID")
	}
	return nil
}

func newForbiddenError(format string, a ...interface{}) error {
	return &service_common.CodedError{
		Code: http.StatusForbidden,
		Err:  errors.Errorf(format, a...),
	}
}

func (s *HTTPServer) isEventsAdmin(ctx context.Context, userID string) bool {
	return s.AdminList.IsEventsAdmin(ctx, userID)
}

// isEditor returns true if userID is an editor of channelID, and false otherwise.
func (s *HTTPServer) isEditor(ctx context.Context, userID, channelID string) bool {
	isEditor, err := s.HallpassClient.IsEditor(ctx, userID, channelID)
	if err != nil {
		s.Log.Log("channelID", channelID, "userID", userID, "err", err, "error getting editor status")
		return false
	}

	return isEditor
}
