package api

import (
	"context"
	"net/http"
	"strconv"

	"code.justin.tv/feeds/errors"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/spade"
	"code.justin.tv/twitch-events/gea/cmd/gea/internal/api/models"
)

const (
	followsLimit    = 1000
	batchCheckLimit = 100
)

func (s *HTTPServer) followEvent(r *http.Request) (interface{}, error) {
	followCount := int64(-1)

	userID, err := requireUserID(r)
	if err != nil {
		return nil, err
	}
	eventID, err := requireString(r, "event_id")
	if err != nil {
		return nil, err
	}
	ctx := r.Context()

	success, err := s.EventFollows.Follow(ctx, eventID, userID)
	if err != nil {
		return nil, err
	}

	if success {
		followCount, err = s.OracleDB.IncrementEventFollowCount(ctx, eventID)
		if err != nil {
			s.Log.Log("event_id", eventID, "user_id", userID, "err", err, "error incrementing event follow count")
			return nil, nil
		}
	}

	s.sendServerFollowEvent(ctx, userID, eventID, followCount, "follow")

	return nil, nil
}

func (s *HTTPServer) unfollowEvent(r *http.Request) (interface{}, error) {
	followCount := int64(-1)

	userID, err := requireUserID(r)
	if err != nil {
		return nil, err
	}
	eventID, err := requireString(r, "event_id")
	if err != nil {
		return nil, err
	}
	ctx := r.Context()

	success, err := s.EventFollows.Unfollow(ctx, eventID, userID)
	if err != nil {
		return nil, err
	}

	if success {
		followCount, err = s.OracleDB.DecrementEventFollowCount(ctx, eventID)
		if err != nil {
			s.Log.Log("event_id", eventID, "user_id", userID, "err", err, "error decrementing event follow count")
			return nil, nil
		}
	}

	s.sendServerFollowEvent(ctx, userID, eventID, followCount, "unfollow")

	return nil, nil
}

type EventFollowers struct {
	UserIDs []string `json:"user_ids"`
	Cursor  string   `json:"cursor"`
}

func (s *HTTPServer) getEventFollowers(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	eventID, err := requireString(r, "event_id")
	if err != nil {
		return nil, err
	}
	limit, cursor, err := getPagination(r, 100, followsLimit)
	if err != nil {
		return nil, err
	}

	followers, err := s.EventFollows.GetFollowersByEventID(ctx, eventID, limit, cursor)
	if err != nil {
		return nil, err
	}

	return &EventFollowers{UserIDs: followers.UserIDs, Cursor: followers.Cursor}, nil
}

type FollowedEvents struct {
	EventIDs []string `json:"event_ids"`
	Cursor   string   `json:"cursor"`
}

func (s *HTTPServer) getFollowedEvents(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	userID, err := requireUserID(r)
	if err != nil {
		return nil, err
	}
	limit, cursor, err := getPagination(r, 100, followsLimit)
	if err != nil {
		return nil, err
	}

	events, err := s.EventFollows.GetFollowedEventsByUserID(ctx, userID, limit, cursor)
	if err != nil {
		return nil, err
	}

	return &FollowedEvents{EventIDs: events.EventIDs, Cursor: events.Cursor}, nil
}

func (s *HTTPServer) checkFollowedEvents(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	userID, err := requireUserID(r)
	if err != nil {
		return nil, err
	}
	eventIDs, err := requireStrings(r, "event_ids")
	if err != nil {
		return nil, err
	}
	if len(eventIDs) > batchCheckLimit {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Errorf("cannot check more than %v events", batchCheckLimit),
		}
	}

	followedEventIDs, err := s.EventFollows.GetFollowedEventsByUserIDAndEventIDs(ctx, userID, eventIDs)
	if err != nil {
		return nil, err
	}

	return &FollowedEvents{EventIDs: followedEventIDs}, nil
}

func (s *HTTPServer) sendServerFollowEvent(ctx context.Context, userID, eventID string, followCount int64, action string) {
	numericalUserID, err := strconv.ParseInt(userID, 10, 64)
	if err != nil {
		s.Log.LogCtx(ctx, "userID", userID, errors.Wrap(err, "error parsing userID to int64 for tracking"))
		return
	}

	serverFollowTrackingEvent := models.ServerFollowTracking{
		Action:      action,
		UserID:      numericalUserID,
		EventID:     eventID,
		FollowCount: followCount,
	}

	s.SpadeClient.QueueEvents(spade.Event{
		Name:       "oracle_follow_server",
		Properties: serverFollowTrackingEvent,
	})
}
