package api

import (
	"context"
	"encoding/json"
	"net/http"

	"code.justin.tv/feeds/errors"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/twitch-events/gea/cmd/gea/internal/api/models"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/types"
)

func (s *HTTPServer) requireLocalizationParams(ctx context.Context, r *http.Request) (*models.V1LocalizationParams, string, *db.Event, error) {
	userID, err := requireUserID(r)
	if err != nil {
		return nil, "", nil, err
	}

	var params models.V1LocalizationParams
	if err = json.NewDecoder(r.Body).Decode(&params); err != nil {
		return nil, "", nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "could not parse body"),
		}
	}

	dbEvent, err := s.requireDBEvent(ctx, params.EventID)
	if err != nil {
		return nil, "", nil, err
	}

	return &params, userID, dbEvent, nil
}

func (s *HTTPServer) addLocalization(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	params, userID, dbEvent, err := s.requireLocalizationParams(ctx, r)
	if err != nil {
		return nil, err
	}

	if err = s.canAddLocalization(ctx, userID); err != nil {
		return nil, err
	}

	localization, err := s.EventHandlers.AddLocalization(ctx, dbEvent, &types.LocalizationUpdateParams{
		EventID:     params.EventID,
		Language:    params.Language,
		Title:       params.Title,
		Description: params.Description,
		ChannelID:   params.ChannelID,
	})
	if err != nil {
		return nil, err
	}

	return localization, nil
}

func (s *HTTPServer) updateLocalization(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	params, userID, dbEvent, err := s.requireLocalizationParams(ctx, r)
	if err != nil {
		return nil, err
	}

	if err = s.canUpdateLocalization(ctx, userID); err != nil {
		return nil, err
	}

	localization, err := s.EventHandlers.UpdateLocalization(ctx, dbEvent, &types.LocalizationUpdateParams{
		EventID:     params.EventID,
		Language:    params.Language,
		Title:       params.Title,
		Description: params.Description,
		ChannelID:   params.ChannelID,
	})
	if err != nil {
		return nil, err
	}

	return localization, nil
}

func (s *HTTPServer) removeLocalization(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	params, userID, dbEvent, err := s.requireLocalizationParams(ctx, r)
	if err != nil {
		return nil, err
	}

	if err = s.canDeleteLocalization(ctx, userID); err != nil {
		return nil, err
	}

	localization, err := s.EventHandlers.RemoveLocalization(ctx, dbEvent, &types.LocalizationUpdateParams{
		EventID:     params.EventID,
		Language:    params.Language,
		Title:       params.Title,
		Description: params.Description,
		ChannelID:   params.ChannelID,
	})
	if err != nil {
		return nil, err
	}

	return localization, nil
}
