package api

import (
	"context"
	"net/http"
	"sort"
	"strconv"
	"strings"
	"sync"
	"time"

	"encoding/json"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/twitch-events/gea/cmd/gea/internal/api/models"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/types"
	jax "code.justin.tv/web/jax/client"
)

// Exclude Premiere events to mitigate issue where spam premiere events are appearing in the
// events panel in the game directory.
var featuredEventTypes = []string{
	types.EventTypeSegment,
	types.EventTypeSingle,
}

type SuggestionsConfig struct {
	timeWindow *distconf.Duration

	liveLimit          *distconf.Int
	futureLimit        *distconf.Int
	pastLimit          *distconf.Int
	minimumConcurrents *distconf.Int
	configByGame       *distconf.Str

	gameConfigsLock sync.RWMutex
	gameConfigs     map[string]GameConfig
}

type GameConfig struct {
	FeaturedChannels []string `json:"featured_channels"`
}

func (c *SuggestionsConfig) Load(dconf *distconf.Distconf) error {
	c.timeWindow = dconf.Duration("gea.suggestions.time_window", 2*7*24*time.Hour)
	c.liveLimit = dconf.Int("gea.suggestions.limits.live", 4)
	c.futureLimit = dconf.Int("gea.suggestions.limits.future", 6)
	c.pastLimit = dconf.Int("gea.suggestions.limits.past", 6)
	c.minimumConcurrents = dconf.Int("gea.suggestions.minimum_concurrents", 100)
	c.configByGame = dconf.Str("gea.suggestions.config_by_game", "")

	return nil
}

func (c *SuggestionsConfig) getFeaturedChannels(gameID string) []string {
	var featuredChannels []string
	c.gameConfigsLock.RLock()
	defer c.gameConfigsLock.RUnlock()
	if gameConfig, ok := c.gameConfigs[gameID]; ok {
		featuredChannels = append(featuredChannels, gameConfig.FeaturedChannels...)
	}
	return featuredChannels
}

func (s *HTTPServer) loadConfigByGame() {
	config := map[string]GameConfig{}

	str := s.SuggestionsConfig.configByGame.Get()
	if str != "" {
		err := json.Unmarshal([]byte(str), &config)
		if err != nil {
			s.Log.Log("err", err, "Could not load featured channels by game")
			return
		}
	}

	s.SuggestionsConfig.gameConfigsLock.Lock()
	s.SuggestionsConfig.gameConfigs = config
	s.SuggestionsConfig.gameConfigsLock.Unlock()
}

func (s *HTTPServer) getEventSuggestionsByGame(r *http.Request) (interface{}, error) {
	ctx := r.Context()

	gameID, err := requireString(r, "game_id")
	if err != nil {
		return nil, err
	}

	var languages []string
	langParam := r.URL.Query().Get("language")
	if langParam != "" {
		languages = strings.Split(langParam, ",")
	}

	now := s.Clock.NowUTC()

	liveEventIDs, err := s.getSuggestedLiveEventIDs(ctx, now, gameID, languages)
	if err != nil {
		return nil, err
	}

	futureEventIDs, err := s.getSuggestedFutureEventIDs(ctx, now, gameID, languages)
	if err != nil {
		return nil, err
	}

	pastEventIDs, err := s.getSuggestedPastEventIDs(ctx, now, gameID, languages)
	if err != nil {
		return nil, err
	}

	return &cachedResponse{
		payload: models.SuggestionsResult{
			Live:   liveEventIDs,
			Future: futureEventIDs,
			Past:   pastEventIDs,
		},
		cacheTime: 2 * time.Minute,
	}, nil
}

func (s *HTTPServer) getSuggestedLiveEventIDs(ctx context.Context, now time.Time, gameID string, languages []string) ([]string, error) {
	filter := db.BroadcastFilter{
		GameIDs:   []string{gameID},
		Languages: languages,
		StartTimeWindow: &db.TimeWindow{
			End: &now,
		},
		EndTimeWindow: &db.TimeWindow{
			Start: &now,
		},
		Types: featuredEventTypes,
	}

	featuredChannels := s.SuggestionsConfig.getFeaturedChannels(gameID)
	if len(featuredChannels) > 0 {
		filter.ChannelIDs = featuredChannels
	}

	liveBroadcasts, err := s.OracleDB.GetBroadcastsByHype(ctx, &filter, false, "", 100)
	if err != nil {
		return nil, err
	}

	filteredLiveBroadcasts, err := s.filterAndSortLiveBroadcasts(ctx, liveBroadcasts)
	if err != nil {
		return nil, err
	}

	liveEventIDs := make([]string, len(filteredLiveBroadcasts))
	for i, broadcast := range filteredLiveBroadcasts {
		liveEventIDs[i] = broadcast.EventID
	}

	return liveEventIDs, nil
}

func (s *HTTPServer) getSuggestedFutureEventIDs(ctx context.Context, now time.Time, gameID string, languages []string) ([]string, error) {
	futureTimestamp := now.Add(s.SuggestionsConfig.timeWindow.Get())
	filter := db.BroadcastFilter{
		GameIDs:   []string{gameID},
		Languages: languages,
		StartTimeWindow: &db.TimeWindow{
			Start: &now,
			End:   &futureTimestamp,
		},
		Types: featuredEventTypes,
	}

	featuredChannels := s.SuggestionsConfig.getFeaturedChannels(gameID)
	if len(featuredChannels) > 0 {
		filter.ChannelIDs = featuredChannels
	}

	limit := int(s.SuggestionsConfig.futureLimit.Get())

	eventIDs, err := s.OracleDB.GetEventIDsSortedByHype(ctx, &filter, false, "", limit)
	if err != nil {
		return nil, err
	}

	futureEventIDs := make([]string, 0, len(eventIDs.EventIDs))
	futureEventIDs = append(futureEventIDs, eventIDs.EventIDs...)

	return s.sortEventIDsByComparator(ctx, futureEventIDs, isStartTimeEarlier)
}

func (s *HTTPServer) getSuggestedPastEventIDs(ctx context.Context, now time.Time, gameID string, languages []string) ([]string, error) {
	pastTimestamp := now.Add(-s.SuggestionsConfig.timeWindow.Get())
	filter := db.BroadcastFilter{
		GameIDs:   []string{gameID},
		Languages: languages,
		StartTimeWindow: &db.TimeWindow{
			Start: &pastTimestamp,
			End:   &now,
		},
		Types: featuredEventTypes,
	}

	featuredChannels := s.SuggestionsConfig.getFeaturedChannels(gameID)
	if len(featuredChannels) > 0 {
		filter.ChannelIDs = featuredChannels
	}

	pastEventIDs, err := s.OracleDB.GetEventIDsSortedByHype(ctx, &filter, false, "", int(s.SuggestionsConfig.pastLimit.Get()))
	if err != nil {
		return nil, err
	}

	return s.sortEventIDsByComparator(ctx, pastEventIDs.EventIDs, isEndTimeLater)
}

type liveBroadcastSorter struct {
	broadcasts []*db.Broadcast
	streams    map[string]*jax.Stream
}

func (s *liveBroadcastSorter) Len() int { return len(s.broadcasts) }
func (s *liveBroadcastSorter) Swap(i, j int) {
	s.broadcasts[i], s.broadcasts[j] = s.broadcasts[j], s.broadcasts[i]
}

func (s *liveBroadcastSorter) concurrentForChannel(id string) int {
	stream, ok := s.streams[id]
	if !ok || stream == nil {
		return 0
	}
	if stream.Properties.Usher.ChannelCount == nil {
		return 0
	}

	return *stream.Properties.Usher.ChannelCount
}

func (s *liveBroadcastSorter) Less(i, j int) bool {
	if s.broadcasts[i] == nil {
		return s.broadcasts[j] != nil
	}
	if s.broadcasts[j] == nil {
		return false
	}

	return s.concurrentForChannel(s.broadcasts[i].ChannelID) >= s.concurrentForChannel(s.broadcasts[j].ChannelID)
}

func (s *liveBroadcastSorter) Filter(minConcurrents int, limit int) []*db.Broadcast {
	if limit >= len(s.broadcasts) {
		limit = len(s.broadcasts)
	}
	var i int
	for i = 0; i < limit; i++ {
		if s.concurrentForChannel(s.broadcasts[i].ChannelID) < minConcurrents {
			return s.broadcasts[0:i]
		}
	}
	return s.broadcasts[0:i]
}

func (s *HTTPServer) filterAndSortLiveBroadcasts(ctx context.Context, broadcasts []*db.Broadcast) ([]*db.Broadcast, error) {
	channelIDs := make([]string, len(broadcasts))
	for i, b := range broadcasts {
		channelIDs[i] = b.ChannelID
	}

	// Get streams from jax
	liveStreams, err := s.JaxClient.GetStreamsByChannelIDs(ctx, channelIDs, nil, nil)
	if err != nil {
		return nil, err
	}

	// Create map of channelID => *Stream
	streamsByChannel := make(map[string]*jax.Stream)
	for _, stream := range liveStreams.Hits {
		channelID := *stream.Properties.Rails.ChannelID
		streamsByChannel[strconv.Itoa(channelID)] = stream
	}

	// Create sorter
	sorter := &liveBroadcastSorter{
		broadcasts: broadcasts,
		streams:    streamsByChannel,
	}

	// Sort
	sort.Sort(sorter)

	// Filter
	return sorter.Filter(int(s.SuggestionsConfig.minimumConcurrents.Get()), int(s.SuggestionsConfig.liveLimit.Get())), nil
}

type eventComparator func(a, b *db.Event) bool

type eventsSorter struct {
	events     []*db.Event
	comparator eventComparator
}

func (s *eventsSorter) Len() int { return len(s.events) }
func (s *eventsSorter) Swap(i, j int) {
	s.events[i], s.events[j] = s.events[j], s.events[i]
}
func (s *eventsSorter) Less(i, j int) bool {
	if s.comparator == nil {
		return i < j
	}
	return s.comparator(s.events[i], s.events[j])
}

func isStartTimeEarlier(a, b *db.Event) bool {
	if a == nil || a.StartTime == nil {
		return b != nil && b.StartTime != nil
	}
	if b == nil || b.StartTime == nil {
		return false
	}
	return a.StartTime.Before(*b.StartTime)
}

func isEndTimeLater(a, b *db.Event) bool {
	if a == nil || a.EndTime == nil {
		return b != nil && b.EndTime != nil
	}
	if b == nil || b.EndTime == nil {
		return false
	}
	return a.EndTime.After(*b.EndTime)
}

func (s *HTTPServer) sortEventIDsByComparator(ctx context.Context, eventIDs []string, comparator eventComparator) ([]string, error) {
	if len(eventIDs) == 0 {
		return []string{}, nil
	}
	events, err := s.OracleDB.GetEvents(ctx, eventIDs, false)
	if err != nil {
		return nil, err
	}

	sorter := &eventsSorter{
		events:     events,
		comparator: comparator,
	}
	sort.Sort(sorter)

	ids := make([]string, len(events))
	for i, event := range sorter.events {
		ids[i] = event.ID
	}

	return ids, nil
}
