package api

import (
	"context"
	"strconv"
	"time"

	"code.justin.tv/twitch-events/gea/cmd/gea/internal/api/models"
	"code.justin.tv/twitch-events/gea/internal/db"
	"github.com/pkg/errors"
)

// ParamsToCreateEventTracking converts CreateEventParams to an "event create" tracking event
func (s *HTTPServer) paramsToCreateEventTracking(ctx context.Context, userID, eventID string, now time.Time, params models.V1CreateEventParams) *models.ServerEventTracking {
	numericalUserID, err := strconv.ParseInt(userID, 10, 64)
	if err != nil {
		numericalUserID = 0
		s.Log.LogCtx(ctx, errors.Wrap(err, "error parsing userID to int64 for tracking"))
	}

	gameID := "0"
	if params.GameID != nil {
		gameID = *params.GameID
	}
	numericalGameID, err := strconv.ParseInt(gameID, 10, 64)
	if err != nil {
		numericalGameID = 0
		s.Log.LogCtx(ctx, errors.Wrap(err, "error parsing gameID to int64 for tracking"))
	}

	parentEventID := ""
	if params.ParentID != nil {
		parentEventID = *params.ParentID
	}

	var startTimeUnix *int64
	if params.StartTime != nil {
		tempStart := params.StartTime.Unix()
		startTimeUnix = &tempStart
	}

	var endTimeUnix *int64
	if params.EndTime != nil {
		tempEnd := params.EndTime.Unix()
		endTimeUnix = &tempEnd
	}

	timeZoneID := ""
	if params.TimeZoneID != nil {
		timeZoneID = *params.TimeZoneID
	}

	channelID := ""
	if params.ChannelID != nil {
		channelID = *params.ChannelID
	}

	title := ""
	if params.Title != nil {
		title = *params.Title
	}

	description := ""
	if params.Description != nil {
		description = *params.Description
	}

	coverImageID := ""
	if params.CoverImageID != nil {
		coverImageID = params.CoverImageID.ID
	}

	return &models.ServerEventTracking{
		Action:        "create",
		UserID:        numericalUserID,
		EventID:       eventID,
		EventType:     params.Type,
		ParentEventID: parentEventID,
		CreatedAt:     now.Unix(),
		UpdatedAt:     now.Unix(),
		StartTime:     startTimeUnix,
		EndTime:       endTimeUnix,
		TimeZoneID:    timeZoneID,
		ChannelID:     channelID,
		Title:         title,
		Description:   description,
		GameID:        numericalGameID,
		CoverImageID:  coverImageID,
	}
}

// ParamsToUpdateEventTracking converts UpdateEventParams to an "event update" tracking event
func (s *HTTPServer) paramsToUpdateEventTracking(ctx context.Context, userID, eventID string, dbEventGameID *string, dbEventCreatedAt, now time.Time, params models.V1UpdateEventParams) *models.ServerEventTracking {
	numericalUserID, err := strconv.ParseInt(userID, 10, 64)
	if err != nil {
		numericalUserID = 0
		s.Log.LogCtx(ctx, errors.Wrap(err, "error parsing userID to int64 for tracking"))
	}

	gameID := "0"
	if dbEventGameID != nil {
		gameID = *dbEventGameID
	}
	numericalGameID, err := strconv.ParseInt(gameID, 10, 64)
	if err != nil {
		numericalGameID = 0
		s.Log.LogCtx(ctx, errors.Wrap(err, "error parsing gameID to int64 for tracking"))
	}

	eventType := ""
	if params.Type != nil {
		eventType = *params.Type
	}

	parentEventID := ""
	if params.ParentID != nil {
		parentEventID = *params.ParentID
	}

	var startTimeUnix *int64
	if params.StartTime != nil {
		tempStart := params.StartTime.Unix()
		startTimeUnix = &tempStart
	}

	var endTimeUnix *int64
	if params.EndTime != nil {
		tempEnd := params.EndTime.Unix()
		endTimeUnix = &tempEnd
	}

	timeZoneID := ""
	if params.TimeZoneID != nil {
		timeZoneID = *params.TimeZoneID
	}

	channelID := ""
	if params.ChannelID != nil {
		channelID = *params.ChannelID
	}

	title := ""
	if params.Title != nil {
		title = *params.Title
	}

	description := ""
	if params.Description != nil {
		description = *params.Description
	}

	coverImageID := ""
	if params.CoverImageID != nil {
		coverImageID = params.CoverImageID.ID
	}

	return &models.ServerEventTracking{
		Action:        "update",
		UserID:        numericalUserID,
		EventID:       eventID,
		EventType:     eventType,
		ParentEventID: parentEventID,
		CreatedAt:     dbEventCreatedAt.Unix(),
		UpdatedAt:     now.Unix(),
		StartTime:     startTimeUnix,
		EndTime:       endTimeUnix,
		TimeZoneID:    timeZoneID,
		ChannelID:     channelID,
		Title:         title,
		Description:   description,
		GameID:        numericalGameID,
		CoverImageID:  coverImageID,
	}
}

// DBEventToDeleteEventTracking converts a dbEvent to an "event delete" tracking event
func (s *HTTPServer) dbEventToDeleteEventTracking(ctx context.Context, userID, eventID string, deletedAt time.Time, dbEvent *db.Event) *models.ServerEventTracking {
	numericalUserID, err := strconv.ParseInt(userID, 10, 64)
	if err != nil {
		numericalUserID = 0
		s.Log.LogCtx(ctx, errors.Wrap(err, "error parsing userID to int64 for tracking"))
	}

	gameID := "0"
	if dbEvent.GameID != nil {
		gameID = *dbEvent.GameID
	}
	numericalGameID, err := strconv.ParseInt(gameID, 10, 64)
	if err != nil {
		numericalGameID = 0
		s.Log.LogCtx(ctx, errors.Wrap(err, "error parsing gameID to int64 for tracking"))
	}

	parentEventID := ""
	if dbEvent.ParentID != nil {
		parentEventID = *dbEvent.ParentID
	}

	var startTimeUnix *int64
	if dbEvent.StartTime != nil {
		tempStart := dbEvent.StartTime.Unix()
		startTimeUnix = &tempStart
	}

	var endTimeUnix *int64
	if dbEvent.EndTime != nil {
		tempEnd := dbEvent.EndTime.Unix()
		endTimeUnix = &tempEnd
	}

	timeZoneID := ""
	if dbEvent.TimeZoneID != nil {
		timeZoneID = *dbEvent.TimeZoneID
	}

	channelID := ""
	if dbEvent.ChannelID != nil {
		channelID = *dbEvent.ChannelID
	}

	title := ""
	if dbEvent.Title != nil {
		title = *dbEvent.Title
	}

	description := ""
	if dbEvent.Description != nil {
		description = *dbEvent.Description
	}

	coverImageID := ""
	if dbEvent.CoverImageID != nil {
		coverImageID = dbEvent.CoverImageID.ID
	}

	return &models.ServerEventTracking{
		Action:        "remove",
		UserID:        numericalUserID,
		EventID:       eventID,
		EventType:     dbEvent.Type,
		ParentEventID: parentEventID,
		CreatedAt:     dbEvent.CreatedAt.Unix(),
		UpdatedAt:     deletedAt.Unix(),
		StartTime:     startTimeUnix,
		EndTime:       endTimeUnix,
		TimeZoneID:    timeZoneID,
		ChannelID:     channelID,
		Title:         title,
		Description:   description,
		GameID:        numericalGameID,
		CoverImageID:  coverImageID,
	}
}
