package main

import (
	"net"
	"net/http"
	"os"
	"os/signal"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/metrics/sfx/sfxstatsd"
	"code.justin.tv/feeds/metrics/statsdim"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedsqs"
	"code.justin.tv/feeds/spade"
	"code.justin.tv/feeds/spade/spadedconf"
	"code.justin.tv/feeds/xray/plugins/ecs"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/twitch-events/gea/cmd/gea/internal/api"
	"code.justin.tv/twitch-events/gea/internal/auth"
	"code.justin.tv/twitch-events/gea/internal/cache"
	channelevent "code.justin.tv/twitch-events/gea/internal/channel-event"
	"code.justin.tv/twitch-events/gea/internal/clock"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/filter"
	"code.justin.tv/twitch-events/gea/internal/follows"
	"code.justin.tv/twitch-events/gea/internal/hallpass"
	"code.justin.tv/twitch-events/gea/internal/hypeman"
	hypemanscheduler "code.justin.tv/twitch-events/gea/internal/hypeman-scheduler"
	hypemanworker "code.justin.tv/twitch-events/gea/internal/hypeman-worker"
	"code.justin.tv/twitch-events/gea/internal/hypeman-worker/recipient"
	"code.justin.tv/twitch-events/gea/internal/images"
	jax "code.justin.tv/twitch-events/gea/internal/jax-client"
	"code.justin.tv/twitch-events/gea/internal/pubsub"
	"code.justin.tv/twitch-events/gea/internal/types"
	harddelete "code.justin.tv/twitch-events/gea/internal/user-hard-delete"
	"code.justin.tv/twitch-events/gea/internal/video"
	"code.justin.tv/vod/vodapi/rpc/vodapi"
	discovery "code.justin.tv/web/discovery/client"
	jaxclient "code.justin.tv/web/jax/client"
	"code.justin.tv/web/users-service/client/channels"
	usersservice "code.justin.tv/web/users-service/client/usersclient_internal"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sqs"
)

const (
	teamName    = "events"
	serviceName = "gea"
)

// CodeVersion is set by the build script
var CodeVersion string

func sfxStastdConfig() sfxstatsd.SetupConfig {
	return sfxstatsd.SetupConfig{
		CounterMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "http.*.code.*",
				DimensionsMap: "-.handler.-.http_code",
				MetricName:    "http.hits",
			},
			{
				MetricPath:    "*.msg_input",
				DimensionsMap: "sqs_queue_name.%",
				MetricName:    "dequeue_size",
			},
			{
				MetricPath:    "db.*.status.*",
				DimensionsMap: "-.query.-.status",
				MetricName:    "db.hits",
			},
		},
		TimingMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "http.*.time",
				DimensionsMap: "%.handler.%",
			},
			{
				MetricPath:    "*.sqs.ReceiveMessage",
				DimensionsMap: "sqs_queue_name.%.%",
			},
			{
				MetricPath:    "db.*.time",
				DimensionsMap: "%.query.%",
			},
		},
	}
}

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion:    CodeVersion,
		SfxSetupConfig: sfxStastdConfig(),
	},
}

// nolint: structcheck
type injectables struct {
	syncUploadClient *images.SynchronousUploadClient
	hypemanClient    hypeman.Client
	hallpassClient   hallpass.Client
	jaxClient        jax.Client
	spadeClient      api.SpadeClient
	getChannelClient channelevent.GetChannelClient
	vodAPIClient     video.VodAPIClient
	adminList        auth.AdminList
	authUsersClient  auth.Client

	schedulerNowFunc                  func() time.Time
	pubsubNowFunc                     func() time.Time
	createNotificationJobsDoneFunc    func()
	createChannelEventUpdatesDoneFunc func()

	notificationsSNSClient hypemanworker.NotificationsSNSClient

	channelEventSQSClient channelevent.SQSClient

	getUsersClient filter.GetUsersClient
	pubsubClient   pubsub.Client
	clock          clock.Clock
}

type clientConfig struct {
	jaxURL          *distconf.Str
	discoveryURL    *distconf.Str
	hallpassURL     *distconf.Str
	usersServiceURL *distconf.Str
	pubsubURL       *distconf.Str
}

func (c *clientConfig) Load(dconf *distconf.Distconf) error {
	c.jaxURL = dconf.Str("gea.clients.jax.url", "")
	if c.jaxURL.Get() == "" {
		return errors.New("unable to find a valid Jax url")
	}

	c.discoveryURL = dconf.Str("gea.clients.discovery.url", "")
	if c.discoveryURL.Get() == "" {
		return errors.New("unable to find a valid Discovery url")
	}

	c.hallpassURL = dconf.Str("gea.clients.hallpass.url", "")
	if c.hallpassURL.Get() == "" {
		return errors.New("unable to find a valid hallpass url")
	}

	c.usersServiceURL = dconf.Str("gea.clients.users_service.url", "")
	if c.usersServiceURL.Get() == "" {
		return errors.New("unable to find a valid users service url")
	}

	c.pubsubURL = dconf.Str("gea.clients.pubsub.url", "")
	if c.pubsubURL.Get() == "" {
		return errors.New("unable to find a valid pubsub url")
	}

	return nil
}

type service struct {
	injectables
	osExit        func(code int)
	onListen      func(net.Addr)
	sigChan       chan os.Signal
	serviceCommon service_common.ServiceCommon
	runner        service_common.ServiceRunner

	configs struct {
		httpConfig              api.HTTPConfig
		eventFollowsConfig      follows.EventFollowsConfig
		imageURLClientConfig    images.ImageURLConfig
		imageUploadConfig       images.ImageUploadServiceConfig
		oracleDBConfig          db.Config
		suggestionsConfig       api.SuggestionsConfig
		syncUploadClientConfig  images.SynchronousUploadClientConfig
		distconfAdminListConfig api.DistconfAdminListConfig

		hypemanSchedulerConfig       hypemanscheduler.SchedulerConfig
		hypemanClientConfig          hypeman.Config
		jobStatusClientConfig        hypemanscheduler.JobStatusClientConfig
		clientConfig                 clientConfig
		spadeConfig                  spadedconf.Config
		channelEventSchedulerConfig  channelevent.SchedulerConfig
		channelEventSQSClientConfig  channelevent.SQSClientConfig
		hypemanWorkerConfig          hypemanworker.WorkerConfig
		notificationsSNSClientConfig hypemanworker.NotificationsSNSClientConfig
		userHardDeleteWorkerConfig   harddelete.WorkerConfig

		cacheConfig             cache.Config
		eventCacheConfig        types.EventCacheConfig
		channelEventCacheConfig channelevent.CacheConfig
		metadataCacheConfig     types.MetadataCacheConfig

		videoFinderConfig video.FinderConfig
		videoCacheConfig  video.CacheConfig

		baseRecipientLoaderConfig   recipient.BaseRecipientLoaderConfig
		loadEventsThrottlerConfig   types.DistconfLoadEventsThrottlerConfig
		channelEventPublisherConfig channelevent.PublisherConfig
	}

	server         api.HTTPServer
	imageUploader  images.ImageUploadService
	imageURLClient *images.ImageURLClient

	innerDBClient *db.Impl
	oracleDB      db.DB
	cacheService  *cache.Service

	eventHandlers *types.EventHandlers
	eventFollows  *follows.EventFollows
	filter        *filter.Filter

	hypemanScheduler      *hypemanscheduler.Scheduler
	channelEventScheduler *channelevent.Scheduler
	jobStatusClient       *hypemanscheduler.JobStatusClient

	hypemanWorker        *hypemanworker.Worker
	userHardDeleteWorker *harddelete.Worker

	liveEventLoader       *channelevent.LiveEventLoader
	channelEventPublisher *channelevent.Publisher
}

func (f *service) setup() error {
	if err := f.serviceCommon.Setup(); err != nil {
		return err
	}
	if err := service_common.LoadConfigs(
		f.serviceCommon.Config,
		&f.configs.httpConfig,
		&f.configs.eventFollowsConfig,
		&f.configs.imageURLClientConfig,
		&f.configs.imageUploadConfig,
		&f.configs.oracleDBConfig,
		&f.configs.suggestionsConfig,
		&f.configs.syncUploadClientConfig,
		&f.configs.distconfAdminListConfig,

		&f.configs.hypemanSchedulerConfig,
		&f.configs.hypemanClientConfig,
		&f.configs.jobStatusClientConfig,
		&f.configs.clientConfig,
		&f.configs.spadeConfig,

		&f.configs.hypemanWorkerConfig,
		&f.configs.notificationsSNSClientConfig,

		&f.configs.userHardDeleteWorkerConfig,

		&f.configs.cacheConfig,
		&f.configs.eventCacheConfig,
		&f.configs.channelEventCacheConfig,
		&f.configs.metadataCacheConfig,

		&f.configs.videoFinderConfig,
		&f.configs.videoCacheConfig,

		&f.configs.baseRecipientLoaderConfig,
		&f.configs.loadEventsThrottlerConfig,

		&f.configs.channelEventPublisherConfig,
		&f.configs.channelEventSchedulerConfig,
		&f.configs.channelEventSQSClientConfig,
	); err != nil {
		return err
	}

	if err := f.configs.oracleDBConfig.LoadSecrets(f.serviceCommon.Secrets); err != nil {
		return err
	}

	dbClient, err := db.NewClient(&f.configs.oracleDBConfig, f.serviceCommon.Log)
	if err != nil {
		return err
	}
	f.innerDBClient = dbClient
	dbStatsSender := &service_common.StatSender{
		SubStatter:   f.serviceCommon.Statsd.NewSubStatter("db"),
		ErrorTracker: &f.serviceCommon.ErrorTracker,
	}
	f.oracleDB = &db.InstrumentedDBClient{
		Inner: f.innerDBClient,
		Log:   f.serviceCommon.Log,
		Stats: dbStatsSender,
	}

	if f.getUsersClient == nil {
		usersServiceHost := f.configs.clientConfig.usersServiceURL.Get()
		usersClient, innerErr := usersservice.NewClient(f.foundationClient(usersServiceHost))
		if innerErr != nil {
			return innerErr
		}

		f.getUsersClient = usersClient
	}
	f.filter = &filter.Filter{
		GetUsersClient: f.getUsersClient,
		Log:            f.serviceCommon.Log,
	}

	f.imageURLClient = &images.ImageURLClient{
		Config: f.configs.imageURLClientConfig,
		Log:    f.serviceCommon.Log,
	}

	if f.vodAPIClient == nil {
		vodAPIURL := f.configs.videoFinderConfig.VodAPIURL.Get()
		f.vodAPIClient = vodapi.NewVodApiProtobufClient(vodAPIURL, twitchclient.NewHTTPClient(f.foundationClient(vodAPIURL)))
	}

	videoFinder := &video.Finder{
		Config: &f.configs.videoFinderConfig,
		VodAPI: f.vodAPIClient,
	}

	f.cacheService = &cache.Service{Config: &f.configs.cacheConfig}

	eventCache := &types.EventCache{
		Config: &f.configs.eventCacheConfig,
		Cache:  f.cacheService,
		Stats: &service_common.StatSender{
			SubStatter:   f.serviceCommon.Statsd.NewSubStatter("cache"),
			ErrorTracker: &f.serviceCommon.ErrorTracker,
		},
		Log: f.serviceCommon.Log,
	}
	channelEventCache := &channelevent.Cache{
		Config: &f.configs.channelEventCacheConfig,
		Cache:  f.cacheService,
		Stats: &service_common.StatSender{
			SubStatter:   f.serviceCommon.Statsd.NewSubStatter("cache"),
			ErrorTracker: &f.serviceCommon.ErrorTracker,
		},
		Log: f.serviceCommon.Log,
	}
	metadataCache := &types.MetadataCache{
		Config: &f.configs.metadataCacheConfig,
		Cache:  f.cacheService,
		Stats: &service_common.StatSender{
			SubStatter:   f.serviceCommon.Statsd.NewSubStatter("metadata_cache"),
			ErrorTracker: &f.serviceCommon.ErrorTracker,
		},
		Log: f.serviceCommon.Log,
	}
	videoCache := &video.Cache{
		Config: &f.configs.videoCacheConfig,
		Cache:  f.cacheService,
		Log:    f.serviceCommon.Log,
	}

	usersServiceHost := f.configs.clientConfig.usersServiceURL.Get()
	usersClient, err := usersservice.NewClient(f.foundationClient(usersServiceHost))
	if err != nil {
		return err
	}

	baseEventHandler := types.BaseEventHandler{
		OracleDB: f.oracleDB,
		Log:      f.serviceCommon.Log,
	}
	timetableEventHandler := &types.TimetableEventHandler{
		BaseEventHandler: baseEventHandler,
		ImageURLClient:   f.imageURLClient,
		UserService:      usersClient,
	}
	segmentEventHandler := &types.SegmentEventHandler{
		BaseEventHandler: baseEventHandler,
		EventCache:       eventCache,
		MetadataCache:    metadataCache,
		ImageURLClient:   f.imageURLClient,
		VideoFinder:      videoFinder,
		TimetableHandler: timetableEventHandler,
		UserService:      usersClient,
	}
	timetableEventHandler.SegmentHandler = segmentEventHandler

	f.eventHandlers = &types.EventHandlers{
		OracleDB:      f.oracleDB,
		Log:           f.serviceCommon.Log,
		Filter:        f.filter,
		EventCache:    eventCache,
		MetadataCache: metadataCache,
		VideoCache:    videoCache,
		LoadEventsThrottler: types.NewLoadEventsThrottler(
			&f.configs.loadEventsThrottlerConfig,
			&clock.RealClock{},
		),
		Handlers: []types.EventHandler{
			types.EventHandler(&types.SingleEventHandler{
				BaseEventHandler: baseEventHandler,
				ImageURLClient:   f.imageURLClient,
				VideoFinder:      videoFinder,
				UserService:      usersClient,
			}),
			types.EventHandler(&types.PremiereEventHandler{
				BaseEventHandler: baseEventHandler,
				ImageURLClient:   f.imageURLClient,
				UserService:      usersClient,
			}),
			types.EventHandler(timetableEventHandler),
			types.EventHandler(segmentEventHandler),
		},
	}

	session, awsConf := service_common.CreateAWSSession(f.serviceCommon.Config)
	dynamoClient := dynamodb.New(session, awsConf...)
	f.eventFollows = &follows.EventFollows{
		Dynamo:                 dynamoClient,
		Config:                 &f.configs.eventFollowsConfig,
		Log:                    f.serviceCommon.Log,
		RequireConsistentReads: f.serviceCommon.Config.Bool("dynamo.consistent_reads", false).Get(),
	}

	if f.syncUploadClient == nil {
		f.syncUploadClient = &images.SynchronousUploadClient{
			Config:       &f.configs.syncUploadClientConfig,
			S3HTTPClient: &http.Client{},
			Log:          f.serviceCommon.Log,
		}
	}

	if f.hallpassClient == nil {
		f.hallpassClient, err = hallpass.NewClient(f.foundationClient(f.configs.clientConfig.hallpassURL.Get()))
		if err != nil {
			return err
		}
	}

	if f.spadeClient == nil {
		spadeClient := spade.Client{
			Config:     &f.configs.spadeConfig,
			HTTPClient: &http.Client{},
			Logger:     f.serviceCommon.Log,
		}
		spadeClient.Setup()
		f.spadeClient = &spadeClient
	}

	if f.clock == nil {
		f.clock = &clock.RealClock{}
	}

	if f.adminList == nil {
		f.adminList = &api.DistconfAdminList{
			Config: &f.configs.distconfAdminListConfig,
		}
	}

	sqsClient := sqs.New(session, awsConf...)
	if f.hypemanClient == nil {
		f.hypemanClient = &hypeman.ClientImpl{
			Sqs:    sqsClient,
			Config: &f.configs.hypemanClientConfig,
			Log:    f.serviceCommon.Log,
			CtxLog: &f.serviceCommon.Ctxlog,
		}
	}
	f.jobStatusClient = &hypemanscheduler.JobStatusClient{
		Dynamo: dynamoClient,
		Config: &f.configs.jobStatusClientConfig,
		Log:    f.serviceCommon.Log,
	}
	if f.jaxClient == nil {
		webJaxClient, innerErr := jaxclient.NewClient(f.foundationClient(f.configs.clientConfig.jaxURL.Get()))
		if innerErr != nil {
			return innerErr
		}
		f.jaxClient = &jax.ClientWithWrappedErrors{
			InnerClient: webJaxClient,
		}
	}

	f.hypemanScheduler = &hypemanscheduler.Scheduler{
		OracleDB:                       f.oracleDB,
		JobStatus:                      f.jobStatusClient,
		Log:                            f.serviceCommon.Log,
		Config:                         &f.configs.hypemanSchedulerConfig,
		HypemanClient:                  f.hypemanClient,
		JaxClient:                      f.jaxClient,
		NowFunc:                        f.schedulerNowFunc,
		CreateNotificationJobsDoneFunc: f.createNotificationJobsDoneFunc,
	}

	channelEventSQSClient := f.injectables.channelEventSQSClient
	if channelEventSQSClient == nil {
		channelEventSQSClient = &channelevent.SQSClientImpl{
			Log:    f.serviceCommon.Log,
			Config: &f.configs.channelEventSQSClientConfig,
			SQS:    sqsClient,
		}
	}

	f.channelEventScheduler = &channelevent.Scheduler{
		Cache:                             channelEventCache,
		Config:                            &f.configs.channelEventSchedulerConfig,
		EventHandlers:                     f.eventHandlers,
		Log:                               f.serviceCommon.Log,
		OracleDB:                          f.oracleDB,
		SQS:                               channelEventSQSClient,
		NowFunc:                           f.pubsubNowFunc,
		CreateChannelEventUpdatesDoneFunc: f.createChannelEventUpdatesDoneFunc,
	}

	if f.getUsersClient == nil {
		f.getUsersClient = usersClient
	}
	if f.getChannelClient == nil {
		channelClient, innerErr := channels.NewClient(f.foundationClient(usersServiceHost))
		if innerErr != nil {
			return innerErr
		}

		f.getChannelClient = channelClient
	}
	f.filter = &filter.Filter{
		GetUsersClient: f.getUsersClient,
		Log:            f.serviceCommon.Log,
	}

	err = f.serviceCommon.XRay.WithPlugin(&ecs.Plugin{})
	if err != nil {
		f.serviceCommon.Log.Log("err", err, "unable to enable ecs plugin")
	}

	s3Client := s3.New(session, awsConf...)
	f.imageUploader = images.ImageUploadService{
		SQSQueueProcessor: feedsqs.SQSQueueProcessor{
			Conf: &f.configs.imageUploadConfig.SQSQueueProcessorConfig,
			Log:  f.serviceCommon.Log,
			Sqs:  sqs.New(session, awsConf...),
			Ch:   &f.serviceCommon.Ctxlog,
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("upload"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
		},
		Config:       &f.configs.imageUploadConfig,
		S3:           s3Client,
		SyncUploader: f.syncUploadClient,
	}

	discoveryClient, err := discovery.NewClient(f.foundationClient(f.configs.clientConfig.discoveryURL.Get()))
	if err != nil {
		return err
	}

	snsClient := sns.New(session, awsConf...)
	notificationsSNSClient := f.injectables.notificationsSNSClient
	if notificationsSNSClient == nil {
		notificationsSNSClient = &hypemanworker.NotificationsSNSClientImpl{
			Config:    &f.configs.notificationsSNSClientConfig,
			SNSClient: snsClient,
			Log:       f.serviceCommon.Log,
		}
	}

	baseRecipientLoader := recipient.BaseRecipientLoader{
		EventFollowsClient: f.eventFollows,
		Config:             &f.configs.baseRecipientLoaderConfig,
	}
	recipientLoaders := &recipient.Loaders{
		TypedEventHandlers: f.eventHandlers,
		TypedRecipientLoaders: []recipient.TypedRecipientLoader{
			&recipient.SingleRecipientLoader{
				BaseRecipientLoader: baseRecipientLoader,
			},
			&recipient.PremiereRecipientLoader{
				BaseRecipientLoader: baseRecipientLoader,
			},
			&recipient.SegmentRecipientLoader{
				BaseRecipientLoader: baseRecipientLoader,
				Log:                 f.serviceCommon.Log,
				StatusClient:        f.jobStatusClient,
				Clock:               f.clock,
				TypedEventHandlers:  f.eventHandlers,
			},
		},
	}
	f.hypemanWorker = &hypemanworker.Worker{
		SQSQueueProcessor: feedsqs.SQSQueueProcessor{
			Conf: &f.configs.hypemanWorkerConfig.SQSQueueProcessorConfig,
			Log:  f.serviceCommon.Log,
			Sqs:  sqs.New(session, awsConf...),
			Ch:   &f.serviceCommon.Ctxlog,
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("hypeman_worker"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
		},
		Config:                 &f.configs.hypemanWorkerConfig,
		DiscoveryClient:        discoveryClient,
		EventFollowsClient:     f.eventFollows,
		DBClient:               f.oracleDB,
		NotificationsSNSClient: notificationsSNSClient,
		ImageURLClient:         f.imageURLClient,
		RecipientLoaders:       recipientLoaders,
	}

	f.userHardDeleteWorker = &harddelete.Worker{
		SQSQueueProcessor: feedsqs.SQSQueueProcessor{
			Conf: &f.configs.userHardDeleteWorkerConfig.SQSQueueProcessorConfig,
			Log:  f.serviceCommon.Log,
			Sqs:  sqs.New(session, awsConf...),
			Ch:   &f.serviceCommon.Ctxlog,
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("user_hard_delete_worker"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
		},
		Config:             &f.configs.userHardDeleteWorkerConfig,
		EventFollowsClient: f.eventFollows,
		DBClient:           f.oracleDB,
	}

	f.liveEventLoader = &channelevent.LiveEventLoader{
		EventHandlers:    f.eventHandlers,
		OracleDB:         f.oracleDB,
		Clock:            f.clock,
		GetChannelClient: f.getChannelClient,
	}

	if f.pubsubClient == nil {
		pubsubClient, err := pubsub.NewPubSubClient(f.configs.clientConfig.pubsubURL.Get())
		if err != nil {
			return err
		}
		f.pubsubClient = pubsubClient
	}

	f.channelEventPublisher = &channelevent.Publisher{
		SQSQueueProcessor: feedsqs.SQSQueueProcessor{
			Conf: &f.configs.channelEventPublisherConfig.SQSQueueProcessorConfig,
			Log:  f.serviceCommon.Log,
			Sqs:  sqs.New(session, awsConf...),
			Ch:   &f.serviceCommon.Ctxlog,
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("channel_event_publisher"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
		},
		Config:          &f.configs.channelEventPublisherConfig,
		EventHandlers:   f.eventHandlers,
		PubsubClient:    f.pubsubClient,
		LiveEventLoader: f.liveEventLoader,
		Throttler:       channelEventCache,
	}

	if f.authUsersClient == nil {
		f.authUsersClient = &api.AuthUsersClientImpl{
			UsersClient: usersClient,
			Log:         f.serviceCommon.Log,
		}
	}

	return nil
}

func (f *service) foundationClient(host string) twitchclient.ClientConf {
	rtWrappers := []func(http.RoundTripper) http.RoundTripper{
		f.serviceCommon.XRay.RoundTripper,
	}
	return twitchclient.ClientConf{
		Host:                 host,
		Stats:                f.serviceCommon.Statsd,
		RoundTripperWrappers: rtWrappers,
		ElevateKey:           f.serviceCommon.ElevateLogKey,
		Logger:               f.serviceCommon.Log,
		DimensionKey:         &f.serviceCommon.CtxDimensions,
	}
}

func (f *service) inject() {
	f.server = api.HTTPServer{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Config: &f.configs.httpConfig.BaseHTTPServerConfig,
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
			Dims:        &f.serviceCommon.CtxDimensions,
			Log:         f.serviceCommon.Log,
			ElevateKey:  f.serviceCommon.ElevateLogKey,
			Ctxlog:      &f.serviceCommon.Ctxlog,
			OnListen:    f.onListen,
			PanicLogger: f.serviceCommon.PanicLogger,
			XRay:        f.serviceCommon.XRay,
		},
		Config: &f.configs.httpConfig,

		EventFollows:          f.eventFollows,
		EventHandlers:         f.eventHandlers,
		HypemanScheduler:      f.hypemanScheduler,
		HypemanWorker:         f.hypemanWorker,
		UserHardDeleteWorker:  f.userHardDeleteWorker,
		ImageUploader:         &f.imageUploader,
		JaxClient:             f.jaxClient,
		ChannelEventScheduler: f.channelEventScheduler,
		HallpassClient:        f.hallpassClient,
		SpadeClient:           f.spadeClient,
		OracleDB:              f.oracleDB,
		SuggestionsConfig:     &f.configs.suggestionsConfig,
		Clock:                 f.clock,
		LiveEventLoader:       f.liveEventLoader,
		ChannelEventPublisher: f.channelEventPublisher,
		AdminList:             f.adminList,
		AuthUsersClient:       f.authUsersClient,
	}
	f.server.SetupRoutes = f.server.Routes
	f.server.HealthCheck = f.server.CheckAWS

	f.runner = service_common.ServiceRunner{
		Log: f.serviceCommon.Log,
		Services: []service_common.Service{
			&f.server,
			&f.serviceCommon,
			&f.imageUploader,
			f.spadeClient,
			f.hypemanWorker,
			f.userHardDeleteWorker,
			f.cacheService,
			f.channelEventPublisher,
		},
		SigChan:      f.sigChan,
		SignalNotify: signal.Notify,
	}
}

func (f *service) main() {
	if err := f.setup(); err != nil {
		if f.serviceCommon.Log != nil {
			f.serviceCommon.Log.Log("err", err, "Unable to load initial config")
		}
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		time.Sleep(time.Millisecond * 150)
		f.osExit(1)
		return
	}
	f.inject()
	if err := f.runner.Execute(); err != nil {
		f.serviceCommon.Log.Log("err", err, "wait to end finished with an error")
		time.Sleep(time.Millisecond * 150)
		f.osExit(1)
		return
	}
	f.serviceCommon.Log.Log("Finished Main")
}

func main() {
	instance.main()
}
