// +build integration

package main

import (
	"context"
	"fmt"
	"net"
	"net/http"
	"os"
	"reflect"
	"strconv"
	"syscall"
	"testing"
	"time"

	"strings"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/spade"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/twitch-events/gea/cmd/gea/internal/api/mocks"
	"code.justin.tv/twitch-events/gea/cmd/gea/internal/api/models"
	"code.justin.tv/twitch-events/gea/internal/db"
	testutils "code.justin.tv/twitch-events/gea/internal/test-utils"
	"code.justin.tv/twitch-events/gea/internal/video"
	"code.justin.tv/twitch-events/gea/lib/geaclient"
	"code.justin.tv/vod/vodapi/rpc/vodapi"
	vodapi_utils "code.justin.tv/vod/vodapi/rpc/vodapi/utils"
	usersmodels "code.justin.tv/web/users-service/models"
	. "github.com/smartystreets/goconvey/convey"
	mock "github.com/stretchr/testify/mock"
)

var (
	languageEN               = "EN"
	languageKR               = "KR"
	overwatchGameID          = "488552"
	numericalOverwatchGameID = 488552
	irlGameID                = "345345" // Arbitrarily chosen value
	timeZoneAmericaLA        = "America/Los_Angeles"
	timeZoneEuropeBudapest   = "Europe/Budapest"
	defaultCoverImageID      = "town"

	jorasteUserID = "125690176" // Used for archive video tests.  Gary's non-twitch account
)

func timestamp() string {
	return time.Now().Format(time.RFC3339Nano)
}

func timestampUser() string {
	return strconv.FormatInt(time.Now().UnixNano(), 10)
}

func defaultCreateSingleEventParams(ownerID string, startTime time.Time, duration time.Duration) geaclient.CreateSingleEventParams {
	title := "TestIntegration_SingleDefault title " + timestamp()
	description := "TestIntegration_SingleDefault description" + timestamp()
	endTime := startTime.Add(duration)

	return geaclient.CreateSingleEventParams{
		OwnerID:     ownerID,
		StartTime:   startTime,
		EndTime:     endTime,
		Language:    languageEN,
		Title:       title,
		Description: description,
		ChannelID:   ownerID,
		GameID:      overwatchGameID,
	}
}

func TestGetMetadata(t *testing.T) {
	t.Parallel()
	ts := startServer(t, newDefaultInjectables())
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx
		ownerID := jorasteUserID

		Convey("Should be able to get the metadata of a single event", func() {
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			eventParams := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			cEv, cErr := client.CreateSingleEvent(ctx, eventParams, ownerID, nil)
			So(cErr, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			gE, gErr := client.GetEventMetadata(ctx, cEv.ID, nil, nil)
			So(gErr, ShouldBeNil)
			So(gE.ID, ShouldNotBeEmpty)
			So(gE.Title, ShouldNotBeEmpty)
			So(gE.Description, ShouldNotBeEmpty)
			So(gE.OGTitle, ShouldNotBeEmpty)
			So(gE.OGDescription, ShouldNotBeEmpty)
			So(gE.OGURL, ShouldNotBeEmpty)
			So(gE.OGImage, ShouldNotBeEmpty)
			So(gE.OGImageWidth, ShouldNotBeEmpty)
			So(gE.OGImageHeight, ShouldNotBeEmpty)
			So(gE.OGType, ShouldNotBeEmpty)
			So(gE.TwitterCard, ShouldNotBeEmpty)
			So(gE.TwitterTitle, ShouldNotBeEmpty)
			So(gE.TwitterDescription, ShouldNotBeEmpty)
			So(gE.TwitterImage, ShouldNotBeEmpty)
			So(gE.TwitterURL, ShouldNotBeEmpty)
		})

		Convey("Should be able to get the metadata of a single event that has no time zone", func() {
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			eventParams := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			eventParams.TimeZoneID = ""
			cEv, cErr := client.CreateSingleEvent(ctx, eventParams, ownerID, nil)
			So(cErr, ShouldBeNil)
			So(cEv, ShouldNotBeNil)
			So(cEv.TimeZoneID, ShouldBeEmpty)

			gE, gErr := client.GetEventMetadata(ctx, cEv.ID, nil, nil)
			So(gErr, ShouldBeNil)
			So(gE, ShouldNotBeNil)
			So(gE.Description, ShouldNotBeEmpty)

			// Since no time zone is set, the times should be localized to pacific time.
			if !strings.Contains(gE.Description, "PST") && !strings.Contains(gE.Description, "PDT") {
				t.Fatalf("times in metadata description \"%s\" were not localized to pacific time", gE.Description)
			}
		})

		Convey("Should be able to get the metadata of a single event spanning more than a day", func() {
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			eventParams := defaultCreateSingleEventParams(ownerID, startTime, 24*time.Hour)
			cEv, cErr := client.CreateSingleEvent(ctx, eventParams, ownerID, nil)
			So(cErr, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			gE, gErr := client.GetEventMetadata(ctx, cEv.ID, nil, nil)
			So(gErr, ShouldBeNil)
			So(gE.ID, ShouldNotBeEmpty)
			So(gE.Title, ShouldNotBeEmpty)
			So(gE.Description, ShouldNotBeEmpty)
			So(gE.OGTitle, ShouldNotBeEmpty)
			So(gE.OGDescription, ShouldNotBeEmpty)
			So(gE.OGURL, ShouldNotBeEmpty)
			So(gE.OGImage, ShouldNotBeEmpty)
			So(gE.OGImageWidth, ShouldNotBeEmpty)
			So(gE.OGImageHeight, ShouldNotBeEmpty)
			So(gE.OGType, ShouldNotBeEmpty)
			So(gE.TwitterCard, ShouldNotBeEmpty)
			So(gE.TwitterTitle, ShouldNotBeEmpty)
			So(gE.TwitterDescription, ShouldNotBeEmpty)
			So(gE.TwitterImage, ShouldNotBeEmpty)
			So(gE.TwitterURL, ShouldNotBeEmpty)
		})

		Convey("Should be able to get the metadata of a premiere event", func() {
			startTime := time.Now()
			createParams := defaultCreatePremiereEventParams(ownerID, startTime, time.Second)
			cEv, cErr := client.CreatePremiereEvent(ctx, createParams, ownerID, nil)
			So(cErr, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			gE, gErr := client.GetEventMetadata(ctx, cEv.ID, nil, nil)
			So(gErr, ShouldBeNil)
			So(gE.ID, ShouldNotBeEmpty)
			So(gE.Title, ShouldNotBeEmpty)
			So(gE.Description, ShouldNotBeEmpty)
			So(gE.OGTitle, ShouldNotBeEmpty)
			So(gE.OGDescription, ShouldNotBeEmpty)
			So(gE.OGURL, ShouldNotBeEmpty)
			So(gE.OGImage, ShouldNotBeEmpty)
			So(gE.OGImageWidth, ShouldNotBeEmpty)
			So(gE.OGImageHeight, ShouldNotBeEmpty)
			So(gE.OGType, ShouldNotBeEmpty)
			So(gE.TwitterCard, ShouldNotBeEmpty)
			So(gE.TwitterTitle, ShouldNotBeEmpty)
			So(gE.TwitterDescription, ShouldNotBeEmpty)
			So(gE.TwitterImage, ShouldNotBeEmpty)
			So(gE.TwitterURL, ShouldNotBeEmpty)
		})

		Convey("Should be able to get the metadata of a timetable and segment event", func() {
			cTt, ctErr := client.CreateTimetableEvent(ctx, timetableParams(ownerID), ownerID, nil)
			So(ctErr, ShouldBeNil)
			So(cTt, ShouldNotBeNil)

			// Testing on timetable with no segments
			gE, gErr := client.GetEventMetadata(ctx, cTt.ID, nil, nil)
			So(gErr, ShouldBeNil)
			So(gE.ID, ShouldNotBeEmpty)
			So(gE.Title, ShouldNotBeEmpty)
			So(gE.Description, ShouldNotBeNil)
			So(gE.OGTitle, ShouldNotBeEmpty)
			So(gE.OGDescription, ShouldNotBeNil)
			So(gE.OGURL, ShouldNotBeEmpty)
			So(gE.OGImage, ShouldNotBeEmpty)
			So(gE.OGImageWidth, ShouldNotBeEmpty)
			So(gE.OGImageHeight, ShouldNotBeEmpty)
			So(gE.OGType, ShouldNotBeEmpty)
			So(gE.TwitterCard, ShouldNotBeEmpty)
			So(gE.TwitterTitle, ShouldNotBeEmpty)
			So(gE.TwitterDescription, ShouldNotBeNil)
			So(gE.TwitterImage, ShouldNotBeEmpty)
			So(gE.TwitterURL, ShouldNotBeEmpty)

			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			endTime := startTime.Add(time.Hour)
			cSeg, csErr := client.CreateSegmentEvent(ctx, segmentParams(cTt.ID, ownerID, overwatchGameID, startTime, endTime), ownerID, nil)
			So(csErr, ShouldBeNil)
			So(cSeg, ShouldNotBeNil)

			gE, gErr = client.GetEventMetadata(ctx, cSeg.ID, nil, nil)
			So(gErr, ShouldBeNil)
			So(gE.ID, ShouldNotBeEmpty)
			So(gE.Title, ShouldNotBeEmpty)
			So(gE.Description, ShouldNotBeEmpty)
			So(gE.OGTitle, ShouldNotBeEmpty)
			So(gE.OGDescription, ShouldNotBeEmpty)
			So(gE.OGURL, ShouldNotBeEmpty)
			So(gE.OGImage, ShouldNotBeEmpty)
			So(gE.OGImageWidth, ShouldNotBeEmpty)
			So(gE.OGImageHeight, ShouldNotBeEmpty)
			So(gE.OGType, ShouldNotBeEmpty)
			So(gE.TwitterCard, ShouldNotBeEmpty)
			So(gE.TwitterTitle, ShouldNotBeEmpty)
			So(gE.TwitterDescription, ShouldNotBeEmpty)
			So(gE.TwitterImage, ShouldNotBeEmpty)
			So(gE.TwitterURL, ShouldNotBeEmpty)

			// Testing on timetable with segments
			gE, gErr = client.GetEventMetadata(ctx, cTt.ID, nil, nil)
			So(gErr, ShouldBeNil)
			So(gE.ID, ShouldNotBeEmpty)
			So(gE.Title, ShouldNotBeEmpty)
			So(gE.Description, ShouldNotBeEmpty)
			So(gE.OGTitle, ShouldNotBeEmpty)
			So(gE.OGDescription, ShouldNotBeEmpty)
			So(gE.OGURL, ShouldNotBeEmpty)
			So(gE.OGImage, ShouldNotBeEmpty)
			So(gE.OGImageWidth, ShouldNotBeEmpty)
			So(gE.OGImageHeight, ShouldNotBeEmpty)
			So(gE.OGType, ShouldNotBeEmpty)
			So(gE.TwitterCard, ShouldNotBeEmpty)
			So(gE.TwitterTitle, ShouldNotBeEmpty)
			So(gE.TwitterDescription, ShouldNotBeEmpty)
			So(gE.TwitterImage, ShouldNotBeEmpty)
			So(gE.TwitterURL, ShouldNotBeEmpty)
		})
	})
}

func TestIntegration_Single(t *testing.T) {
	t.Parallel()
	ts := startServer(t, newDefaultInjectables())
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		Convey("Should be able to create and get a single event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			endTime := startTime.Add(time.Hour)
			title := "TestIntegration_Single title " + timestamp()
			description := "TestIntegration_Single description" + timestamp()

			cEv, cErr := client.CreateSingleEvent(ctx, geaclient.CreateSingleEventParams{
				OwnerID:     ownerID,
				StartTime:   startTime,
				EndTime:     endTime,
				TimeZoneID:  timeZoneAmericaLA,
				Language:    languageEN,
				Title:       title,
				Description: description,
				ChannelID:   ownerID,
				GameID:      overwatchGameID,
			}, ownerID, nil)
			So(cErr, ShouldBeNil)
			So(cEv, ShouldNotBeNil)
			So(cEv.ID, ShouldNotBeEmpty)
			So(cEv.OwnerID, ShouldEqual, ownerID)
			So(cEv.Type, ShouldEqual, "single")
			So(cEv.StartTime, ShouldEqual, startTime)
			So(cEv.EndTime, ShouldEqual, endTime)
			So(cEv.TimeZoneID, ShouldEqual, timeZoneAmericaLA)
			So(cEv.Language, ShouldEqual, languageEN)
			So(cEv.Title, ShouldEqual, title)
			So(cEv.Description, ShouldEqual, description)
			So(cEv.ChannelID, ShouldEqual, ownerID)
			So(cEv.GameID, ShouldEqual, overwatchGameID)

			gE, gErr := client.GetEvent(ctx, cEv.ID, nil, nil)
			So(gErr, ShouldBeNil)
			So(gE, ShouldHaveSameTypeAs, &geaclient.SingleEvent{})
			gEv := gE.(*geaclient.SingleEvent)
			So(gEv.ID, ShouldEqual, cEv.ID)
			So(gEv.OwnerID, ShouldEqual, ownerID)
			So(gEv.Type, ShouldEqual, "single")
			So(gEv.StartTime, ShouldEqual, startTime)
			So(gEv.EndTime, ShouldEqual, endTime)
			So(gEv.TimeZoneID, ShouldEqual, timeZoneAmericaLA)
			So(gEv.Language, ShouldEqual, languageEN)
			So(gEv.Title, ShouldEqual, title)
			So(gEv.Description, ShouldEqual, description)
			So(gEv.ChannelID, ShouldEqual, ownerID)
			So(gEv.GameID, ShouldEqual, overwatchGameID)
		})

		Convey("Should not be able to create create event without an owner", func() {
			ownerID := timestampUser()
			startTime := time.Now().UTC().Add(time.Hour)

			params := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			params.OwnerID = ""

			ev, err := client.CreateSingleEvent(ctx, params, ownerID, nil)
			So(err, ShouldNotBeNil)
			So(geaclient.ErrorCode(err), ShouldEqual, http.StatusBadRequest)
			So(ev, ShouldBeNil)
		})

		Convey("Should not be able to create single event with the same start and end time", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			createParams := defaultCreateSingleEventParams(ownerID, startTime, 0)
			cEv, err := client.CreateSingleEvent(ctx, createParams, ownerID, nil)
			So(err, ShouldNotBeNil)
			So(cEv, ShouldBeNil)
		})

		Convey("Should not be able to create single event with an invalid time zone", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			createParams := defaultCreateSingleEventParams(ownerID, startTime, 0)
			createParams.TimeZoneID = "USA/Los_Angeles"

			cEv, err := client.CreateSingleEvent(ctx, createParams, ownerID, nil)
			So(err, ShouldNotBeNil)
			So(geaclient.ErrorCode(err), ShouldEqual, http.StatusBadRequest)
			So(cEv, ShouldBeNil)
		})

		Convey("Should not be able to create a single event for a different owner", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			params := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)

			userID := timestampUser()
			So(ownerID, ShouldNotEqual, userID)

			_, cErr := client.CreateSingleEvent(ctx, params, userID, nil)
			So(cErr, ShouldNotBeNil)
			So(geaclient.ErrorCode(cErr), ShouldEqual, http.StatusForbidden)
		})

		Convey("Should not be able to create a single event for a different channel", func() {
			ownerID := timestampUser()
			params := defaultCreateSingleEventParams(ownerID, time.Now(), time.Hour)
			params.ChannelID = timestampUser()

			_, cErr := client.CreateSingleEvent(ctx, params, ownerID, nil)
			So(cErr, ShouldNotBeNil)
			So(geaclient.ErrorCode(cErr), ShouldEqual, http.StatusForbidden)
		})

		Convey("Should be able to create and get multiple events", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			endTime := startTime.Add(time.Hour)
			title := "TestingIntegration_Single title " + timestamp()
			description := "TestingIntegration_Single description " + timestamp()

			ev1, cErr := client.CreateSingleEvent(ctx, geaclient.CreateSingleEventParams{
				OwnerID:     ownerID,
				StartTime:   startTime,
				EndTime:     endTime,
				TimeZoneID:  timeZoneAmericaLA,
				Language:    languageEN,
				Title:       title,
				Description: description,
				ChannelID:   ownerID,
				GameID:      overwatchGameID,
			}, ownerID, nil)
			So(cErr, ShouldBeNil)

			ev2, cErr := client.CreateSingleEvent(ctx, geaclient.CreateSingleEventParams{
				OwnerID:     ownerID,
				StartTime:   startTime,
				EndTime:     endTime,
				TimeZoneID:  timeZoneAmericaLA,
				Language:    languageEN,
				Title:       title,
				Description: description,
				ChannelID:   ownerID,
				GameID:      overwatchGameID,
			}, ownerID, nil)

			So(cErr, ShouldBeNil)

			rEvents, rErr := client.GetEvents(ctx, []string{ev1.ID, ev2.ID, "blahblah"}, nil, nil)
			So(rErr, ShouldBeNil)
			So(rEvents, ShouldHaveLength, 2)
			So(rEvents[0].GetID(), ShouldEqual, ev1.ID)
			So(rEvents[1].GetID(), ShouldEqual, ev2.ID)
		})

		Convey("Should be able update a single event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			createParams := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)

			cEv, cErr := client.CreateSingleEvent(ctx, createParams, ownerID, nil)
			So(cErr, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			Convey("When all fields are updated", func() {
				updatedStartTime := createParams.StartTime.Add(time.Minute)
				updatedEndTime := createParams.StartTime.Add(time.Hour)
				updatedTimeZone := timeZoneEuropeBudapest
				updatedLanguage := languageKR
				updatedTitle := "TestingIntegration_Single updated title " + timestamp()
				updatedDesc := "TestingIntegration_Single updated desc " + timestamp()
				updatedGameID := irlGameID

				uEv, uErr := client.UpdateSingleEvent(ctx, cEv.ID, geaclient.UpdateSingleEventParams{
					StartTime:   &updatedStartTime,
					EndTime:     &updatedEndTime,
					TimeZoneID:  &updatedTimeZone,
					Language:    &updatedLanguage,
					Title:       &updatedTitle,
					Description: &updatedDesc,
					GameID:      &updatedGameID,
				}, ownerID, nil)

				So(uErr, ShouldBeNil)
				So(uEv, ShouldNotBeNil)

				So(uEv.StartTime, ShouldEqual, updatedStartTime)
				So(uEv.EndTime, ShouldEqual, updatedEndTime)
				So(uEv.TimeZoneID, ShouldEqual, updatedTimeZone)
				So(uEv.Language, ShouldEqual, updatedLanguage)
				So(uEv.Title, ShouldEqual, updatedTitle)
				So(uEv.Description, ShouldEqual, updatedDesc)
				So(uEv.GameID, ShouldEqual, updatedGameID)
			})

			Convey("When some fields are updated", func() {
				updatedTitle := "TestingIntegration_Single updated title " + timestamp()

				uEv, uErr := client.UpdateSingleEvent(ctx, cEv.ID, geaclient.UpdateSingleEventParams{
					Title: &updatedTitle,
				}, ownerID, nil)

				So(uErr, ShouldBeNil)
				So(uEv, ShouldNotBeNil)

				// Assert that only the supplied field was updated.
				So(uEv.Title, ShouldEqual, updatedTitle)

				// Assert that the remaining fields are unchanged.
				So(uEv.OwnerID, ShouldEqual, cEv.OwnerID)
				So(uEv.StartTime, ShouldEqual, cEv.StartTime)
				So(uEv.EndTime, ShouldEqual, cEv.EndTime)
				So(uEv.Language, ShouldEqual, cEv.Language)
				So(uEv.Description, ShouldEqual, cEv.Description)
				So(uEv.ChannelID, ShouldEqual, cEv.ChannelID)
				So(uEv.GameID, ShouldEqual, cEv.GameID)
				So(uEv.TimeZoneID, ShouldEqual, cEv.TimeZoneID)
			})
		})

		Convey("Should not be able update events owned by a different user", func() {
			ownerID := timestampUser()
			createParams := defaultCreateSingleEventParams(ownerID, time.Now(), time.Hour)
			cEv, err := client.CreateSingleEvent(ctx, createParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			userID := timestampUser()
			So(userID, ShouldNotEqual, ownerID)

			updatedTitle := "TestingIntegration_Single updated title " + timestamp()
			uEv, err := client.UpdateSingleEvent(ctx, cEv.ID, geaclient.UpdateSingleEventParams{
				Title: &updatedTitle,
			}, userID, nil)
			So(err, ShouldNotBeNil)
			So(clients.ErrorCode(err), ShouldEqual, http.StatusForbidden)
			So(uEv, ShouldBeNil)
		})

		// For now, only admins can change the channel an event is broadcast on.
		Convey("Non-admins should not be able to broadcast event on another channel", func() {
			ownerID := timestampUser()
			createParams := defaultCreateSingleEventParams(ownerID, time.Now(), time.Hour)
			cEv, err := client.CreateSingleEvent(ctx, createParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			updatedChannelID := timestampUser()
			uEv, err := client.UpdateSingleEvent(ctx, cEv.ID, geaclient.UpdateSingleEventParams{
				ChannelID: &updatedChannelID,
			}, ownerID, nil)
			So(err, ShouldNotBeNil)
			So(clients.ErrorCode(err), ShouldEqual, http.StatusForbidden)
			So(uEv, ShouldBeNil)
		})

		// For now, only admins can change the owner of an event.
		Convey("Non-admins should not be able to change the owner of an event", func() {
			ownerID := timestampUser()
			createParams := defaultCreateSingleEventParams(ownerID, time.Now(), time.Hour)
			cEv, err := client.CreateSingleEvent(ctx, createParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			updatedOwnerID := timestampUser()
			uEv, err := client.UpdateSingleEvent(ctx, cEv.ID, geaclient.UpdateSingleEventParams{
				OwnerID: &updatedOwnerID,
			}, ownerID, nil)
			So(err, ShouldNotBeNil)
			So(clients.ErrorCode(err), ShouldEqual, http.StatusForbidden)
			So(uEv, ShouldBeNil)
		})

		Convey("A user should be able to delete their own event", func() {
			ownerID := timestampUser()
			createParams := defaultCreateSingleEventParams(ownerID, time.Now(), time.Hour)
			cEv, err := client.CreateSingleEvent(ctx, createParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			dEv, dErr := client.DeleteEvent(ctx, cEv.ID, ownerID, nil)
			So(dErr, ShouldBeNil)
			So(dEv, ShouldNotBeNil)
			So(dEv.GetID(), ShouldEqual, cEv.ID)

			_, gErr := client.GetEvent(ctx, cEv.ID, nil, nil)
			So(gErr, ShouldNotBeNil)
			So(geaclient.ErrorCode(gErr), ShouldEqual, http.StatusNotFound)

			Convey("Should be able to get that event even if deleted", func() {
				gEv, gErr := client.GetEvent(ctx, cEv.ID, &geaclient.GetEventOpts{GetDeleted: true}, nil)
				So(gErr, ShouldBeNil)
				So(gEv, ShouldNotBeNil)
				So(gEv.GetID(), ShouldEqual, cEv.ID)
			})

			Convey("Should be able to bulk get that event even if deleted", func() {
				gEv, gErr := client.GetEvents(ctx, []string{cEv.ID}, &geaclient.GetEventsOpts{GetDeleted: true}, nil)
				So(gErr, ShouldBeNil)
				So(gEv, ShouldHaveLength, 1)
				So(gEv[0].GetID(), ShouldEqual, cEv.ID)
			})
		})

		Convey("A user should not be able to delete someone else's event", func() {
			ownerID := timestampUser()
			createParams := defaultCreateSingleEventParams(ownerID, time.Now(), time.Hour)
			cEv, err := client.CreateSingleEvent(ctx, createParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			userID := timestampUser()
			So(userID, ShouldNotEqual, ownerID)

			dEv, dErr := client.DeleteEvent(ctx, cEv.ID, userID, nil)
			So(dErr, ShouldNotBeNil)
			So(geaclient.ErrorCode(dErr), ShouldEqual, http.StatusForbidden)
			So(dEv, ShouldBeNil)
		})

		Convey("Should be able to create non-UTC events and get them back in UTC", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.FixedZone("America/Los_Angeles", -8*3600))

			cEvParams := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			cEv, cErr := client.CreateSingleEvent(ctx, cEvParams, ownerID, nil)
			So(cErr, ShouldBeNil)
			So(cEv, ShouldNotBeNil)
			So(cEv.StartTime, ShouldEqual, startTime.UTC())
			So(cEv.EndTime, ShouldEqual, startTime.UTC().Add(time.Hour))

			Convey("updating it to non-UTC should also change it back to UTC", func() {
				startTime2 := time.Date(2017, 01, 02, 01, 02, 03, 00, time.FixedZone("America/Los_Angeles", -8*3600))
				endTime2 := startTime2.Add(time.Hour)
				uEv, uErr := client.UpdateSingleEvent(ctx, cEv.ID, geaclient.UpdateSingleEventParams{
					StartTime: &startTime2,
					EndTime:   &endTime2,
				}, ownerID, nil)
				So(uErr, ShouldBeNil)
				So(uEv, ShouldNotBeNil)
				So(uEv.StartTime, ShouldEqual, startTime2.UTC())
				So(uEv.EndTime, ShouldEqual, endTime2.UTC())
			})
		})

		Convey("Should be able to query for events", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev1Params := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			ev1, err := client.CreateSingleEvent(ctx, ev1Params, ownerID, nil)
			So(err, ShouldBeNil)
			So(ev1, ShouldNotBeNil)

			ev2Params := defaultCreateSingleEventParams(ownerID, startTime.Add(time.Hour*2), time.Hour)
			ev2, err := client.CreateSingleEvent(ctx, ev2Params, ownerID, nil)
			So(err, ShouldBeNil)
			So(ev2, ShouldNotBeNil)

			Convey("filtering for events on a channel that start after a given time", func() {
				ids, err := client.GetEventIDsByChannelIDs(
					ctx, []string{ownerID}, &geaclient.GetEIDsByCIDsOpts{StartTimeAfter: &startTime}, nil)
				So(err, ShouldBeNil)
				So(ids.EventIDs, ShouldHaveLength, 2)
				So(ids.EventIDs[0], ShouldEqual, ev1.ID)
				So(ids.EventIDs[1], ShouldEqual, ev2.ID)

				Convey("time window should work if not in UTC", func() {
					nycLocation, err := time.LoadLocation("America/New_York")
					So(err, ShouldBeNil)
					So(nycLocation, ShouldNotBeNil)
					nycStartTime := startTime.In(nycLocation)
					ids, err := client.GetEventIDsByChannelIDs(
						ctx, []string{ownerID}, &geaclient.GetEIDsByCIDsOpts{StartTimeAfter: &nycStartTime}, nil)

					So(err, ShouldBeNil)
					So(ids.EventIDs, ShouldHaveLength, 2)
					So(ids.EventIDs[0], ShouldEqual, ev1.ID)
					So(ids.EventIDs[1], ShouldEqual, ev2.ID)
				})
			})

			Convey("filtering for events on a channel that end within a given window", func() {
				endTimeAfter := ev2.EndTime
				endTimeBefore := endTimeAfter.Add(time.Minute)
				opts := &geaclient.GetEIDsByCIDsOpts{
					EndTimeAfter:  &endTimeAfter,
					EndTimeBefore: &endTimeBefore,
				}
				ids, err := client.GetEventIDsByChannelIDs(ctx, []string{ownerID}, opts, nil)

				So(err, ShouldBeNil)
				So(ids.EventIDs, ShouldHaveLength, 1)
				So(ids.EventIDs[0], ShouldEqual, ev2.ID)
				So(ids.Cursor, ShouldBeBlank)
				Convey("time window should work if not in UTC", func() {
					nycLocation, err := time.LoadLocation("America/New_York")
					So(err, ShouldBeNil)
					So(nycLocation, ShouldNotBeNil)

					nycEndTimeAfter := endTimeAfter.In(nycLocation)
					nycEndTimeBefore := endTimeBefore.In(nycLocation)
					nycOpts := &geaclient.GetEIDsByCIDsOpts{
						EndTimeAfter:  &nycEndTimeAfter,
						EndTimeBefore: &nycEndTimeBefore,
					}
					ids, err := client.GetEventIDsByChannelIDs(ctx, []string{ownerID}, nycOpts, nil)

					So(err, ShouldBeNil)
					So(ids.EventIDs, ShouldHaveLength, 1)
					So(ids.EventIDs[0], ShouldEqual, ev2.ID)
					So(ids.Cursor, ShouldBeBlank)
				})
			})

			Convey("filtering for events of a certain type that start after a given time", func() {
				g1, err := client.GetEventIDsByTypes(ctx, []string{geaclient.EventTypeSingle}, &geaclient.GetEIDsByTypesOpts{
					StartTimeAfter: &startTime,
					Limit:          refInt(1),
				}, nil)
				So(err, ShouldBeNil)
				So(g1.EventIDs, ShouldHaveLength, 1)
				So(g1.Cursor, ShouldNotBeEmpty)

				g2, err := client.GetEventIDsByTypes(ctx, []string{geaclient.EventTypeSingle}, &geaclient.GetEIDsByTypesOpts{
					StartTimeAfter: &startTime,
					Limit:          refInt(1),
					Cursor:         &g1.Cursor,
				}, nil)
				So(err, ShouldBeNil)
				So(g2.EventIDs, ShouldHaveLength, 1)
				So(g2.EventIDs[0], ShouldNotEqual, g1.EventIDs[0])
			})
		})

		Convey("Should be able to get game suggestions", func() {
			ownerID := timestampUser()
			startTime := time.Now().UTC().Add(time.Hour)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			sug, err := client.GetEventSuggestionsForGame(ctx, overwatchGameID, nil, nil)
			So(err, ShouldBeNil)
			So(sug, ShouldNotBeNil)
			So(len(sug.Future), ShouldBeGreaterThan, 0)
		})
	})
}

func TestIntegration_FeaturedSuggestions(t *testing.T) {
	featuredChannelID := timestampUser()
	ts := startServer(t, newDefaultInjectables(), map[string][]byte{
		"gea.suggestions.config_by_game": []byte(`
			{
				"` + overwatchGameID + `": {
					"featured_channels": ["` + featuredChannelID + `"]
				}
			}`),
		"gea.suggestions.limits.future": []byte("1"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		Convey("Featured events should always be suggested", func() {
			randomChannelID := timestampUser()

			startTime := time.Now().UTC().Add(time.Hour)

			featuredEv, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(featuredChannelID, startTime, time.Hour), featuredChannelID, nil)
			So(err, ShouldBeNil)
			So(featuredEv, ShouldNotBeNil)

			randomEv, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(randomChannelID, startTime, time.Hour), randomChannelID, nil)
			So(err, ShouldBeNil)
			So(randomEv, ShouldNotBeNil)

			// This makes sure that the random event we just created has a higher hype score than the featured event
			err = client.FollowEvent(ctx, randomEv.ID, randomChannelID, nil)
			So(err, ShouldBeNil)

			sug, err := client.GetEventSuggestionsForGame(ctx, overwatchGameID, nil, nil)
			So(err, ShouldBeNil)
			So(sug, ShouldNotBeNil)
			So(len(sug.Future), ShouldBeGreaterThan, 0)

			events, err := client.GetEvents(ctx, sug.Future, nil, nil)
			So(err, ShouldBeNil)
			So(len(events), ShouldBeGreaterThan, 0)
			// We check for channel here because there's a possible race condition where if another event
			// is created on our featured channel that has a high follow count, we won't get the event we created
			hasFeaturedEvent := false
			for _, e := range events {
				if ev, ok := e.(*geaclient.SingleEvent); ok {
					if ev.ChannelID == featuredChannelID {
						hasFeaturedEvent = true
					}
				}
			}
			So(hasFeaturedEvent, ShouldBeTrue)
		})
	})
}

func TestIntegration_AllowSpecialUsersToCreateOnAnyChannel(t *testing.T) {
	t.Parallel()

	injectables := newDefaultInjectables()
	injectables.adminList = nil

	ts := startServer(t, injectables, map[string][]byte{
		"gea.admin_users": []byte(",special_user_id,"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		Convey("Special user should be able to create an event on someone else's channel", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			createParams := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)

			cEv, cErr := client.CreateSingleEvent(ctx, createParams, "special_user_id", nil)
			So(cErr, ShouldBeNil)
			So(cEv, ShouldNotBeNil)
		})
	})
}

// There will be a period of time where the database contains events that don't have timezones, and events will be
// created without time zones.  These tests are meant to verify that the API will still work until Visage and the
// db are fully updated.
func TestIntegration_SingleEventsWithoutTimeZones(t *testing.T) {
	t.Parallel()
	ts := startServer(t, newDefaultInjectables())
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		Convey("Should be able to create a single event without a time zone", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			endTime := startTime.Add(time.Hour)

			cEv, cErr := client.CreateSingleEvent(ctx, geaclient.CreateSingleEventParams{
				OwnerID:     ownerID,
				StartTime:   startTime,
				EndTime:     endTime,
				Language:    languageEN,
				Title:       "TestIntegration_Single title " + timestamp(),
				Description: "TestIntegration_Single description" + timestamp(),
				ChannelID:   ownerID,
				GameID:      overwatchGameID,
			}, ownerID, nil)
			So(cErr, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			eventID := cEv.ID

			Convey("Should be able to get a single event that doesn't have a time zone", func() {
				ev, err := client.GetEvent(ctx, eventID, nil, nil)
				So(err, ShouldBeNil)
				So(ev, ShouldNotBeNil)
			})

			Convey("Should be able to update a single event that doesn't have a time zone", func() {
				updatedDesc := "TestIntegration_Single updated description" + timestamp()
				ev, err := client.UpdateSingleEvent(ctx, eventID, geaclient.UpdateSingleEventParams{
					Description: &updatedDesc,
				}, ownerID, nil)
				So(err, ShouldBeNil)
				So(ev, ShouldNotBeNil)
			})

			Convey("Should be able to delete a single event that doesn't have a time zone", func() {
				ev, err := client.DeleteSingleEvent(ctx, eventID, ownerID, nil)
				So(err, ShouldBeNil)
				So(ev, ShouldNotBeNil)
			})
		})
	})
}

func TestIntegration_FilteringEventsOwnedByBannedUsers(t *testing.T) {
	bannedUser := timestampUser()

	injectables := newDefaultInjectables()
	getUsersClient := NewGetUsersClientStub()
	getUsersClient.SetBannedUsers([]string{bannedUser})
	injectables.getUsersClient = getUsersClient

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		Convey("GetEvent should return a 404 when given an event ID that is owned by a banned user", func() {
			startTime := time.Now().UTC().Add(time.Hour)
			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(bannedUser, startTime, time.Hour), bannedUser, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			event, err := client.GetEvent(ctx, ev.ID, nil, nil)
			So(geaclient.ErrorCode(err), ShouldEqual, http.StatusNotFound)
			So(event, ShouldBeNil)
		})

		Convey("GetEvents should not return events owned by banned users", func() {
			startTime := time.Now().UTC().Add(time.Hour)
			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(bannedUser, startTime, time.Hour), bannedUser, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)
			bannedUserEventID := ev.ID

			validUser := timestampUser()
			ev, err = client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(validUser, startTime, time.Hour), validUser, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)
			validUserEventID := ev.ID

			events, err := client.GetEvents(ctx, []string{bannedUserEventID, validUserEventID}, nil, nil)
			So(err, ShouldBeNil)
			So(events, ShouldHaveLength, 1)
			So(events[0], ShouldNotBeNil)
			So(events[0].GetID(), ShouldEqual, validUserEventID)
		})
	})
}

func TestIntegration_Cache(t *testing.T) {
	t.Parallel()
	ts := startServer(t, newDefaultInjectables())
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		Convey("GetEvent should return a cached event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			cEv, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			ev1, err := client.GetEvent(ctx, cEv.ID, nil, nil)
			So(err, ShouldBeNil)
			So(ev1, ShouldNotBeNil)
			So(ev1.GetID(), ShouldEqual, cEv.ID)

			// This bypasses cache invalidation
			newTitle := "test_" + timestamp()
			_, err = ts.thisInstance.innerDBClient.Exec("update event_nodes set title = $1 where id = $2", newTitle, cEv.ID)
			So(err, ShouldBeNil)

			// This should fetch the cached version
			ev2, err := client.GetEvent(ctx, cEv.ID, nil, nil)
			So(err, ShouldBeNil)
			So(ev2, ShouldNotBeNil)
			sEv2 := ev2.(*geaclient.SingleEvent)
			So(sEv2.Title, ShouldEqual, cEv.Title)

			// This should fetch the db version
			ev3, err := client.GetEvent(ctx, cEv.ID, &geaclient.GetEventOpts{SkipCache: refBool(true)}, nil)
			So(err, ShouldBeNil)
			So(ev3, ShouldNotBeNil)
			sEv3 := ev3.(*geaclient.SingleEvent)
			So(sEv3.Title, ShouldNotEqual, cEv.Title)
			So(sEv3.Title, ShouldEqual, newTitle)
		})

		Convey("GetEvents should return cached events", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			e1, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(e1, ShouldNotBeNil)

			// This puts it into the cache
			ges1, err := client.GetEvents(ctx, []string{e1.ID}, nil, nil)
			So(err, ShouldBeNil)
			So(ges1, ShouldHaveLength, 1)
			So(ges1[0].GetID(), ShouldEqual, e1.ID)

			// This bypasses cache invalidation
			newTitle := "test_" + timestamp()
			_, err = ts.thisInstance.innerDBClient.Exec("update event_nodes set title = $1 where id = $2", newTitle, e1.ID)
			So(err, ShouldBeNil)

			e2, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(e2, ShouldNotBeNil)

			// This should fetch from both the cache and the db
			ges2, err := client.GetEvents(ctx, []string{e1.ID, e2.ID}, nil, nil)
			So(err, ShouldBeNil)
			So(ges2, ShouldHaveLength, 2)
			ge1 := findEvent(ges2, e1.ID)
			So(ge1, ShouldNotBeNil)
			sge1 := ge1.(*geaclient.SingleEvent)
			So(sge1.Title, ShouldEqual, e1.Title)
			ge2 := findEvent(ges2, e2.ID)
			So(ge2, ShouldNotBeNil)
			So(ge2.GetID(), ShouldEqual, e2.ID)

			// This should fetch from just the db
			ges3, err := client.GetEvents(ctx, []string{e1.ID, e2.ID}, &geaclient.GetEventsOpts{SkipCache: refBool(true)}, nil)
			So(err, ShouldBeNil)
			So(ges3, ShouldHaveLength, 2)
			ge3 := findEvent(ges3, e1.ID)
			So(ge3, ShouldNotBeNil)
			sge3 := ge3.(*geaclient.SingleEvent)
			So(sge3.Title, ShouldEqual, newTitle)
		})

		Convey("GetEventMetadata should return a cached event", func() {
			ownerID := jorasteUserID
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			cEv, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			ev1, err := client.GetEventMetadata(ctx, cEv.ID, nil, nil)
			So(err, ShouldBeNil)
			So(ev1, ShouldNotBeNil)
			So(ev1.ID, ShouldEqual, cEv.ID)

			// This bypasses cache invalidation
			newTitle := "test_" + timestamp()
			_, err = ts.thisInstance.innerDBClient.Exec("update event_nodes set title = $1 where id = $2", newTitle, cEv.ID)
			So(err, ShouldBeNil)

			// This should fetch the cached version
			ev2, err := client.GetEventMetadata(ctx, cEv.ID, nil, nil)
			So(err, ShouldBeNil)
			So(ev2, ShouldNotBeNil)
			sEv2 := ev2
			So(sEv2.Title, ShouldContainSubstring, cEv.Title)

			// This should fetch the db version
			ev3, err := client.GetEventMetadata(ctx, cEv.ID, &geaclient.GetEventMetadataOpts{SkipCache: refBool(true)}, nil)
			So(err, ShouldBeNil)
			So(ev3, ShouldNotBeNil)
			sEv3 := ev3
			So(sEv3.Title, ShouldNotContainSubstring, cEv.Title)
			So(sEv3.Title, ShouldContainSubstring, newTitle)
		})
	})
}

type eventsErrorPair struct {
	events []geaclient.Event
	err    error
}

func TestIntegration_ThrottlingGetEvent(t *testing.T) {
	t.Parallel()
	ts := startServer(t, newDefaultInjectables())
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		Convey("Simultaneous requests to GetEvents should return valid results", func() {
			// Given several new events and their IDs ...
			numEventIDs := 3
			eventIDs := make([]string, 0)
			for i := 0; i < numEventIDs; i++ {
				ownerID := timestampUser()
				startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
				cEv, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
				So(err, ShouldBeNil)
				So(cEv, ShouldNotBeNil)

				eventIDs = append(eventIDs, cEv.ID)
			}

			resultChan := make(chan *eventsErrorPair)

			// Launch several goroutines to get the same events.  Assuming other tests have not loaded the events
			// we created (possible since our tests run in parallel and share a database, but unlikely), the server
			// should be processing several requests for events that are not in the cache.  The server should have
			// one of these requests query the database, and the other requests should wait on the results of the
			// first.
			//
			// We can't explicitly verify that this happened from an integration test, but we can verify that
			// the server did not panic and returned valid responses.

			numGoroutines := 4
			for i := 0; i < numGoroutines; i++ {
				go func() {
					retrievedEvents, err := client.GetEvents(ctx, eventIDs, nil, nil)

					// We can't do assertions on the results in this goroutine - GoConvey doesn't allow it.
					// Instead we push the results in a channel that the test's main goroutine can read and do assertions
					// on.
					resultChan <- &eventsErrorPair{
						events: retrievedEvents,
						err:    err,
					}
				}()
			}

			// Accumulate the results from the different goroutines in a slice.
			results := make([]*eventsErrorPair, 0, numGoroutines)
			for i := 0; i < numGoroutines; i++ {
				var result *eventsErrorPair

				select {
				case result = <-resultChan:

				// If a goroutine never returns, kill the test.
				case <-time.After(time.Second * 2):
					t.Fatal("TestIntegration_ThrottlingGetEvent timed out waiting for GetEvents to return")
				}

				results = append(results, result)
			}

			for _, result := range results {
				So(result, ShouldNotBeNil)
				So(result.err, ShouldBeNil)
				So(result.events, ShouldNotBeNil)
				So(result.events, ShouldHaveLength, numEventIDs)
			}
		})
	})
}

func defaultCreatePremiereEventParams(ownerID string, startTime time.Time, duration time.Duration) geaclient.CreatePremiereEventParams {
	endTime := startTime.Add(duration)

	return geaclient.CreatePremiereEventParams{
		OwnerID:     ownerID,
		StartTime:   startTime,
		EndTime:     endTime,
		TimeZoneID:  timeZoneAmericaLA,
		Language:    languageEN,
		Title:       "TestIntegration_Premiere title " + timestamp(),
		Description: "TestIntegration_Premiere description" + timestamp(),
		ChannelID:   ownerID,
		GameID:      overwatchGameID,
		PremiereID:  "premiere_" + timestamp(),
	}
}

func TestIntegration_Premiere(t *testing.T) {
	t.Parallel()
	ts := startServer(t, newDefaultInjectables())
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		Convey("Should be able to create and get a premiere event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			endTime := startTime.Add(time.Hour)
			title := "TestIntegration_Premiere title " + timestamp()
			description := "TestIntegration_Premiere description" + timestamp()
			premiereID := "premiere_" + timestamp()

			cEv, cErr := client.CreatePremiereEvent(ctx, geaclient.CreatePremiereEventParams{
				OwnerID:     ownerID,
				StartTime:   startTime,
				EndTime:     endTime,
				TimeZoneID:  timeZoneAmericaLA,
				Language:    languageEN,
				Title:       title,
				Description: description,
				ChannelID:   ownerID,
				GameID:      overwatchGameID,
				PremiereID:  premiereID,
			}, ownerID, nil)
			So(cErr, ShouldBeNil)
			So(cEv, ShouldNotBeNil)
			So(cEv.ID, ShouldNotBeEmpty)
			So(cEv.OwnerID, ShouldEqual, ownerID)
			So(cEv.Type, ShouldEqual, "premiere")
			So(cEv.StartTime, ShouldEqual, startTime)
			So(cEv.EndTime, ShouldEqual, endTime)
			So(cEv.TimeZoneID, ShouldEqual, timeZoneAmericaLA)
			So(cEv.Language, ShouldEqual, languageEN)
			So(cEv.Title, ShouldEqual, title)
			So(cEv.Description, ShouldEqual, description)
			So(cEv.ChannelID, ShouldEqual, ownerID)
			So(cEv.GameID, ShouldEqual, overwatchGameID)
			So(cEv.PremiereID, ShouldEqual, premiereID)

			gE, gErr := client.GetEvent(ctx, cEv.ID, nil, nil)
			So(gErr, ShouldBeNil)
			So(gE, ShouldHaveSameTypeAs, &geaclient.PremiereEvent{})
			gEv := gE.(*geaclient.PremiereEvent)
			So(gEv.ID, ShouldEqual, cEv.ID)
			So(gEv.OwnerID, ShouldEqual, ownerID)
			So(gEv.Type, ShouldEqual, "premiere")
			So(gEv.StartTime, ShouldEqual, startTime)
			So(gEv.EndTime, ShouldEqual, endTime)
			So(gEv.TimeZoneID, ShouldEqual, timeZoneAmericaLA)
			So(gEv.Language, ShouldEqual, languageEN)
			So(gEv.Title, ShouldEqual, title)
			So(gEv.Description, ShouldEqual, description)
			So(gEv.ChannelID, ShouldEqual, ownerID)
			So(gEv.GameID, ShouldEqual, overwatchGameID)
			So(gEv.PremiereID, ShouldEqual, premiereID)
		})

		Convey("Should not be able to create a premiere with an empty premiere_id", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			endTime := startTime.Add(time.Hour)
			title := "TestIntegration_Premiere title " + timestamp()
			description := "TestIntegration_Premiere description" + timestamp()

			_, cErr := client.CreatePremiereEvent(ctx, geaclient.CreatePremiereEventParams{
				OwnerID:     ownerID,
				StartTime:   startTime,
				EndTime:     endTime,
				TimeZoneID:  timeZoneAmericaLA,
				Language:    languageEN,
				Title:       title,
				Description: description,
				ChannelID:   ownerID,
				GameID:      overwatchGameID,
			}, ownerID, nil)
			So(geaclient.ErrorCode(cErr), ShouldEqual, http.StatusBadRequest)
			So(cErr, ShouldNotBeNil)
		})

		Convey("Should not be able to create a premiere event with the same start and end time", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			createParams := defaultCreatePremiereEventParams(ownerID, startTime, 0)
			cEv, err := client.CreatePremiereEvent(ctx, createParams, ownerID, nil)
			So(err, ShouldNotBeNil)
			So(cEv, ShouldBeNil)
		})

		Convey("Should not be able to create a premiere event for a different owner", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			params := defaultCreatePremiereEventParams(ownerID, startTime, time.Hour)

			userID := timestampUser()
			So(ownerID, ShouldNotEqual, userID)

			_, cErr := client.CreatePremiereEvent(ctx, params, userID, nil)
			So(cErr, ShouldNotBeNil)
			So(geaclient.ErrorCode(cErr), ShouldEqual, http.StatusForbidden)
		})

		Convey("Should not be able to create a premiere event for a different channel", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			endTime := startTime.Add(time.Hour)
			title := "TestingIntegration_Premiere DifferentChannel title " + timestamp()
			description := "TestingIntegration_Premiere DifferentChannel description " + timestamp()
			premiereID := "premiere_" + timestamp()

			channelID := timestampUser()

			_, cErr := client.CreatePremiereEvent(ctx, geaclient.CreatePremiereEventParams{
				PremiereID:  premiereID,
				OwnerID:     ownerID,
				StartTime:   startTime,
				EndTime:     endTime,
				TimeZoneID:  timeZoneAmericaLA,
				Language:    languageEN,
				Title:       title,
				Description: description,
				ChannelID:   channelID,
				GameID:      overwatchGameID,
			}, ownerID, nil)
			So(cErr, ShouldNotBeNil)
			So(geaclient.ErrorCode(cErr), ShouldEqual, http.StatusForbidden)
		})

		Convey("Should not be able to create an overlapping premiere event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			createParams := defaultCreatePremiereEventParams(ownerID, startTime, time.Hour)
			cEv, err := client.CreatePremiereEvent(ctx, createParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			Convey("Should not be able to overlap the start time", func() {
				params := defaultCreatePremiereEventParams(ownerID, startTime.Add(-30*time.Minute), time.Hour)
				ev, err := client.CreatePremiereEvent(ctx, params, ownerID, nil)
				So(err, ShouldNotBeNil)
				So(geaclient.ErrorCode(err), ShouldEqual, http.StatusBadRequest)
				So(ev, ShouldBeNil)
			})

			Convey("Should not be able to overlap the end time", func() {
				params := defaultCreatePremiereEventParams(ownerID, startTime.Add(30*time.Minute), time.Hour)
				ev, err := client.CreatePremiereEvent(ctx, params, ownerID, nil)
				So(err, ShouldNotBeNil)
				So(geaclient.ErrorCode(err), ShouldEqual, http.StatusBadRequest)
				So(ev, ShouldBeNil)
			})

			Convey("Should not be able to span the event", func() {
				params := defaultCreatePremiereEventParams(ownerID, startTime.Add(-30*time.Minute), 2*time.Hour)
				ev, err := client.CreatePremiereEvent(ctx, params, ownerID, nil)
				So(err, ShouldNotBeNil)
				So(geaclient.ErrorCode(err), ShouldEqual, http.StatusBadRequest)
				So(ev, ShouldBeNil)
			})

			Convey("Should not be able to be within the event", func() {
				params := defaultCreatePremiereEventParams(ownerID, startTime.Add(15*time.Minute), 30*time.Minute)
				ev, err := client.CreatePremiereEvent(ctx, params, ownerID, nil)
				So(err, ShouldNotBeNil)
				So(geaclient.ErrorCode(err), ShouldEqual, http.StatusBadRequest)
				So(ev, ShouldBeNil)
			})

			Convey("Should not be update an event to overlap another premiere", func() {
				params := defaultCreatePremiereEventParams(ownerID, startTime.Add(-90*time.Minute), time.Hour)
				ev, err := client.CreatePremiereEvent(ctx, params, ownerID, nil)
				So(err, ShouldBeNil)
				So(ev, ShouldNotBeNil)

				newEndTime := startTime.Add(2 * time.Hour)
				uEv, err := client.UpdatePremiereEvent(ctx, ev.ID, geaclient.UpdatePremiereEventParams{
					EndTime: &newEndTime,
				}, ownerID, nil)
				So(err, ShouldNotBeNil)
				So(geaclient.ErrorCode(err), ShouldEqual, http.StatusBadRequest)
				So(uEv, ShouldBeNil)
			})
		})

		Convey("Should be able to update a premiere event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			createParams := defaultCreatePremiereEventParams(ownerID, startTime, time.Hour)
			cEv, err := client.CreatePremiereEvent(ctx, createParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(cEv, ShouldNotBeNil)

			Convey("When all fields are changed", func() {
				updatedStartTime := createParams.StartTime.Add(time.Minute)
				updatedEndTime := createParams.StartTime.Add(time.Hour)
				updatedTimeZone := "America/New_York"
				updatedLanguage := languageKR
				updatedTitle := "TestingIntegration_Premiere updated title " + timestamp()
				updatedDesc := "TestingIntegration_Premiere updated desc " + timestamp()
				updatedGameID := irlGameID
				updatedPremiereID := timestamp()

				uEv, uErr := client.UpdatePremiereEvent(ctx, cEv.ID, geaclient.UpdatePremiereEventParams{
					StartTime:   &updatedStartTime,
					EndTime:     &updatedEndTime,
					TimeZoneID:  &updatedTimeZone,
					Language:    &updatedLanguage,
					Title:       &updatedTitle,
					Description: &updatedDesc,
					GameID:      &updatedGameID,
					PremiereID:  &updatedPremiereID,
				}, ownerID, nil)

				So(uErr, ShouldBeNil)
				So(uEv, ShouldNotBeNil)

				So(uEv.StartTime, ShouldEqual, updatedStartTime)
				So(uEv.EndTime, ShouldEqual, updatedEndTime)
				So(uEv.Language, ShouldEqual, updatedLanguage)
				So(uEv.Title, ShouldEqual, updatedTitle)
				So(uEv.Description, ShouldEqual, updatedDesc)
				So(uEv.GameID, ShouldEqual, updatedGameID)
				So(uEv.PremiereID, ShouldEqual, updatedPremiereID)
				So(uEv.TimeZoneID, ShouldEqual, updatedTimeZone)

				So(uEv.OwnerID, ShouldEqual, cEv.OwnerID)
				So(uEv.ChannelID, ShouldEqual, cEv.ChannelID)
			})

			// There was a bug where updating premiere events would fail if a premiere ID is not  given.
			// This tests verifies that it was fixed.
			Convey("When no premiere ID is provided", func() {
				updatedTitle := "TestingIntegration_Single updated title " + timestamp()
				updateParams := geaclient.UpdatePremiereEventParams{
					Title: &updatedTitle,
				}
				uEv, uErr := client.UpdatePremiereEvent(ctx, cEv.ID, updateParams, ownerID, nil)
				So(uErr, ShouldBeNil)
				So(uEv, ShouldNotBeNil)

				So(uEv.Title, ShouldEqual, updatedTitle)
			})
		})
	})
}

func timetableParams(ownerID string) geaclient.CreateTimetableEventParams {
	title := "TestIntegration_TimetableDefault title " + timestamp()
	description := "TestIntegration_TimetableDefault description" + timestamp()

	return geaclient.CreateTimetableEventParams{
		OwnerID:     ownerID,
		TimeZoneID:  timeZoneAmericaLA,
		Language:    languageEN,
		Title:       title,
		Description: description,
	}
}

func segmentParams(parentID string, channelID string, gameID string, startTime time.Time, endTime time.Time) geaclient.CreateSegmentEventParams {
	title := "TestIntegration_SegmentDefault title " + timestamp()
	description := "TestIntegration_SegmentDefault description" + timestamp()

	return geaclient.CreateSegmentEventParams{
		OwnerID:     channelID,
		ParentID:    parentID,
		StartTime:   startTime,
		EndTime:     endTime,
		Title:       title,
		Description: description,
		ChannelID:   channelID,
		GameID:      gameID,
	}
}

func TestIntegration_Timetable(t *testing.T) {
	t.Parallel()
	ts := startServer(t, newDefaultInjectables())
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		Convey("Should be able to create a timetable event", func() {
			ownerID := timestampUser()
			title := "TestIntegration_Timetable title " + timestamp()
			description := "TestIntegration_Timetable description" + timestamp()

			cEv, cErr := client.CreateTimetableEvent(ctx, geaclient.CreateTimetableEventParams{
				OwnerID:     ownerID,
				TimeZoneID:  timeZoneAmericaLA,
				Language:    languageEN,
				Title:       title,
				Description: description,
			}, ownerID, nil)
			So(cErr, ShouldBeNil)
			So(cEv, ShouldNotBeNil)
			So(cEv.ID, ShouldNotBeEmpty)
			So(cEv.OwnerID, ShouldEqual, ownerID)
			So(cEv.Type, ShouldEqual, "timetable")
			So(cEv.StartTime, ShouldBeNil)
			So(cEv.EndTime, ShouldBeNil)
			So(cEv.TimeZoneID, ShouldEqual, timeZoneAmericaLA)
			So(cEv.Language, ShouldEqual, languageEN)
			So(cEv.Title, ShouldEqual, title)
			So(cEv.Description, ShouldEqual, description)
			So(cEv.ChannelIDs, ShouldBeEmpty)
			So(cEv.GameIDs, ShouldBeEmpty)

			gE, gErr := client.GetEvent(ctx, cEv.ID, nil, nil)
			So(gErr, ShouldBeNil)
			So(gE, ShouldHaveSameTypeAs, &geaclient.TimetableEvent{})
			gEv := gE.(*geaclient.TimetableEvent)
			So(gEv.ID, ShouldEqual, cEv.ID)
			So(gEv.OwnerID, ShouldEqual, ownerID)
			So(gEv.Type, ShouldEqual, "timetable")
			So(gEv.StartTime, ShouldBeNil)
			So(gEv.EndTime, ShouldBeNil)
			So(gEv.TimeZoneID, ShouldEqual, timeZoneAmericaLA)
			So(gEv.Language, ShouldEqual, languageEN)
			So(gEv.Title, ShouldEqual, title)
			So(gEv.Description, ShouldEqual, description)
			So(gEv.ChannelIDs, ShouldNotBeNil)
			So(gEv.ChannelIDs, ShouldBeEmpty)
			So(gEv.GameIDs, ShouldNotBeNil)
			So(gEv.GameIDs, ShouldBeEmpty)

			Convey("and can add a segment", func() {
				title := "TestIntegration_Segment title " + timestamp()
				description := "TestIntegration_Segment description" + timestamp()
				startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
				endTime := startTime.Add(time.Hour)

				cSeg, cErr := client.CreateSegmentEvent(ctx, geaclient.CreateSegmentEventParams{
					OwnerID:     ownerID,
					ParentID:    cEv.ID,
					StartTime:   startTime,
					EndTime:     endTime,
					Title:       title,
					Description: description,
					ChannelID:   ownerID,
					GameID:      overwatchGameID,
				}, ownerID, nil)
				So(cErr, ShouldBeNil)
				So(cSeg, ShouldNotBeNil)
				So(cSeg.ID, ShouldNotBeEmpty)
				So(cSeg.OwnerID, ShouldEqual, ownerID)
				So(cSeg.Type, ShouldEqual, "segment")
				So(cSeg.ParentID, ShouldEqual, cEv.ID)
				So(cSeg.StartTime, ShouldEqual, startTime)
				So(cSeg.EndTime, ShouldEqual, endTime)
				So(cSeg.TimeZoneID, ShouldEqual, cEv.TimeZoneID)
				So(cSeg.Language, ShouldEqual, cEv.Language)
				So(cSeg.Title, ShouldEqual, title)
				So(cSeg.Description, ShouldEqual, description)
				So(cSeg.ChannelID, ShouldEqual, ownerID)
				So(cSeg.GameID, ShouldEqual, overwatchGameID)

				gE, gErr := client.GetEvent(ctx, cSeg.ID, nil, nil)
				So(gErr, ShouldBeNil)
				So(gE, ShouldHaveSameTypeAs, &geaclient.SegmentEvent{})
				gSeg := gE.(*geaclient.SegmentEvent)
				So(gSeg.ID, ShouldEqual, cSeg.ID)
				So(gSeg.OwnerID, ShouldEqual, ownerID)
				So(gSeg.Type, ShouldEqual, "segment")
				So(gSeg.ParentID, ShouldEqual, cEv.ID)
				So(gSeg.StartTime, ShouldEqual, startTime)
				So(gSeg.EndTime, ShouldEqual, endTime)
				So(gSeg.TimeZoneID, ShouldEqual, cEv.TimeZoneID)
				So(gSeg.Language, ShouldEqual, cEv.Language)
				So(gSeg.Title, ShouldEqual, title)
				So(gSeg.Description, ShouldEqual, description)
				So(gSeg.ChannelID, ShouldEqual, ownerID)
				So(gSeg.GameID, ShouldEqual, overwatchGameID)
			})

			Convey("Should not be able to create a segment event with the same start and end time", func() {
				title := "TestIntegration_Segment title " + timestamp()
				description := "TestIntegration_Segment description" + timestamp()
				startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

				cSeg, cErr := client.CreateSegmentEvent(ctx, geaclient.CreateSegmentEventParams{
					OwnerID:     ownerID,
					ParentID:    cEv.ID,
					StartTime:   startTime,
					EndTime:     startTime,
					Title:       title,
					Description: description,
					ChannelID:   ownerID,
					GameID:      overwatchGameID,
				}, ownerID, nil)
				So(cErr, ShouldNotBeNil)
				So(cSeg, ShouldBeNil)
			})

			Convey("Should not be able to create a segment event with a different owner", func() {
				title := "TestIntegration_Segment title " + timestamp()
				description := "TestIntegration_Segment description" + timestamp()
				startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

				segmentOwnerID := timestampUser()
				cSeg, cErr := client.CreateSegmentEvent(ctx, geaclient.CreateSegmentEventParams{
					OwnerID:     segmentOwnerID,
					ParentID:    cEv.ID,
					StartTime:   startTime,
					EndTime:     startTime,
					Title:       title,
					Description: description,
					ChannelID:   segmentOwnerID,
					GameID:      overwatchGameID,
				}, segmentOwnerID, nil)
				So(cErr, ShouldNotBeNil)
				So(cSeg, ShouldBeNil)
			})
		})

		Convey("Should be able to add a segment to a timetable event and have that update fields on the timetable event", func() {
			ownerID := timestampUser()
			cTt, err := client.CreateTimetableEvent(ctx, timetableParams(ownerID), ownerID, nil)
			So(err, ShouldBeNil)
			So(cTt, ShouldNotBeNil)

			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			endTime := startTime.Add(time.Hour)
			cSeg, err := client.CreateSegmentEvent(ctx, segmentParams(cTt.ID, ownerID, overwatchGameID, startTime, endTime), ownerID, nil)
			So(err, ShouldBeNil)
			So(cSeg, ShouldNotBeNil)

			gE, err := client.GetEvent(ctx, cTt.ID, nil, nil)
			So(err, ShouldBeNil)
			So(gE, ShouldHaveSameTypeAs, &geaclient.TimetableEvent{})
			gTt := gE.(*geaclient.TimetableEvent)
			So(gTt.StartTime, ShouldNotBeNil)
			So(*gTt.StartTime, ShouldEqual, startTime)
			So(gTt.EndTime, ShouldNotBeNil)
			So(*gTt.EndTime, ShouldEqual, endTime)
			So(gTt.ChannelIDs, ShouldHaveLength, 1)
			So(gTt.ChannelIDs[0], ShouldEqual, ownerID)
			So(gTt.GameIDs, ShouldHaveLength, 1)
			So(gTt.GameIDs[0], ShouldEqual, overwatchGameID)

			Convey("Deleting the event should delete the segment", func() {
				dTt, err := client.DeleteEvent(ctx, cTt.ID, ownerID, nil)
				So(err, ShouldBeNil)
				So(dTt, ShouldNotBeNil)

				gSeg, err := client.GetEvent(ctx, cSeg.GetID(), nil, nil)
				So(err, ShouldNotBeNil)
				So(geaclient.ErrorCode(err), ShouldEqual, http.StatusNotFound)
				So(gSeg, ShouldBeNil)
			})
		})

		Convey("Should be able to delete a timetable event with no segments", func() {
			ownerID := timestampUser()
			cTt, err := client.CreateTimetableEvent(ctx, timetableParams(ownerID), ownerID, nil)
			So(err, ShouldBeNil)
			So(cTt, ShouldNotBeNil)

			dTt, err := client.DeleteEvent(ctx, cTt.ID, ownerID, nil)
			So(err, ShouldBeNil)
			So(dTt, ShouldNotBeNil)

			gTt, err := client.GetEvent(ctx, cTt.GetID(), nil, nil)
			So(err, ShouldNotBeNil)
			So(geaclient.ErrorCode(err), ShouldEqual, http.StatusNotFound)
			So(gTt, ShouldBeNil)
		})

		Convey("Deleting a timetable event should delete its segments", func() {
			ownerID := timestampUser()
			cTt, err := client.CreateTimetableEvent(ctx, timetableParams(ownerID), ownerID, nil)
			So(err, ShouldBeNil)
			So(cTt, ShouldNotBeNil)

			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
			endTime := startTime.Add(time.Hour)
			cSeg, err := client.CreateSegmentEvent(ctx, segmentParams(cTt.ID, ownerID, overwatchGameID, startTime, endTime), ownerID, nil)
			So(err, ShouldBeNil)
			So(cSeg, ShouldNotBeNil)

			dTt, err := client.DeleteEvent(ctx, cTt.ID, ownerID, nil)
			So(err, ShouldBeNil)
			So(dTt, ShouldNotBeNil)

			gSeg, err := client.GetEvent(ctx, cSeg.GetID(), nil, nil)
			So(err, ShouldNotBeNil)
			So(geaclient.ErrorCode(err), ShouldEqual, http.StatusNotFound)
			So(gSeg, ShouldBeNil)
		})

		Convey("Should be able to get all timetables that a user owns", func() {
			ownerID := timestampUser()
			timetable1, err := client.CreateTimetableEvent(ctx, timetableParams(ownerID), ownerID, nil)
			So(err, ShouldBeNil)
			So(timetable1, ShouldNotBeNil)

			timetable2, err := client.CreateTimetableEvent(ctx, timetableParams(ownerID), ownerID, nil)
			So(err, ShouldBeNil)
			So(timetable2, ShouldNotBeNil)

			eventIDs, err := client.GetManagedCollectionIDsByOwner(ctx, ownerID, ownerID, nil, nil)
			So(err, ShouldBeNil)
			So(eventIDs.EventIDs, ShouldHaveLength, 2)
			// Should return in ascending order by title
			So(eventIDs.EventIDs[0], ShouldEqual, timetable1.ID)
			So(eventIDs.EventIDs[1], ShouldEqual, timetable2.ID)

			Convey("Should be controllable by a cursor", func() {
				eventIDs, err := client.GetManagedCollectionIDsByOwner(ctx, ownerID, ownerID, &geaclient.GetCollectionIDsOpts{
					Limit: 1,
				}, nil)
				So(err, ShouldBeNil)
				So(eventIDs.EventIDs, ShouldHaveLength, 1)
				So(eventIDs.EventIDs[0], ShouldEqual, timetable1.ID)
				So(eventIDs.Cursor, ShouldNotEqual, "")

				cursor := eventIDs.Cursor
				eventIDs, err = client.GetManagedCollectionIDsByOwner(ctx, ownerID, ownerID, &geaclient.GetCollectionIDsOpts{
					Limit:  1,
					Cursor: cursor,
				}, nil)
				So(err, ShouldBeNil)
				So(eventIDs.EventIDs, ShouldHaveLength, 1)
				So(eventIDs.EventIDs[0], ShouldEqual, timetable2.ID)
				So(eventIDs.Cursor, ShouldNotEqual, cursor)

				cursor = eventIDs.Cursor
				eventIDs, err = client.GetManagedCollectionIDsByOwner(ctx, ownerID, ownerID, &geaclient.GetCollectionIDsOpts{
					Limit:  1,
					Cursor: cursor,
				}, nil)
				So(err, ShouldBeNil)
				So(eventIDs.EventIDs, ShouldHaveLength, 0)
				So(eventIDs.Cursor, ShouldEqual, "")
			})

			Convey("Should not be able to get timetables for a channel that you're not allowed to manage", func() {
				ownerID := timestampUser()
				timetable1, err := client.CreateTimetableEvent(ctx, timetableParams(ownerID), ownerID, nil)
				So(err, ShouldBeNil)
				So(timetable1, ShouldNotBeNil)

				// This user is not allowed to manage ownerID's events because it's not related to the event
				// in any way - it's not the owner, not an editor for the owner, not on the consul whitelist, etc.
				randomID := timestampUser()

				eventIDs, err := client.GetManagedCollectionIDsByOwner(ctx, randomID, ownerID, nil, nil)

				So(eventIDs, ShouldBeNil)
				So(err, ShouldNotBeNil)
				So(geaclient.ErrorCode(err), ShouldEqual, http.StatusForbidden)
			})

			Convey("Should not return deleted timetables", func() {
				_, err := client.DeleteEvent(ctx, timetable2.ID, ownerID, nil)
				So(err, ShouldBeNil)

				eventIDs, err := client.GetManagedCollectionIDsByOwner(ctx, ownerID, ownerID, nil, nil)
				So(err, ShouldBeNil)
				So(eventIDs.EventIDs, ShouldHaveLength, 1)
				So(eventIDs.EventIDs[0], ShouldEqual, timetable1.ID)
			})
		})
	})
}

func TestIntegration_EventLocalization(t *testing.T) {
	t.Parallel()

	injectables := newDefaultInjectables()
	injectables.adminList = &testutils.AllAuthenticatedAdminList{}

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		Convey("Should be able to add localization to an event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			krChannelID := timestampUser()
			addLoc, err := client.AddLocalization(ctx, geaclient.AddLocalizationParams{
				EventID:   ev.ID,
				Language:  languageKR,
				ChannelID: &krChannelID,
			}, ownerID, nil)
			So(err, ShouldBeNil)
			So(addLoc, ShouldNotBeNil)

			addEvIDs, err := client.GetEventIDsByChannelIDs(ctx, []string{krChannelID}, &geaclient.GetEIDsByCIDsOpts{StartTimeAfter: &startTime}, nil)
			So(err, ShouldBeNil)
			So(addEvIDs, ShouldNotBeNil)
			So(addEvIDs.EventIDs, ShouldHaveLength, 1)
			So(addEvIDs.EventIDs, ShouldContain, ev.ID)

			rmLoc, err := client.RemoveLocalization(ctx, ev.ID, languageKR, ownerID, nil)
			So(err, ShouldBeNil)
			So(rmLoc, ShouldNotBeNil)

			rmEvIDs, err := client.GetEventIDsByChannelIDs(ctx, []string{krChannelID}, &geaclient.GetEIDsByCIDsOpts{StartTimeAfter: &startTime}, nil)
			So(err, ShouldBeNil)
			So(rmEvIDs, ShouldNotBeNil)
			So(rmEvIDs.EventIDs, ShouldBeEmpty)
		})

		Convey("Should be able to update localization on an event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			krTitle := "KR Localization"
			loc, err := client.AddLocalization(ctx, geaclient.AddLocalizationParams{
				EventID:  ev.ID,
				Language: languageKR,
				Title:    &krTitle,
			}, ownerID, nil)
			So(err, ShouldBeNil)
			So(loc, ShouldNotBeNil)

			// Add a Channel ID to the localization
			krChannelID1 := timestampUser()
			_, err = client.UpdateLocalization(ctx, geaclient.UpdateLocalizationParams{
				EventID:   ev.ID,
				Language:  languageKR,
				Title:     &krTitle,
				ChannelID: &krChannelID1,
			}, ownerID, nil)
			So(err, ShouldBeNil)

			evIDs1, err := client.GetEventIDsByChannelIDs(ctx, []string{krChannelID1}, &geaclient.GetEIDsByCIDsOpts{StartTimeAfter: &startTime}, nil)
			So(err, ShouldBeNil)
			So(evIDs1, ShouldNotBeNil)
			So(evIDs1.EventIDs, ShouldHaveLength, 1)
			So(evIDs1.EventIDs, ShouldContain, ev.ID)

			// Modify the Channel ID on the localization
			krChannelID2 := timestampUser()
			_, err = client.UpdateLocalization(ctx, geaclient.UpdateLocalizationParams{
				EventID:   ev.ID,
				Language:  languageKR,
				Title:     &krTitle,
				ChannelID: &krChannelID2,
			}, ownerID, nil)
			So(err, ShouldBeNil)

			evIDs2, err := client.GetEventIDsByChannelIDs(ctx, []string{krChannelID1}, &geaclient.GetEIDsByCIDsOpts{StartTimeAfter: &startTime}, nil)
			So(err, ShouldBeNil)
			So(evIDs2, ShouldNotBeNil)
			So(evIDs2.EventIDs, ShouldBeEmpty)

			evIDs3, err := client.GetEventIDsByChannelIDs(ctx, []string{krChannelID2}, &geaclient.GetEIDsByCIDsOpts{StartTimeAfter: &startTime}, nil)
			So(err, ShouldBeNil)
			So(evIDs3, ShouldNotBeNil)
			So(evIDs3.EventIDs, ShouldHaveLength, 1)
			So(evIDs3.EventIDs, ShouldContain, ev.ID)

			// Remove the Channel ID on the localization
			_, err = client.UpdateLocalization(ctx, geaclient.UpdateLocalizationParams{
				EventID:  ev.ID,
				Language: languageKR,
				Title:    &krTitle,
			}, ownerID, nil)
			So(err, ShouldBeNil)

			evIDs4, err := client.GetEventIDsByChannelIDs(ctx, []string{krChannelID2}, &geaclient.GetEIDsByCIDsOpts{StartTimeAfter: &startTime}, nil)
			So(err, ShouldBeNil)
			So(evIDs4, ShouldNotBeNil)
			So(evIDs4.EventIDs, ShouldBeEmpty)
		})

		Convey("Should not be able to add a localization of the same language to an event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			_, err = client.AddLocalization(ctx, geaclient.AddLocalizationParams{
				EventID:  ev.ID,
				Language: languageEN,
				Title:    refStr("EN Localization"),
			}, ownerID, nil)
			So(err, ShouldNotBeNil)
		})

		Convey("Should not be able to add a localization with the same channel to an event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			_, err = client.AddLocalization(ctx, geaclient.AddLocalizationParams{
				EventID:   ev.ID,
				Language:  languageKR,
				ChannelID: &ownerID,
			}, ownerID, nil)
			So(err, ShouldNotBeNil)
		})

		Convey("Should not be able to add a duplicate localization to an event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			_, err = client.AddLocalization(ctx, geaclient.AddLocalizationParams{
				EventID:  ev.ID,
				Language: languageKR,
				Title:    refStr("KR Localization"),
			}, ownerID, nil)
			So(err, ShouldBeNil)

			_, err = client.AddLocalization(ctx, geaclient.AddLocalizationParams{
				EventID:  ev.ID,
				Language: languageKR,
				Title:    refStr("KR Localization 2"),
			}, ownerID, nil)
			So(err, ShouldNotBeNil)
		})
	})
}

func TestIntegration_EventFollows(t *testing.T) {
	t.Parallel()
	ts := startServer(t, newDefaultInjectables())
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := context.Background()

		Convey("Should be able to follow and unfollow an event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			userID := timestampUser()
			err = client.FollowEvent(ctx, ev.ID, userID, nil)
			So(err, ShouldBeNil)

			followers, err := client.GetEventFollowers(ctx, ev.ID, nil, nil)
			So(err, ShouldBeNil)
			So(followers.UserIDs, ShouldHaveLength, 1)
			So(followers.UserIDs[0], ShouldEqual, userID)

			err = client.UnfollowEvent(ctx, ev.ID, userID, nil)
			So(err, ShouldBeNil)

			f2, err := client.GetEventFollowers(ctx, ev.ID, nil, nil)
			So(err, ShouldBeNil)
			So(f2.UserIDs, ShouldBeEmpty)
		})

		Convey("Follow and Unfollow should change the follow count", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			u1 := timestampUser()
			err = client.FollowEvent(ctx, ev.ID, u1, nil)
			So(err, ShouldBeNil)

			u2 := timestampUser()
			err = client.FollowEvent(ctx, ev.ID, u2, nil)
			So(err, ShouldBeNil)

			s1, err := client.GetEventStats(ctx, []string{ev.ID}, nil)
			So(err, ShouldBeNil)
			So(s1, ShouldNotBeNil)
			So(s1.Items, ShouldHaveLength, 1)
			So(s1.Items[0].EventID, ShouldEqual, ev.ID)
			So(s1.Items[0].FollowCount, ShouldEqual, 2)

			err = client.UnfollowEvent(ctx, ev.ID, u1, nil)
			So(err, ShouldBeNil)

			s2, err := client.GetEventStats(ctx, []string{ev.ID}, nil)
			So(err, ShouldBeNil)
			So(s2, ShouldNotBeNil)
			So(s2.Items, ShouldHaveLength, 1)
			So(s2.Items[0].EventID, ShouldEqual, ev.ID)
			So(s2.Items[0].FollowCount, ShouldEqual, 1)
		})

		Convey("Follow should increment the follow count only once", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			userID := timestampUser()
			err = client.FollowEvent(ctx, ev.ID, userID, nil)
			So(err, ShouldBeNil)

			s1, err := client.GetEventStats(ctx, []string{ev.ID}, nil)
			So(err, ShouldBeNil)
			So(s1, ShouldNotBeNil)
			So(s1.Items, ShouldHaveLength, 1)
			So(s1.Items[0].EventID, ShouldEqual, ev.ID)
			So(s1.Items[0].FollowCount, ShouldEqual, 1)

			err = client.FollowEvent(ctx, ev.ID, userID, nil)
			So(err, ShouldBeNil)

			s2, err := client.GetEventStats(ctx, []string{ev.ID}, nil)
			So(err, ShouldBeNil)
			So(s2, ShouldNotBeNil)
			So(s2.Items, ShouldHaveLength, 1)
			So(s2.Items[0].EventID, ShouldEqual, ev.ID)
			So(s2.Items[0].FollowCount, ShouldEqual, 1)
		})

		Convey("Should be able to get stats for new event", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			s1, err := client.GetEventStats(ctx, []string{ev.ID}, nil)
			So(err, ShouldBeNil)
			So(s1, ShouldNotBeNil)
			So(s1.Items, ShouldHaveLength, 1)
			So(s1.Items[0].EventID, ShouldEqual, ev.ID)
			So(s1.Items[0].FollowCount, ShouldEqual, 0)
		})

		Convey("Should be able to page through followers", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			u1 := timestampUser()
			err = client.FollowEvent(ctx, ev.ID, u1, nil)
			So(err, ShouldBeNil)

			u2 := timestampUser()
			So(u1, ShouldNotEqual, u2)
			err = client.FollowEvent(ctx, ev.ID, u2, nil)
			So(err, ShouldBeNil)

			f1, err := client.GetEventFollowers(ctx, ev.ID, &geaclient.GetEventFollowersOptions{Limit: refInt(1)}, nil)
			So(err, ShouldBeNil)
			So(f1, ShouldNotBeNil)
			So(f1.UserIDs, ShouldHaveLength, 1)
			// This should be guaranteed as the users should be returned in sorted range-key order
			So(f1.UserIDs[0], ShouldEqual, u1)
			So(f1.Cursor, ShouldNotBeBlank)

			f2, err := client.GetEventFollowers(ctx, ev.ID, &geaclient.GetEventFollowersOptions{Limit: refInt(2), Cursor: &f1.Cursor}, nil)
			So(err, ShouldBeNil)
			So(f2.UserIDs, ShouldHaveLength, 1)
			So(f2.UserIDs[0], ShouldEqual, u2)
			So(f2.Cursor, ShouldBeBlank)
		})

		Convey("Should be able to page through followed events", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev1, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev1, ShouldNotBeNil)

			ev2, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev2, ShouldNotBeNil)
			So(ev1.ID, ShouldNotEqual, ev2.ID)

			userID := timestampUser()
			err = client.FollowEvent(ctx, ev1.ID, userID, nil)
			So(err, ShouldBeNil)

			err = client.FollowEvent(ctx, ev2.ID, userID, nil)
			So(err, ShouldBeNil)

			f1, err := client.GetFollowedEvents(ctx, userID, &geaclient.GetFollowedEventsOptions{Limit: refInt(1)}, nil)
			So(err, ShouldBeNil)
			So(f1.EventIDs, ShouldHaveLength, 1)
			So(f1.Cursor, ShouldNotBeBlank)

			f2, err := client.GetFollowedEvents(ctx, userID, &geaclient.GetFollowedEventsOptions{Limit: refInt(2), Cursor: &f1.Cursor}, nil)
			So(err, ShouldBeNil)
			So(f2.EventIDs, ShouldHaveLength, 1)
			So(f2.Cursor, ShouldBeBlank)

			So(f1.EventIDs[0], ShouldNotEqual, f2.EventIDs[0])
			eventIDs := []string{f1.EventIDs[0], f2.EventIDs[0]}
			So(eventIDs, ShouldContain, ev1.ID)
			So(eventIDs, ShouldContain, ev2.ID)
		})

		Convey("Should be able to check followed events", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev1, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev1, ShouldNotBeNil)

			ev2, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev2, ShouldNotBeNil)
			So(ev2.ID, ShouldNotEqual, ev1.ID)

			ev3, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev3, ShouldNotBeNil)
			So(ev3.ID, ShouldNotEqual, ev1.ID)
			So(ev3.ID, ShouldNotEqual, ev2.ID)

			userID := timestampUser()
			err = client.FollowEvent(ctx, ev1.ID, userID, nil)
			So(err, ShouldBeNil)

			err = client.FollowEvent(ctx, ev2.ID, userID, nil)
			So(err, ShouldBeNil)

			events, err := client.CheckFollowedEvents(ctx, []string{ev1.ID, ev2.ID, ev3.ID}, userID, nil)
			So(err, ShouldBeNil)
			So(events.EventIDs, ShouldHaveLength, 2)
			So(events.Cursor, ShouldBeBlank)

			So(events.EventIDs, ShouldContain, ev1.ID)
			So(events.EventIDs, ShouldContain, ev2.ID)
			So(events.EventIDs, ShouldNotContain, ev3.ID)
		})
	})
}

func TestIntegration_EventVideos(t *testing.T) {
	t.Parallel()
	ts := startServer(t, newDefaultInjectables())
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	expectedVodID := "206245871"
	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := context.Background()

		Convey("Should get empty list if no archives found", func() {
			ownerID := jorasteUserID
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			v, err := client.GetEventVideos(ctx, ev.ID, nil, nil)
			So(err, ShouldBeNil)
			So(v, ShouldNotBeNil)
			So(v.Items, ShouldHaveLength, 0)
			So(v.HasNextPage, ShouldBeFalse)
		})

		Convey("Should get archived video for single event", func() {
			ownerID := jorasteUserID
			startTime := time.Date(2017, 12, 03, 00, 59, 00, 00, time.UTC)

			ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			v, err := client.GetEventVideos(ctx, ev.ID, nil, nil)
			So(err, ShouldBeNil)
			So(v, ShouldNotBeNil)
			So(v.Items, ShouldHaveLength, 1)
			vod := v.Items[0]
			So(vod.VodID, ShouldEqual, expectedVodID)
			So(vod.OffsetSeconds, ShouldEqual, 27)
			So(v.HasNextPage, ShouldBeFalse)

			Convey("Validate that the result is cached", func() {
				// For now we don't invalidate the cache, so this test verifies that we get the same result back even
				// after we change the event.  If we implement cache invalidation for videos, we should change this test
				newStartTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)
				newEndTime := newStartTime.Add(time.Hour)
				uEv, err := client.UpdateSingleEvent(ctx, ev.ID, geaclient.UpdateSingleEventParams{
					StartTime: &newStartTime,
					EndTime:   &newEndTime,
				}, ownerID, nil)
				So(err, ShouldBeNil)
				So(uEv, ShouldNotBeNil)

				v2, err := client.GetEventVideos(ctx, ev.ID, nil, nil)
				So(err, ShouldBeNil)
				So(v2, ShouldNotBeNil)
				So(v2.Items, ShouldHaveLength, 1)
				vod := v.Items[0]
				So(vod.VodID, ShouldEqual, expectedVodID)
				So(vod.OffsetSeconds, ShouldEqual, 27)
				So(v.HasNextPage, ShouldBeFalse)
			})
		})
	})
}

func newTestVod(startTime time.Time, duration int, broadcastType string, views int) *vodapi.Vod {
	return &vodapi.Vod{
		Id:            timestamp(),
		StartedOn:     vodapi_utils.ProtobufTimeAsTimestamp(&startTime),
		BroadcastType: vodapi.ConvertInternalVodType(broadcastType),
		TotalLength:   int64(duration),
		Views:         int64(views),
	}
}

func TestIntegration_EventVideosFilteringAndPagination(t *testing.T) {
	t.Parallel()

	vodAPIClient := &testutils.StubVodAPIClient{}
	injectables := newDefaultInjectables()
	injectables.vodAPIClient = vodAPIClient

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := context.Background()

		ownerID := timestampUser()
		startTime := time.Now().UTC()
		duration := time.Hour * 3

		ev, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, duration), ownerID, nil)
		So(err, ShouldBeNil)
		So(ev, ShouldNotBeNil)
		eventID := ev.ID

		Convey("Should allow paging through the videos sorted by views", func() {
			// Given three videos with varying number of views.
			vodDuration := 60 * 10 // 10 min
			minViewsVod := newTestVod(startTime, vodDuration, video.BroadcastTypeHighlight, 5)
			medViewsVod := newTestVod(startTime, vodDuration, video.BroadcastTypeHighlight, 10)
			maxViewsVod := newTestVod(startTime, vodDuration, video.BroadcastTypeHighlight, 15)
			vodAPIClient.SetVods([]*vodapi.Vod{medViewsVod, minViewsVod, maxViewsVod})

			// Get the first page of videos, in desc order by views, where the page should have at most 1 video.
			resp, err := client.GetEventVideos(ctx, eventID, &geaclient.GetEventArchiveVideosOpts{
				Limit:  refInt(1),
				SortBy: refStr(geaclient.VideoSortByViews),
			}, nil)
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(resp.Items, ShouldHaveLength, 1)

			// Verify that the video with the most views was returned.
			So(resp.Items[0].VodID, ShouldEqual, maxViewsVod.Id)

			// Verify that we should be able to page forwards.
			cursor := resp.Items[0].Cursor
			So(cursor, ShouldNotBeBlank)
			So(resp.HasNextPage, ShouldBeTrue)

			// Get the second page of videos, in desc order by views, where the page should have at most 1 video.
			resp, err = client.GetEventVideos(ctx, eventID, &geaclient.GetEventArchiveVideosOpts{
				Limit:       refInt(1),
				AfterCursor: &cursor,
				SortBy:      refStr(geaclient.VideoSortByViews),
			}, nil)
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(resp.Items, ShouldHaveLength, 1)

			// Verify that the video with the second most views was returned.
			So(resp.Items[0].VodID, ShouldEqual, medViewsVod.Id)

			// Verify that we should be able to page forwards.
			cursor = resp.Items[0].Cursor
			So(cursor, ShouldNotBeBlank)
			So(resp.HasNextPage, ShouldBeTrue)

			// Get the last page of videos.
			resp, err = client.GetEventVideos(ctx, eventID, &geaclient.GetEventArchiveVideosOpts{
				Limit:       refInt(1),
				AfterCursor: &cursor,
				SortBy:      refStr(geaclient.VideoSortByViews),
			}, nil)
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(resp.Items, ShouldHaveLength, 1)

			// Verify that the video with the least number of views was returned.
			So(resp.Items[0].VodID, ShouldEqual, minViewsVod.Id)

			// Verify that we've reached the end of the list.
			cursor = resp.Items[0].Cursor
			So(cursor, ShouldNotBeBlank)
			So(resp.HasNextPage, ShouldBeFalse)
		})

		Convey("Should allow paging through the videos sorted by start time", func() {
			// Given three videos with varying start times.
			vodDuration := 60 * 10 // 10 min
			earlyVod := newTestVod(startTime, vodDuration, video.BroadcastTypeHighlight, 5)
			middleVod := newTestVod(startTime.Add(time.Minute), vodDuration, video.BroadcastTypeHighlight, 10)
			lateVod := newTestVod(startTime.Add(2*time.Minute), vodDuration, video.BroadcastTypeHighlight, 15)
			vodAPIClient.SetVods([]*vodapi.Vod{middleVod, lateVod, earlyVod})

			resp, err := client.GetEventVideos(ctx, eventID, &geaclient.GetEventArchiveVideosOpts{
				Limit:  refInt(2),
				SortBy: refStr(geaclient.VideoSortByStartTime),
			}, nil)
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(resp.Items, ShouldHaveLength, 2)

			So(resp.Items[0].VodID, ShouldEqual, earlyVod.Id)
			So(resp.Items[1].VodID, ShouldEqual, middleVod.Id)

			// Verify that we should be able to page forwards.
			So(resp.Items[0].Cursor, ShouldNotBeBlank)
			So(resp.HasNextPage, ShouldBeTrue)

			resp, err = client.GetEventVideos(ctx, eventID, &geaclient.GetEventArchiveVideosOpts{
				Limit:       refInt(2),
				AfterCursor: &resp.Items[1].Cursor,
				SortBy:      refStr(geaclient.VideoSortByStartTime),
			}, nil)
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(resp.Items, ShouldHaveLength, 1)

			So(resp.Items[0].VodID, ShouldEqual, lateVod.Id)

			// Verify that we've reached the end of the list.
			So(resp.Items[0].Cursor, ShouldNotBeBlank)
			So(resp.HasNextPage, ShouldBeFalse)
		})

		Convey("Should allow filtering by broadcast type", func() {
			vodDuration := 60 * 10 // 10 min

			highlight := newTestVod(startTime, vodDuration, video.BroadcastTypeHighlight, 0)
			archive := newTestVod(startTime, vodDuration, video.BroadcastTypeArchive, 0)
			vodAPIClient.SetVods([]*vodapi.Vod{highlight, archive})

			Convey("Should return both archives and highlights if no broadcast type is given", func() {
				resp, err := client.GetEventVideos(ctx, eventID, nil, nil)
				So(err, ShouldBeNil)
				So(resp, ShouldNotBeNil)
				So(resp.Items, ShouldHaveLength, 2)
			})

			Convey("Should return highlights, when filtering by highlight", func() {
				resp, err := client.GetEventVideos(ctx, eventID, &geaclient.GetEventArchiveVideosOpts{
					VideoType: refStr(geaclient.VideoTypeHighlight),
				}, nil)
				So(err, ShouldBeNil)
				So(resp, ShouldNotBeNil)
				So(resp.Items, ShouldHaveLength, 1)
				So(resp.HasNextPage, ShouldBeFalse)

				So(resp.Items[0].VodID, ShouldEqual, highlight.Id)
			})

			Convey("Should return archives, when filtering by archive", func() {
				resp, err := client.GetEventVideos(ctx, eventID, &geaclient.GetEventArchiveVideosOpts{
					VideoType: refStr(geaclient.VideoTypeArchive),
				}, nil)
				So(err, ShouldBeNil)
				So(resp, ShouldNotBeNil)
				So(resp.Items, ShouldHaveLength, 1)
				So(resp.HasNextPage, ShouldBeFalse)

				So(resp.Items[0].VodID, ShouldEqual, archive.Id)
			})
		})

	})
}

func TestIntegration_GetLiveEvent(t *testing.T) {
	t.Parallel()

	controlledNow := time.Date(1994, time.April, 22, 17, 43, 0, 0, time.UTC)

	injectables := newDefaultInjectables()
	injectables.getChannelClient = testutils.NewGetChannelClientStub(numericalOverwatchGameID)
	injectables.clock = &testutils.StubClock{ControlledNow: controlledNow}

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := context.Background()
		Convey("Should be able to fetch the live event on a channel", func() {
			ownerID := timestampUser()
			monthDuration := time.Hour * 24 * 30

			startTimePast := time.Date(1994, time.January, 29, 01, 02, 03, 00, time.UTC)
			pastEvent, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTimePast, monthDuration), ownerID, nil)
			So(err, ShouldBeNil)
			So(pastEvent, ShouldNotBeNil)

			startTimeLive := time.Date(1994, time.March, 29, 01, 02, 03, 00, time.UTC)
			liveEvent, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTimeLive, monthDuration), ownerID, nil)
			So(err, ShouldBeNil)
			So(liveEvent, ShouldNotBeNil)

			startTimeFuture := time.Date(1994, time.May, 29, 01, 02, 03, 00, time.UTC)
			futureEvent, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTimeFuture, monthDuration), ownerID, nil)
			So(err, ShouldBeNil)
			So(futureEvent, ShouldNotBeNil)

			event, err := client.GetLiveEvent(ctx, ownerID, nil, nil)
			So(err, ShouldBeNil)
			So(event, ShouldNotBeNil)
			So(event, ShouldResemble, liveEvent)
		})

		Convey("Should filter out live events playing a different game than the channel", func() {
			ownerID := timestampUser()
			startTimeLive := time.Date(1994, time.March, 29, 01, 02, 03, 00, time.UTC)
			monthDuration := time.Hour * 24 * 30
			endTimeLive := startTimeLive.Add(monthDuration)
			title := "TestIntegration_SingleSpade title " + startTimeLive.String()
			description := "TestIntegration_SingleSpade description " + startTimeLive.String()
			timezoneID := "America/New_York"
			createParams := geaclient.CreateSingleEventParams{
				OwnerID:     ownerID,
				StartTime:   startTimeLive,
				EndTime:     endTimeLive,
				Language:    languageEN,
				Title:       title,
				Description: description,
				ChannelID:   ownerID,
				GameID:      "123",
				TimeZoneID:  timezoneID,
			}
			event1, err := client.CreateSingleEvent(ctx, createParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(event1, ShouldNotBeNil)

			event, err := client.GetLiveEvent(ctx, ownerID, nil, nil)
			So(err, ShouldBeNil)
			So(event, ShouldBeNil)
		})

		Convey("Should NOT return Premiere events", func() {
			ownerID := timestampUser()

			startTimeLive := time.Date(1994, time.March, 29, 01, 02, 03, 00, time.UTC)
			monthDuration := time.Hour * 24 * 30
			liveEvent, err := client.CreatePremiereEvent(ctx, defaultCreatePremiereEventParams(ownerID, startTimeLive, monthDuration), ownerID, nil)
			So(err, ShouldBeNil)
			So(liveEvent, ShouldNotBeNil)

			event, err := client.GetLiveEvent(ctx, ownerID, nil, nil)
			So(err, ShouldBeNil)
			So(event, ShouldBeNil)
		})

		Convey("Should prioritize a live event that starts closer to the current time", func() {
			ownerID := timestampUser()
			startTimeLive1 := time.Date(1994, time.March, 29, 01, 02, 03, 00, time.UTC)
			startTimeLive2 := time.Date(1994, time.April, 01, 01, 02, 03, 00, time.UTC)
			monthDuration := time.Hour * 24 * 30

			event1, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTimeLive1, monthDuration), ownerID, nil)
			So(err, ShouldBeNil)
			So(event1, ShouldNotBeNil)

			event2, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTimeLive2, monthDuration), ownerID, nil)
			So(err, ShouldBeNil)
			So(event2, ShouldNotBeNil)

			event, err := client.GetLiveEvent(ctx, ownerID, nil, nil)
			So(err, ShouldBeNil)
			So(event, ShouldNotBeNil)
			So(event, ShouldResemble, event2)
		})
	})
}

func TestIntegration_Paging(t *testing.T) {
	t.Parallel()
	ts := startServer(t, newDefaultInjectables())
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	// Given a timetable event ...
	var timetableEvent *geaclient.TimetableEvent
	// and some segment events ...
	numSegments := 10
	var segmentEvents []*geaclient.SegmentEvent

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx
		ownerID := timestampUser()

		// Create the timetable event only once to speed up test execution.
		if timetableEvent == nil {
			var err error
			timetableEvent, err = client.CreateTimetableEvent(ctx, timetableParams(ownerID), ownerID, nil)
			So(err, ShouldBeNil)
			So(timetableEvent, ShouldNotBeNil)
		}

		// Create the segment events only once, to speed up test execution.
		if segmentEvents == nil {
			segmentEvents = make([]*geaclient.SegmentEvent, 0, numSegments)

			// Create several segment events on the timetable event.  Record the segment events in ascending
			// start time order.
			startTime := time.Date(2017, 01, 01, 01, 02, 00, 00, time.UTC)
			for i := 0; i < numSegments; i++ {
				startTime = startTime.Add(time.Hour)
				endTime := startTime.Add(time.Minute * time.Duration(30))
				segmentParams := segmentParams(timetableEvent.ID, ownerID, irlGameID, startTime, endTime)

				segmentEvent, err := client.CreateSegmentEvent(ctx, segmentParams, ownerID, nil)
				So(err, ShouldBeNil)
				So(segmentEvent, ShouldNotBeNil)

				segmentEvents = append(segmentEvents, segmentEvent)
			}
		}

		Convey("Should be able to page through all segments, starting from the earliest", func() {
			expectedEventIDs := make([]string, numSegments)
			for i, segmentEvent := range segmentEvents {
				expectedEventIDs[i] = segmentEvent.ID
			}
			limit := 4
			testPagingForwardThroughSegments(ctx, client, limit, timetableEvent.ID, "", nil, expectedEventIDs)
		})

		Convey("Should be able to page through all segments, starting from an event ID", func() {
			startIndex := 2
			expectedEventIDs := make([]string, 0, numSegments-startIndex)
			for i := startIndex; i < numSegments; i++ {
				expectedEventIDs = append(expectedEventIDs, segmentEvents[i].ID)
			}

			limit := 3
			testPagingForwardThroughSegments(ctx, client, limit, timetableEvent.ID, expectedEventIDs[0], nil, expectedEventIDs)
		})

		Convey("Should fallback to paging from the beginning if given a starting event ID that does not exist", func() {
			expectedEventIDs := make([]string, numSegments)
			for i, segmentEvent := range segmentEvents {
				expectedEventIDs[i] = segmentEvent.ID
			}

			limit := 3
			testPagingForwardThroughSegments(ctx, client, limit, timetableEvent.ID, "does-not-exist", nil, expectedEventIDs)
		})

		Convey("Should return an error if given a bad cursor", func() {
			v1Cursor := "10"
			limit := 3
			filterOptions := &geaclient.GetEIDsFilterOpts{
				ParentEventIDs: []string{timetableEvent.ID},
				Cursor:         &v1Cursor,
				Limit:          &limit,
			}

			resp, err := client.GetEventIDsByFilterOptions(ctx, filterOptions, nil)
			So(err, ShouldNotBeNil)
			So(resp, ShouldBeNil)
		})

		Convey("Should NOT return an error if given a cursor that points to an event that doesn't exist", func() {
			cursor, err := db.EncodeNewV2Cursor("does-not-exist", time.Now())
			So(err, ShouldBeNil)

			limit := 3
			filterOptions := &geaclient.GetEIDsFilterOpts{
				ParentEventIDs: []string{timetableEvent.ID},
				Cursor:         &cursor,
				Limit:          &limit,
			}

			resp, err := client.GetEventIDsByFilterOptions(ctx, filterOptions, nil)
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(resp.EventIDs, ShouldNotBeNil)
		})

		Convey("Should be able to page through segments, starting from the event after a given end time", func() {
			// Pick an end time.
			endTimeIndex := 2
			endTime := segmentEvents[endTimeIndex].EndTime

			// Gather the event IDs that we expect to page through.  We can just get the event IDs after the event at
			// endTimeIndex because our events do not overlap.
			startIndex := endTimeIndex + 1
			expectedEventIDs := make([]string, 0, numSegments-startIndex)
			for i := startIndex; i < numSegments; i++ {
				expectedEventIDs = append(expectedEventIDs, segmentEvents[i].ID)
			}

			limit := 3
			testPagingForwardThroughSegments(ctx, client, limit, timetableEvent.ID, "", &endTime, expectedEventIDs)
		})

		Convey("Should be able to page backwards through segments, starting from the latest event ID", func() {
			// Get the cursor that points to our latest event.
			lastSegmentEvent := segmentEvents[len(segmentEvents)-1]
			limit := 1
			filterOptions := &geaclient.GetEIDsFilterOpts{
				ParentEventIDs: []string{timetableEvent.ID},
				Limit:          &limit,
			}
			filterOptions.AddIDFirstPageOption(lastSegmentEvent.ID)
			resp, err := client.GetEventIDsByFilterOptions(ctx, filterOptions, nil)
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(resp.HasPreviousPage, ShouldBeTrue)
			So(resp.FirstIDCursor, ShouldNotBeEmpty)
			cursor := resp.FirstIDCursor

			// Get the event IDs that we expect to page through if we started from the above cursor, and paged backwards.
			// Record the event IDs in descending start time order.
			expectedEventIDs := make([]string, numSegments-1)
			j := numSegments - 2
			for i := 0; i < numSegments-1; i++ {
				expectedEventIDs[i] = segmentEvents[j].ID
				j--
			}

			limit = 3
			expectedNumIDs := len(expectedEventIDs)
			var expectedNumPages int
			if expectedNumIDs%limit == 0 {
				expectedNumPages += expectedNumIDs / limit
			} else {
				expectedNumPages += expectedNumIDs/limit + 1
			}

			actualEventIDsInDescOrder := make([]string, 0, expectedNumIDs)
			hasPreviousPage := true
			page := 0

			// Page backwards, starting from the cursor to the latest event, till there are no more previous pages.
			for hasPreviousPage {
				// Get the previous page of segment event IDs on the timetable event.
				filterOptions := &geaclient.GetEIDsFilterOpts{
					ParentEventIDs:  []string{timetableEvent.ID},
					Cursor:          &cursor,
					Limit:           &limit,
					GetPreviousPage: true,
				}

				resp, err := client.GetEventIDsByFilterOptions(ctx, filterOptions, nil)
				So(err, ShouldBeNil)
				So(resp, ShouldNotBeNil)

				// Each page of events should be ordered by start time in asc order.  Reverse this so that we
				// can keep track of all events seen in descending order.
				for i := len(resp.EventIDs) - 1; i >= 0; i-- {
					actualEventIDsInDescOrder = append(actualEventIDsInDescOrder, resp.EventIDs[i])
				}

				// Verify that the page contains the expected number of IDs, and that we stop paging at the right place.
				So(len(resp.EventIDs), ShouldBeLessThanOrEqualTo, limit)
				if isLastPage := page == expectedNumPages-1; isLastPage {
					So(resp.HasPreviousPage, ShouldBeFalse)
				} else {
					So(resp.HasPreviousPage, ShouldBeTrue)
				}

				// Set up for the next iteration.
				cursor = resp.FirstIDCursor
				hasPreviousPage = resp.HasPreviousPage
				page++
			}

			So(actualEventIDsInDescOrder, ShouldResemble, expectedEventIDs)
		})

		Convey("Latest event should have the correct HasPreviousPage and HasNextPage", func() {
			lastSegmentEvent := segmentEvents[len(segmentEvents)-1]
			limit := 1
			filterOptions := &geaclient.GetEIDsFilterOpts{
				ParentEventIDs: []string{timetableEvent.ID},
				Limit:          &limit,
			}
			filterOptions.AddIDFirstPageOption(lastSegmentEvent.ID)
			resp, err := client.GetEventIDsByFilterOptions(ctx, filterOptions, nil)
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(resp.HasPreviousPage, ShouldBeTrue)
			So(resp.HasNextPage, ShouldBeFalse)
		})

		Convey("First event should have the correct HasPreviousPage and HasNextPage", func() {
			firstSegmentEvent := segmentEvents[0]
			limit := 1
			filterOptions := &geaclient.GetEIDsFilterOpts{
				ParentEventIDs: []string{timetableEvent.ID},
				Limit:          &limit,
			}
			filterOptions.AddIDFirstPageOption(firstSegmentEvent.ID)
			resp, err := client.GetEventIDsByFilterOptions(ctx, filterOptions, nil)
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(resp.HasPreviousPage, ShouldBeFalse)
			So(resp.HasNextPage, ShouldBeTrue)
		})
	})
}

func testPagingForwardThroughSegments(
	ctx context.Context, client geaclient.Client, limit int, timetableEventID string, startAtEventID string, startAfterEndTime *time.Time, expectedEventIDs []string) {

	expectedNumIDs := len(expectedEventIDs)

	var expectedNumPages int
	if expectedNumIDs%limit == 0 {
		expectedNumPages = expectedNumIDs / limit
	} else {
		expectedNumPages = expectedNumIDs/limit + 1
	}

	cursor := ""
	actualEventIDs := make([]string, 0, expectedNumPages)
	hasNextPage := true
	page := 0

	for hasNextPage {
		// Get the next page of segment event IDs on the timetable event.
		filterOptions := &geaclient.GetEIDsFilterOpts{
			ParentEventIDs: []string{timetableEventID},
			Cursor:         &cursor,
			Limit:          &limit,
		}
		if startAtEventID != "" {
			filterOptions.AddIDFirstPageOption(startAtEventID)
		}
		if startAfterEndTime != nil {
			filterOptions.AddEndsAfterFirstPageOption(*startAfterEndTime)
		}

		resp, err := client.GetEventIDsByFilterOptions(ctx, filterOptions, nil)
		So(err, ShouldBeNil)
		So(resp, ShouldNotBeNil)

		actualEventIDs = append(actualEventIDs, resp.EventIDs...)

		// Verify that the page contains the expected number of IDs, and that we stop paging at the right place.
		if isLastPage := page == expectedNumPages-1; isLastPage {
			So(resp.EventIDs, ShouldHaveLength, expectedNumIDs%limit)
			So(resp.HasNextPage, ShouldBeFalse)
		} else {
			So(resp.EventIDs, ShouldHaveLength, limit)
			So(resp.HasNextPage, ShouldBeTrue)
		}

		// Set up for the next iteration.
		cursor = resp.LastIDCursor
		hasNextPage = resp.HasNextPage
		page++
	}

	So(actualEventIDs, ShouldResemble, expectedEventIDs)
}

func TestIntegration_GetManagedLeafIDsByOwner(t *testing.T) {
	t.Parallel()

	now := time.Date(2018, 01, 01, 01, 00, 00, 00, time.UTC)
	injectables := newDefaultInjectables()
	injectables.clock = &testutils.StubClock{ControlledNow: now}

	// Disable the admin list so that we can test authorization.
	injectables.adminList = &testutils.EmptyAdminList{}

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		Convey("Should be able to get leaf IDs if owner", func() {
			ownerID := timestampUser()

			startTime := now.Add(time.Hour)
			params := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			ev, err := client.CreateSingleEvent(ctx, params, ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			eventIDs, err := client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, nil, nil)
			So(err, ShouldBeNil)
			So(eventIDs, ShouldNotBeNil)

			So(eventIDs.EventIDs, ShouldHaveLength, 1)
			So(eventIDs.EventIDs[0], ShouldEqual, ev.ID)

			So(eventIDs.Items, ShouldHaveLength, 1)
			So(eventIDs.EventIDs[0], ShouldEqual, ev.ID)
		})

		Convey("Should get an error if caller is not owner or an editor", func() {
			ownerID := timestampUser()
			callerID := timestampUser()
			So(ownerID, ShouldNotEqual, callerID)

			startTime := now.Add(time.Hour)
			params := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			ev, err := client.CreateSingleEvent(ctx, params, ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			eventIDs, err := client.GetManagedLeafIDsByOwner(ctx, callerID, ownerID, nil, nil)
			So(err, ShouldNotBeNil)
			So(eventIDs, ShouldBeNil)
		})

		Convey("Should not get timetable events", func() {
			ownerID := timestampUser()

			params := timetableParams(ownerID)
			ev, err := client.CreateTimetableEvent(ctx, params, ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			eventIDs, err := client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, nil, nil)
			So(err, ShouldBeNil)
			So(eventIDs, ShouldNotBeNil)
			So(eventIDs.EventIDs, ShouldHaveLength, 0)
		})

		Convey("Should be able to page through leaf IDs", func() {
			ownerID := timestampUser()

			startTime := now

			numEvents := 5
			allEventIDs := make([]string, numEvents)

			// Create several single events, and store their IDs in ascending order by start time.
			for i := 0; i < numEvents; i++ {
				startTime = startTime.Add(time.Hour)
				params := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
				ev, err := client.CreateSingleEvent(ctx, params, ownerID, nil)
				So(err, ShouldBeNil)
				So(ev, ShouldNotBeNil)

				allEventIDs[i] = ev.ID
			}

			// Get the first page of events, sorted in ascending order by start time.
			limit := 2
			opts := &geaclient.GetManagedLeafIDsOpts{
				Limit: limit,
			}
			eventIDs, err := client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, opts, nil)
			So(err, ShouldBeNil)
			So(eventIDs, ShouldNotBeNil)
			So(eventIDs.EventIDs, ShouldHaveLength, 2)
			So(eventIDs.EventIDs[0], ShouldEqual, allEventIDs[0])
			So(eventIDs.EventIDs[1], ShouldEqual, allEventIDs[1])
			So(eventIDs.FirstIDCursor, ShouldNotBeEmpty)
			So(eventIDs.LastIDCursor, ShouldNotBeEmpty)
			So(eventIDs.HasNextPage, ShouldBeTrue)
			So(eventIDs.HasPreviousPage, ShouldBeFalse)

			// Get the next (second) page of events, sorted in ascending order by start time.
			opts = &geaclient.GetManagedLeafIDsOpts{
				Limit:  limit,
				Cursor: eventIDs.LastIDCursor,
			}
			eventIDs, err = client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, opts, nil)
			So(err, ShouldBeNil)
			So(eventIDs, ShouldNotBeNil)
			So(eventIDs.EventIDs, ShouldHaveLength, 2)
			So(eventIDs.EventIDs[0], ShouldEqual, allEventIDs[2])
			So(eventIDs.EventIDs[1], ShouldEqual, allEventIDs[3])
			So(eventIDs.FirstIDCursor, ShouldNotBeEmpty)
			So(eventIDs.LastIDCursor, ShouldNotBeEmpty)
			So(eventIDs.HasNextPage, ShouldBeTrue)
			So(eventIDs.HasPreviousPage, ShouldBeTrue)

			// Get the next (last) page of events, sorted in ascending order by start time.
			opts = &geaclient.GetManagedLeafIDsOpts{
				Limit:  limit,
				Cursor: eventIDs.LastIDCursor,
			}
			eventIDs, err = client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, opts, nil)
			So(err, ShouldBeNil)
			So(eventIDs, ShouldNotBeNil)
			So(eventIDs.EventIDs, ShouldHaveLength, 1)
			So(eventIDs.EventIDs[0], ShouldEqual, allEventIDs[4])
			So(eventIDs.FirstIDCursor, ShouldNotBeEmpty)
			So(eventIDs.LastIDCursor, ShouldNotBeEmpty)
			So(eventIDs.HasNextPage, ShouldBeFalse)
			So(eventIDs.HasPreviousPage, ShouldBeTrue)

			// Get the previous (second) page of events, sorted in ascending order by start time.
			opts = &geaclient.GetManagedLeafIDsOpts{
				Limit:           limit,
				Cursor:          eventIDs.FirstIDCursor,
				GetPreviousPage: true,
			}
			eventIDs, err = client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, opts, nil)
			So(err, ShouldBeNil)
			So(eventIDs, ShouldNotBeNil)
			So(eventIDs.EventIDs, ShouldHaveLength, 2)
			So(eventIDs.EventIDs[0], ShouldEqual, allEventIDs[2])
			So(eventIDs.EventIDs[1], ShouldEqual, allEventIDs[3])
			So(eventIDs.FirstIDCursor, ShouldNotBeEmpty)
			So(eventIDs.LastIDCursor, ShouldNotBeEmpty)
			So(eventIDs.HasNextPage, ShouldBeTrue)
			So(eventIDs.HasPreviousPage, ShouldBeTrue)

			// Get the previous (first) page of events, sorted in ascending order by start time.
			opts = &geaclient.GetManagedLeafIDsOpts{
				Limit:           limit,
				Cursor:          eventIDs.FirstIDCursor,
				GetPreviousPage: true,
			}
			eventIDs, err = client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, opts, nil)
			So(err, ShouldBeNil)
			So(eventIDs, ShouldNotBeNil)
			So(eventIDs.EventIDs, ShouldHaveLength, 2)
			So(eventIDs.EventIDs[0], ShouldEqual, allEventIDs[0])
			So(eventIDs.EventIDs[1], ShouldEqual, allEventIDs[1])
			So(eventIDs.FirstIDCursor, ShouldNotBeEmpty)
			So(eventIDs.LastIDCursor, ShouldNotBeEmpty)
			So(eventIDs.HasNextPage, ShouldBeTrue)
			So(eventIDs.HasPreviousPage, ShouldBeFalse)
		})

		Convey("Should have cursors for every item", func() {
			ownerID := timestampUser()

			startTime := now

			numEvents := 5
			allEventIDs := make([]string, numEvents)

			// Create several single events, and store their IDs in ascending order by start time.
			for i := 0; i < numEvents; i++ {
				startTime = startTime.Add(time.Hour)
				params := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
				ev, err := client.CreateSingleEvent(ctx, params, ownerID, nil)
				So(err, ShouldBeNil)
				So(ev, ShouldNotBeNil)

				allEventIDs[i] = ev.ID
			}

			// Get the first page of events, sorted in ascending order by start time.
			limit := 4
			opts := &geaclient.GetManagedLeafIDsOpts{
				Limit: limit,
			}
			managedLeafIDs, err := client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, opts, nil)
			So(err, ShouldBeNil)
			So(managedLeafIDs, ShouldNotBeNil)

			So(managedLeafIDs.Items, ShouldHaveLength, limit)
			So(managedLeafIDs.EventIDs, ShouldHaveLength, limit)
			So(managedLeafIDs.HasPreviousPage, ShouldBeFalse)
			So(managedLeafIDs.HasNextPage, ShouldBeTrue)

			for i, item := range managedLeafIDs.Items {
				So(item.Cursor, ShouldNotBeEmpty)
				So(item.EventID, ShouldEqual, managedLeafIDs.EventIDs[i])
			}
			So(managedLeafIDs.Items[0].Cursor, ShouldEqual, managedLeafIDs.FirstIDCursor)
			So(managedLeafIDs.Items[limit-1].Cursor, ShouldEqual, managedLeafIDs.LastIDCursor)
		})

		Convey("Should be able to get item in desc order", func() {
			ownerID := timestampUser()

			startTime := now

			numEvents := 5
			allEventIDs := make([]string, numEvents)

			// Create several single events, and store their IDs in ascending order by start time.
			for i := 0; i < numEvents; i++ {
				startTime = startTime.Add(time.Hour)
				params := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
				ev, err := client.CreateSingleEvent(ctx, params, ownerID, nil)
				So(err, ShouldBeNil)
				So(ev, ShouldNotBeNil)

				allEventIDs[i] = ev.ID
			}

			// Get the first page of events, sorted in descending order by start time.
			limit := 4
			opts := &geaclient.GetManagedLeafIDsOpts{
				Limit:      limit,
				Descending: true,
			}
			managedLeafIDs, err := client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, opts, nil)
			So(err, ShouldBeNil)
			So(managedLeafIDs, ShouldNotBeNil)

			So(managedLeafIDs.Items, ShouldHaveLength, limit)
			So(managedLeafIDs.EventIDs, ShouldHaveLength, limit)

			So(managedLeafIDs.Items[limit-1].Cursor, ShouldNotBeEmpty)
		})

		Convey("Should be able to filter by time", func() {
			ownerID := timestampUser()

			startTime := time.Date(2017, 06, 01, 01, 00, 00, 00, time.UTC)
			params := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			ev, err := client.CreateSingleEvent(ctx, params, ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			startTime = time.Date(2017, 07, 01, 01, 00, 00, 00, time.UTC)
			params = defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			ev2, err := client.CreateSingleEvent(ctx, params, ownerID, nil)
			So(err, ShouldBeNil)

			Convey("Should be able to filter by start time", func() {
				opts := &geaclient.GetManagedLeafIDsOpts{
					StartTimeAfter:  refTime(ev.StartTime.Add(-time.Minute)),
					StartTimeBefore: refTime(ev.StartTime.Add(time.Minute)),
				}
				eventIDs, err := client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, opts, nil)
				So(err, ShouldBeNil)
				So(eventIDs, ShouldNotBeNil)
				So(eventIDs.EventIDs, ShouldHaveLength, 1)
				So(eventIDs.EventIDs[0], ShouldEqual, ev.ID)
			})

			Convey("Should be able to filter by end time", func() {
				opts := &geaclient.GetManagedLeafIDsOpts{
					EndTimeAfter:  refTime(ev2.EndTime.Add(-time.Minute)),
					EndTimeBefore: refTime(ev2.EndTime.Add(time.Minute)),
				}
				eventIDs, err := client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, opts, nil)
				So(err, ShouldBeNil)
				So(eventIDs, ShouldNotBeNil)
				So(eventIDs.EventIDs, ShouldHaveLength, 1)
				So(eventIDs.EventIDs[0], ShouldEqual, ev2.ID)
			})
		})

		Convey("Should be get event IDs in descending order", func() {
			ownerID := timestampUser()

			startTime := time.Date(2017, 06, 01, 01, 00, 00, 00, time.UTC)
			params := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			ev, err := client.CreateSingleEvent(ctx, params, ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			startTime = time.Date(2017, 07, 01, 01, 00, 00, 00, time.UTC)
			params = defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			ev2, err := client.CreateSingleEvent(ctx, params, ownerID, nil)
			So(err, ShouldBeNil)

			opts := &geaclient.GetManagedLeafIDsOpts{
				StartTimeAfter:  refTime(ev.StartTime.Add(-time.Minute)),
				StartTimeBefore: refTime(ev2.StartTime.Add(time.Minute)),
				Descending:      true,
			}

			eventIDs, err := client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, opts, nil)
			So(err, ShouldBeNil)
			So(eventIDs, ShouldNotBeNil)
			So(eventIDs.EventIDs, ShouldHaveLength, 2)
			So(eventIDs.EventIDs[0], ShouldEqual, ev2.ID)
			So(eventIDs.EventIDs[1], ShouldEqual, ev.ID)
		})

		Convey("Should be able to start listing using end time", func() {
			ownerID := timestampUser()

			startTime := time.Date(2017, 06, 01, 01, 00, 00, 00, time.UTC)
			params := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			ev, err := client.CreateSingleEvent(ctx, params, ownerID, nil)
			So(err, ShouldBeNil)
			So(ev, ShouldNotBeNil)

			startTime = time.Date(2017, 07, 01, 01, 00, 00, 00, time.UTC)
			params = defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
			ev2, err := client.CreateSingleEvent(ctx, params, ownerID, nil)
			So(err, ShouldBeNil)

			opts := &geaclient.GetManagedLeafIDsOpts{
				StartTimeAfter:     refTime(ev.StartTime.Add(-time.Minute)),
				StartTimeBefore:    refTime(ev2.StartTime.Add(time.Minute)),
				StartListAtEventID: ev2.ID,
			}

			eventIDs, err := client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, opts, nil)
			So(err, ShouldBeNil)
			So(eventIDs, ShouldNotBeNil)
			So(eventIDs.EventIDs, ShouldHaveLength, 1)
			So(eventIDs.EventIDs[0], ShouldEqual, ev2.ID)
			So(eventIDs.HasPreviousPage, ShouldBeTrue)
			So(eventIDs.HasNextPage, ShouldBeFalse)
		})
	})
}

func TestIntegration_GetManagedLeafIDsByParentID(t *testing.T) {
	t.Parallel()

	now := time.Date(2017, 01, 01, 01, 00, 00, 00, time.UTC)
	injectables := newDefaultInjectables()
	injectables.clock = &testutils.StubClock{ControlledNow: now}

	// Disable the admin list so that we can test authorization.
	injectables.adminList = &testutils.EmptyAdminList{}

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		ownerID := timestampUser()
		tParams := timetableParams(ownerID)
		timetableEvent, err := client.CreateTimetableEvent(ctx, tParams, ownerID, nil)
		So(err, ShouldBeNil)
		So(timetableEvent, ShouldNotBeNil)

		startTime := now.Add(time.Hour)
		segParams := segmentParams(timetableEvent.ID, ownerID, overwatchGameID, startTime, startTime.Add(time.Hour))
		segmentEvent, err := client.CreateSegmentEvent(ctx, segParams, ownerID, nil)
		So(err, ShouldBeNil)
		So(segmentEvent, ShouldNotBeNil)

		singleParams := defaultCreateSingleEventParams(ownerID, startTime, time.Hour)
		singleEvent, err := client.CreateSingleEvent(ctx, singleParams, ownerID, nil)
		So(err, ShouldBeNil)
		So(singleEvent, ShouldNotBeNil)

		Convey("Should be able to get children of timetable", func() {
			callerID := ownerID

			eventIDs, err := client.GetManagedLeafIDsByParent(ctx, timetableEvent.ID, callerID, nil, nil)
			So(err, ShouldBeNil)
			So(eventIDs, ShouldNotBeNil)

			So(eventIDs.EventIDs, ShouldHaveLength, 1)
			So(eventIDs.EventIDs[0], ShouldEqual, segmentEvent.ID)
		})

		Convey("Should get an error if caller is not owner or an editor", func() {
			callerID := timestampUser()

			eventIDs, err := client.GetManagedLeafIDsByParent(ctx, timetableEvent.ID, callerID, nil, nil)
			So(err, ShouldNotBeNil)
			So(eventIDs, ShouldBeNil)
		})
	})
}

func TestIntegration_TwitchAdminAuth(t *testing.T) {
	t.Parallel()

	stubAuthUsersClient := testutils.NewStubAuthUsersClient(false, true)

	injectables := newDefaultInjectables()
	injectables.authUsersClient = stubAuthUsersClient

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		normalUserID := timestampUser()
		twitchAdminID := timestampUser()

		tParams := defaultCreateSingleEventParams(normalUserID, time.Now(), time.Hour)
		createdEvent, err := client.CreateSingleEvent(ctx, tParams, normalUserID, nil)
		So(err, ShouldBeNil)
		So(createdEvent, ShouldNotBeNil)

		eventID := createdEvent.ID

		Convey("Twitch Admins should be load anyone's event dashboard", func() {
			managedLeafIDs, err := client.GetManagedLeafIDsByOwner(ctx, normalUserID, twitchAdminID, nil, nil)
			So(err, ShouldBeNil)
			So(managedLeafIDs, ShouldNotBeNil)
			So(managedLeafIDs.Items, ShouldHaveLength, 1)
			So(managedLeafIDs.Items[0].EventID, ShouldEqual, eventID)

			_, err = client.GetManagedCollectionIDsByOwner(ctx, twitchAdminID, normalUserID, nil, nil)
			So(err, ShouldBeNil)
		})

		Convey("Twitch Admins should be allowed to delete anyone's events", func() {
			deletedEvent, err := client.DeleteEvent(ctx, eventID, twitchAdminID, nil)
			So(err, ShouldBeNil)
			So(deletedEvent, ShouldNotBeNil)
		})
	})
}

func TestIntegration_SuspendedUsersAuth(t *testing.T) {
	t.Parallel()

	stubAuthUsersClient := testutils.NewStubAuthUsersClient(true, false)

	injectables := newDefaultInjectables()
	injectables.authUsersClient = stubAuthUsersClient

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx

		ownerID := timestampUser()

		Convey("Suspended users should NOT be allowed to create events", func() {
			stubAuthUsersClient.SetIsSuspended(true)

			tParams := defaultCreateSingleEventParams(ownerID, time.Now(), time.Hour)
			singleEvent, err := client.CreateSingleEvent(ctx, tParams, ownerID, nil)
			So(err, ShouldNotBeNil)
			So(singleEvent, ShouldBeNil)
		})

		Convey("Suspended users should NOT be allowed to delete events", func() {
			// Unsuspend the user and create an event
			stubAuthUsersClient.SetIsSuspended(false)

			tParams := defaultCreateSingleEventParams(ownerID, time.Now(), time.Hour)
			singleEvent, err := client.CreateSingleEvent(ctx, tParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(singleEvent, ShouldNotBeNil)

			// Suspend the user
			stubAuthUsersClient.SetIsSuspended(true)

			// Verify that the user cannot delete the event.
			deletedEvent, err := client.DeleteEvent(ctx, singleEvent.ID, ownerID, nil)
			So(err, ShouldNotBeNil)
			So(deletedEvent, ShouldBeNil)
		})

		Convey("Suspended users should NOT be allowed to update events", func() {
			// Unsuspend the user and create an event
			stubAuthUsersClient.SetIsSuspended(false)

			tParams := defaultCreateSingleEventParams(ownerID, time.Now(), time.Hour)
			singleEvent, err := client.CreateSingleEvent(ctx, tParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(singleEvent, ShouldNotBeNil)

			// Suspend the user
			stubAuthUsersClient.SetIsSuspended(true)

			// Verify that the user cannot update the event.
			updateParams := geaclient.UpdateSingleEventParams{
				Title: refStr("new title"),
			}
			updatedEvent, err := client.UpdateSingleEvent(ctx, singleEvent.ID, updateParams, ownerID, nil)
			So(err, ShouldNotBeNil)
			So(updatedEvent, ShouldBeNil)
		})

		Convey("Suspended users should be allowed to see their events in the dashboard", func() {
			// Unsuspend the user and create an event
			stubAuthUsersClient.SetIsSuspended(false)

			tParams := defaultCreateSingleEventParams(ownerID, time.Now(), time.Hour)
			singleEvent, err := client.CreateSingleEvent(ctx, tParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(singleEvent, ShouldNotBeNil)

			// Suspend the user
			stubAuthUsersClient.SetIsSuspended(true)

			// Verify that the user can see their events in a mangemement context.
			managedLeafIDs, err := client.GetManagedLeafIDsByOwner(ctx, ownerID, ownerID, nil, nil)
			So(err, ShouldBeNil)
			So(managedLeafIDs, ShouldNotBeNil)
			So(managedLeafIDs.Items, ShouldHaveLength, 1)
		})
	})
}

func resetMock(m *mock.Mock) {
	m.ExpectedCalls = nil
	m.Calls = nil
}

type nilCheckMock struct {
	*mocks.SpadeClient `nilcheck:"nodepth"`
}

func TestIntegration_Spade(t *testing.T) {
	t.Parallel()

	controlledNow := time.Unix(999999990, 0).UTC()
	controlledNowUnix := controlledNow.Unix()

	spadeMock := nilCheckMock{&mocks.SpadeClient{}}
	spadeMock.On("Start").Return()

	injectables := newDefaultInjectables()
	injectables.spadeClient = spadeMock
	injectables.clock = &testutils.StubClock{ControlledNow: controlledNow}

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	checkForSpadeEvent := func(expectedEvent spade.Event) {
		events := make([]spade.Event, 0)

		for _, call := range spadeMock.SpadeClient.Mock.Calls {
			if call.Method == "QueueEvents" {
				if event, ok := call.Arguments[0].(spade.Event); ok {
					events = append(events, event)
				}
			}
		}
		count := 0
		for _, event := range events {
			if reflect.DeepEqual(event, expectedEvent) {
				count++
			}
		}
		So(count, ShouldEqual, 1)
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := ts.ctx
		resetMock(&spadeMock.Mock)
		spadeMock.On("Start").Return()
		spadeMock.On("QueueEvents", mock.Anything).Return()

		Convey("Spade event should fire on follow and unfollow", func() {
			ownerID := timestampUser()
			startTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

			ev1, err := client.CreateSingleEvent(ctx, defaultCreateSingleEventParams(ownerID, startTime, time.Hour), ownerID, nil)
			So(err, ShouldBeNil)
			So(ev1, ShouldNotBeNil)

			userID := timestampUser()
			err = client.FollowEvent(ctx, ev1.ID, userID, nil)
			So(err, ShouldBeNil)

			numericalUserID, err := strconv.ParseInt(userID, 10, 64)
			So(err, ShouldBeNil)

			checkForSpadeEvent(spade.Event{
				Name: "oracle_follow_server",
				Properties: models.ServerFollowTracking{
					Action:      "follow",
					UserID:      numericalUserID,
					EventID:     ev1.ID,
					FollowCount: 1,
				},
			})

			err = client.UnfollowEvent(ctx, ev1.ID, userID, nil)
			So(err, ShouldBeNil)

			checkForSpadeEvent(spade.Event{
				Name: "oracle_follow_server",
				Properties: models.ServerFollowTracking{
					Action:      "unfollow",
					UserID:      numericalUserID,
					EventID:     ev1.ID,
					FollowCount: 0,
				},
			})
		})

		Convey("Spade event should fire on event create, update, and delete", func() {
			ownerID := "1234"
			numericalOwnerID := 1234
			startTime := time.Unix(999999999, 0)
			startUnix := startTime.Unix()
			duration := time.Duration(time.Minute * 120)
			endTime := startTime.Add(duration)
			endUnix := endTime.Unix()
			title := "TestIntegration_SingleSpade title " + startTime.String()
			description := "TestIntegration_SingleSpade description " + startTime.String()
			timezoneID := "America/New_York"
			createParams := geaclient.CreateSingleEventParams{
				OwnerID:     ownerID,
				StartTime:   startTime,
				EndTime:     endTime,
				Language:    languageEN,
				Title:       title,
				Description: description,
				ChannelID:   ownerID,
				GameID:      overwatchGameID,
				TimeZoneID:  timezoneID,
			}
			event, err := client.CreateSingleEvent(ctx, createParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(event, ShouldNotBeNil)

			checkForSpadeEvent(spade.Event{
				Name: "oracle_event_server",
				Properties: models.ServerEventTracking{
					Action:        "create",
					UserID:        int64(numericalOwnerID),
					EventID:       event.ID,
					EventType:     "single",
					ParentEventID: "",
					CreatedAt:     controlledNowUnix,
					UpdatedAt:     controlledNowUnix,
					StartTime:     &startUnix,
					EndTime:       &endUnix,
					TimeZoneID:    timezoneID,
					ChannelID:     ownerID,
					Title:         title,
					Description:   description,
					GameID:        int64(numericalOverwatchGameID),
				},
			})

			updateTime := startTime.Add(time.Minute * 10)
			controlledUpdatedAt := controlledNow.Add(time.Minute).UTC()
			controlledUpdatedAtUnix := controlledUpdatedAt.Unix()
			ts.thisInstance.server.Clock = &testutils.StubClock{ControlledNow: controlledUpdatedAt}
			title2 := "TestIntegreation_SingleSpade title2 " + updateTime.String()
			updateParams := geaclient.UpdateSingleEventParams{
				Title: &title2,
			}
			event, err = client.UpdateSingleEvent(ctx, event.ID, updateParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(event, ShouldNotBeNil)

			checkForSpadeEvent(spade.Event{
				Name: "oracle_event_server",
				Properties: models.ServerEventTracking{
					Action:        "update",
					UserID:        int64(numericalOwnerID),
					EventID:       event.ID,
					EventType:     "single",
					ParentEventID: "",
					CreatedAt:     controlledNowUnix,
					UpdatedAt:     controlledUpdatedAtUnix,
					StartTime:     &startUnix,
					EndTime:       &endUnix,
					TimeZoneID:    timezoneID,
					ChannelID:     ownerID,
					Title:         title2,
					Description:   description,
					GameID:        int64(numericalOverwatchGameID),
					CoverImageID:  defaultCoverImageID,
				},
			})

			controlledDeletedAt := controlledUpdatedAt.Add(time.Minute).UTC()
			controlledDeletedAtUnix := controlledDeletedAt.Unix()
			ts.thisInstance.server.Clock = &testutils.StubClock{ControlledNow: controlledDeletedAt}
			deletedEvent, err := client.DeleteEvent(ctx, event.ID, ownerID, nil)
			So(err, ShouldBeNil)
			So(event, ShouldNotBeNil)

			checkForSpadeEvent(spade.Event{
				Name: "oracle_event_server",
				Properties: models.ServerEventTracking{
					Action:        "remove",
					UserID:        int64(numericalOwnerID),
					EventID:       deletedEvent.GetID(),
					EventType:     "single",
					ParentEventID: "",
					CreatedAt:     controlledNowUnix,
					UpdatedAt:     controlledDeletedAtUnix,
					StartTime:     &startUnix,
					EndTime:       &endUnix,
					TimeZoneID:    timezoneID,
					ChannelID:     ownerID,
					Title:         title2,
					Description:   description,
					GameID:        int64(numericalOverwatchGameID),
					CoverImageID:  defaultCoverImageID,
				},
			})
		})
	})
}

func refBool(b bool) *bool {
	return &b
}

func refInt(i int) *int {
	return &i
}

func refStr(s string) *string {
	return &s
}

func refTime(t time.Time) *time.Time {
	return &t
}

func findEvent(events []geaclient.Event, id string) geaclient.Event {
	for _, event := range events {
		if event.GetID() == id {
			return event
		}
	}
	return nil
}

type testSetup struct {
	ctx          context.Context
	client       geaclient.Client
	httpClient   *http.Client
	host         string
	onFinish     func(timeToWait time.Duration)
	thisInstance *service
}

func (t *testSetup) Setup() error {
	ctx, cancelFunc := context.WithTimeout(context.Background(), time.Second*5)
	t.ctx = ctx
	Reset(cancelFunc)
	client, err := geaclient.NewClient(twitchclient.ClientConf{
		Host: t.host,
	})
	if err != nil {
		return err
	}
	t.client = client
	t.httpClient = &http.Client{}
	return nil
}

func request(ts *testSetup, method, url string, body interface{}, into interface{}) error {
	return clients.DoHTTP(ts.ctx, ts.httpClient, method, ts.host+url, nil, body, into, nil)
}

func addMapValues(m *distconf.InMemory, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}

type panicPanic struct{}

func (p panicPanic) OnPanic(pnc interface{}) {
	panic(pnc)
}

// newDefaultInjectables returns a set of default injectables that should be used with most tests.  These injectables
// disable operations that interfere with most tests.
func newDefaultInjectables() injectables {
	return injectables{
		// Use a GetUsersClient that allows all users to pass Term of Service violation checks.
		//
		// The problem was that many of our handlers call the users service to check if a user is banned, so that
		// we can filter out banned content.  We use fake users for many of our tests, which causes the calls to the
		// user service to fail.
		getUsersClient: NewGetUsersClientStub(),

		// Use a channelEventSQSClient that enqueues channel IDs to an in-memory store, instead of a real SQS queue.
		//
		// Some of Gea's handlers enqueue a channel ID to the channel event SQS queue.  This queue gets read
		// by a worker, which pushes the channel's live event information to PubSub.
		// When our tests enqueue channel IDs, workers either running locally, or in the Integration env read these
		// channel IDs, and try to load the games that these channels are playing.  Since the channel IDs are fake,
		// the calls fail, and fill our logs with confusing errors.
		// To work around the problem, we use a stub to prevent our tests from enqueue-ing to the real SQS queue.
		channelEventSQSClient: &stubChannelEventSQSClient{},

		// Use an admin list where no user is an admin.  We want to ensure that the authorization logic correctly
		// gates operations.
		adminList: &testutils.EmptyAdminList{},

		// Use an authUsersClient that returns that all users are not suspended, and are not Twitch admins.
		authUsersClient: testutils.NewStubAuthUsersClient(false, false),
	}
}

// GetUsersClientStub allows tests to set the users that have Term of Service violations.  It allows tests
// to effectively skip checking if an event's owner is banned, unless the test is explicitly exercising
// how that situation is handled.
type GetUsersClientStub struct {
	bannedUsers map[string]struct{}
}

func NewGetUsersClientStub() *GetUsersClientStub {
	return &GetUsersClientStub{
		bannedUsers: map[string]struct{}{},
	}
}

func (c *GetUsersClientStub) GetUsers(ctx context.Context, params *usersmodels.FilterParams, reqOpts *twitchclient.ReqOpts) (*usersmodels.PropertiesResult, error) {
	results := make([]*usersmodels.Properties, 0, len(params.IDs))
	for _, id := range params.IDs {
		if _, banned := c.bannedUsers[id]; banned {
			continue
		}

		prop := &usersmodels.Properties{
			ID: id,
		}
		results = append(results, prop)
	}
	return &usersmodels.PropertiesResult{
		Results: results,
	}, nil
}

func (c *GetUsersClientStub) SetBannedUsers(userIDs []string) {
	for _, userID := range userIDs {
		c.bannedUsers[userID] = struct{}{}
	}
}

func startServer(t *testing.T, i injectables, configs ...map[string][]byte) *testSetup {
	localConf := &distconf.InMemory{}
	err := addMapValues(localConf, map[string][]byte{
		// Do not hit AWS Parameter Store
		// This is because tests are run concurrently making it likely
		// that the Parameter store API limit is reached, thus making the test fail
		"ssm.region":              []byte(""),
		"aws.region":              []byte("us-west-2"),
		"gea.listen_addr":         []byte(":0"),
		"rollbar.access_token":    []byte(""),
		"statsd.hostport":         []byte(""),
		"debug.addr":              []byte(":0"),
		"logging.to_stdout":       []byte("false"),
		"logging.to_stderr":       []byte("true"),
		"dynamo.consistent_reads": []byte("true"),
	})
	if err != nil {
		t.Error(err)
		return nil
	}
	for _, config := range configs {
		err := addMapValues(localConf, config)
		if err != nil {
			t.Error(err)
			return nil
		}
	}

	started := make(chan string)
	finished := make(chan struct{})
	signalToClose := make(chan os.Signal)
	exitCalled := make(chan struct{})
	elevateKey := "hi"
	thisInstance := service{
		injectables: i,
		osExit: func(i int) {
			if i != 0 {
				t.Error("Invalid osExit status code", i)
			}
			close(exitCalled)
		},
		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				CustomReaders: []distconf.Reader{localConf},
				BaseDirectory: "../../",
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
			},
			SfxSetupConfig: sfxStastdConfig(),
			CodeVersion:    CodeVersion,
			Log: &log.ElevatedLog{
				ElevateKey: elevateKey,
				NormalLog: log.ContextLogger{
					Logger: t,
				},
				DebugLog: log.ContextLogger{
					Logger: log.Discard,
				},
				LogToDebug: func(_ ...interface{}) bool {
					return false
				},
			},
			PanicLogger: panicPanic{},
		},
		sigChan: signalToClose,
		onListen: func(listeningAddr net.Addr) {
			started <- fmt.Sprintf("http://localhost:%d", listeningAddr.(*net.TCPAddr).Port)
		},
	}
	thisInstance.serviceCommon.Log.NormalLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	thisInstance.serviceCommon.Log.DebugLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	go func() {
		thisInstance.main()
		close(finished)
	}()

	var addressForIntegrationTests string
	select {
	case <-exitCalled:
		return nil
	case addressForIntegrationTests = <-started:
	case <-time.After(time.Second * 35):
		t.Error("Took to long to start service")
		return nil
	}

	onFinish := func(timeToWait time.Duration) {
		signalToClose <- syscall.SIGTERM
		select {
		case <-finished:
			return
		case <-time.After(timeToWait):
			t.Error("Timed out waiting for server to end")
		}
	}
	return &testSetup{
		host:         addressForIntegrationTests,
		onFinish:     onFinish,
		thisInstance: &thisInstance,
	}
}
