// +build integration

package main

import (
	"context"
	"fmt"
	"strconv"
	"strings"
	"sync"
	"testing"
	"time"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/hypeman"
	hypemanscheduler "code.justin.tv/twitch-events/gea/internal/hypeman-scheduler"
	jax "code.justin.tv/twitch-events/gea/internal/jax-client"
	"code.justin.tv/twitch-events/gea/lib/geaclient"
	jaxclient "code.justin.tv/web/jax/client"
	. "github.com/smartystreets/goconvey/convey"
	net_context "golang.org/x/net/context"
)

// dbLimit is the max number of rows that will be loaded from postgres at a time.  We lower the limit to reduce the
// number of events we have to create in the tests.
const dbLimit = 4

var (
	offsetStartMustAtOrAfter = hypemanscheduler.DefaultOffsetStartMustAtOrAfter
	offsetStartMustBefore    = hypemanscheduler.DefaultOffsetStartMustBefore
	defaultEventLength       = time.Hour
)

func TestSendEventStartedNotifications(t *testing.T) {
	t.Parallel()
	hypemanClient := &stubHypemanClient{}

	injectables := newDefaultInjectables()
	injectables.hypemanClient = hypemanClient

	ts := startServer(t, injectables)
	if ts == nil {
		t.Fatalf("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.client
		ctx := context.Background()

		Convey("Scheduler should queue notification jobs for given events", func() {
			eventIDs := []string{db.NewID(), db.NewID()}

			err := client.SendEventStartedNotifications(ctx, eventIDs, nil)
			So(err, ShouldBeNil)

			addedEventIDs := hypemanClient.getEventIDs()
			So(addedEventIDs, ShouldHaveLength, len(eventIDs))
			So(addedEventIDs, ShouldNotContainDuplicates)

			for _, eventID := range addedEventIDs {
				So(eventIDs, ShouldContain, eventID)
			}

			Convey("Scheduler should not queue jobs for events that already have notification jobs", func() {
				hypemanClient.clear()

				err := client.SendEventStartedNotifications(ctx, eventIDs, nil)
				So(err, ShouldBeNil)

				addedEventIDs := hypemanClient.getEventIDs()
				So(addedEventIDs, ShouldHaveLength, 0)
			})
		})

		Convey("Scheduler should not create multiple notification jobs for an event", func() {
			eventID := db.NewID()
			eventIDs := []string{eventID, eventID}

			err := client.SendEventStartedNotifications(ctx, eventIDs, nil)
			So(err, ShouldBeNil)

			addedEventIDs := hypemanClient.getEventIDs()
			So(addedEventIDs, ShouldHaveLength, 1)
			So(addedEventIDs[0], ShouldEqual, eventID)
		})
	})
}

func TestCreateNotificationJobs(t *testing.T) {
	t.Parallel()
	now := db.ConvertToDBTime(time.Now())
	testInjectables := newTestInjectables()

	injectables := newDefaultInjectables()
	injectables.schedulerNowFunc = testInjectables.schedulerTimeProvider.getTime
	injectables.jaxClient = testInjectables.jaxClient
	injectables.hypemanClient = testInjectables.hypemanClient
	injectables.createNotificationJobsDoneFunc = testInjectables.wg.Done

	ts := startServer(t, injectables, map[string][]byte{
		"hypeman.db_limit": []byte(strconv.Itoa(dbLimit)),
	})
	if ts == nil {
		t.Fatalf("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)

		Convey("Scheduler should queue events that have started recently, whose channels are live.", func() {
			testDef := &testCase{
				eventProps: []*eventProperties{
					{
						startTime:           now,
						whenChannelGoesLive: &now,
					},
				},
				timeSchedulerChecksForEvents: now,
			}
			output := doRun(ts, testInjectables, testDef)
			assertCreatedEventsWereScheduled(testDef, output)

			Convey("Another call should not return events that already been queued.", func() {
				prevScheduledEvents := output.scheduledEvents

				testDef := &testCase{
					timeSchedulerChecksForEvents: now,
				}
				output := doRun(ts, testInjectables, testDef)

				So(output.scheduledEvents, ShouldNotContainDuplicates)
				for _, eventID := range prevScheduledEvents {
					So(output.scheduledEvents, ShouldNotContain, eventID)
				}
			})
		})

		Convey("Scheduler should NOT queue events that have NOT started, and are not starting soon", func() {
			testDef := &testCase{
				eventProps: []*eventProperties{
					{
						startTime:           now.Add(offsetStartMustBefore),
						whenChannelGoesLive: &now,
					},
				},
				timeSchedulerChecksForEvents: now,
			}
			output := doRun(ts, testInjectables, testDef)
			assertCreatedEventsWereNotScheduled(testDef, output)
		})

		Convey("Scheduler should queue events if event started within the relevance window", func() {
			testDef := &testCase{
				eventProps: []*eventProperties{
					{
						startTime:           now,
						whenChannelGoesLive: &now,
					},
				},
				timeSchedulerChecksForEvents: now.Add(offsetStartMustAtOrAfter),
			}
			output := doRun(ts, testInjectables, testDef)
			assertCreatedEventsWereScheduled(testDef, output)
		})

		Convey("Scheduler should NOT queue events if event started past the relevance window", func() {
			testDef := &testCase{
				eventProps: []*eventProperties{
					{
						startTime:           now,
						whenChannelGoesLive: &now,
					},
				},
				timeSchedulerChecksForEvents: now.Add(offsetStartMustAtOrAfter).Add(time.Second),
			}
			output := doRun(ts, testInjectables, testDef)
			assertCreatedEventsWereNotScheduled(testDef, output)
		})

		Convey("Scheduler should NOT queue events that have ended", func() {
			eventEndTime := now.Add(offsetStartMustAtOrAfter)
			testDef := &testCase{
				eventProps: []*eventProperties{
					{
						startTime:           now,
						endTime:             &eventEndTime,
						whenChannelGoesLive: &now,
					},
				},
				timeSchedulerChecksForEvents: eventEndTime.Add(time.Second),
			}
			output := doRun(ts, testInjectables, testDef)
			assertCreatedEventsWereNotScheduled(testDef, output)
		})

		Convey("Scheduler should NOT queue started events that are NOT live", func() {
			timeChannelGoesLive := now.Add(offsetStartMustAtOrAfter)
			testDef := &testCase{
				eventProps: []*eventProperties{
					{
						startTime:           now,
						whenChannelGoesLive: &timeChannelGoesLive,
					},
				},
				timeSchedulerChecksForEvents: now,
			}
			output := doRun(ts, testInjectables, testDef)
			assertCreatedEventsWereNotScheduled(testDef, output)

			Convey("Scheduler should queue the event if it becomes live in the future", func() {
				prevCreatedEvents := output.createdEventIDs
				testDef := &testCase{
					timeSchedulerChecksForEvents: timeChannelGoesLive,
				}
				output := doRun(ts, testInjectables, testDef)

				So(output.scheduledEvents, ShouldNotContainDuplicates)
				for _, eventID := range prevCreatedEvents {
					So(output.scheduledEvents, ShouldContain, eventID)
				}
			})
		})

		Convey("Multiple invocations should not result in duplicates", func() {
			testDef := &testCase{
				eventProps: []*eventProperties{
					{
						startTime:           now,
						whenChannelGoesLive: &now,
					},
					{
						startTime:           now,
						whenChannelGoesLive: &now,
					},
					{
						startTime:           now,
						whenChannelGoesLive: &now,
					},
				},
				timeSchedulerChecksForEvents: now,
				numInvocations:               2,
			}
			output := doRun(ts, testInjectables, testDef)
			assertCreatedEventsWereScheduled(testDef, output)
		})

		Convey("Scheduler should be schedule more events than are returned in a single database call", func() {
			numEvents := dbLimit*3 + 1
			eventProps := make([]*eventProperties, 0, numEvents)
			for i := 0; i < numEvents; i++ {
				eventProps = append(eventProps, &eventProperties{
					startTime:           now,
					whenChannelGoesLive: &now,
				})
			}

			testDef := &testCase{
				eventProps:                   eventProps,
				timeSchedulerChecksForEvents: now,
			}
			output := doRun(ts, testInjectables, testDef)
			assertCreatedEventsWereScheduled(testDef, output)
		})
	})
}

func assertCreatedEventsWereNotScheduled(testDef *testCase, output *testOutput) {
	So(output.createdEventIDs, ShouldHaveLength, len(testDef.eventProps))
	So(output.scheduledEvents, ShouldNotContainDuplicates)

	for _, eventID := range output.createdEventIDs {
		So(output.scheduledEvents, ShouldNotContain, eventID)
	}
}

func assertCreatedEventsWereScheduled(testDef *testCase, output *testOutput) {
	So(output.createdEventIDs, ShouldHaveLength, len(testDef.eventProps))
	So(output.scheduledEvents, ShouldNotContainDuplicates)

	for _, eventID := range output.createdEventIDs {
		So(output.scheduledEvents, ShouldContain, eventID)
	}
}

// ShouldNotContainDuplicates receives exactly one parameter, a slice of strings.
func ShouldNotContainDuplicates(actual interface{}, expected ...interface{}) string {
	if actual == nil {
		return ""
	}
	strSlice := actual.([]string)

	duplicates := make([]string, 0)
	strSet := make(map[string]struct{}, len(strSlice))
	for _, s := range strSlice {
		if _, ok := strSet[s]; ok {
			duplicates = append(duplicates, s)
		}

		strSet[s] = struct{}{}
	}

	if len(duplicates) > 0 {
		return "Expected that slice would not contain duplicates (but it did)!  Duplicate items: " + strings.Join(duplicates, ", ")
	}
	return ""
}

// testInjectables encapsulates dependencies that are injected into the Scheduler to mock services and to help inspect
// what the Scheduler is doing.
type testInjectables struct {
	jaxClient             *stubJaxClient
	hypemanClient         *stubHypemanClient
	schedulerTimeProvider *timeProvider
	wg                    *sync.WaitGroup
}

func newTestInjectables() *testInjectables {
	tp := &timeProvider{}
	return &testInjectables{
		jaxClient: &stubJaxClient{
			timeProvider: tp,
		},
		hypemanClient:         &stubHypemanClient{},
		schedulerTimeProvider: tp,
		wg:                    &sync.WaitGroup{},
	}
}

type testCase struct {
	// eventProps define the events that should be created for a test run, and when their channels go live.
	eventProps []*eventProperties

	// timeSchedulerChecksForEvents defines what the Scheduler thinks the current time is.  This affects
	// the window that events have to be within to be considered relevant, and also whether an event's
	// channel is live.
	timeSchedulerChecksForEvents time.Time

	// numInvocations is an optional param that defines the number of times the Scheduler should create
	// notification jobs.
	numInvocations int
}

// eventProperties describe an event that should be created for a test run.
type eventProperties struct {
	startTime time.Time

	// endTime is an optional event end time.  If not specified, the end time is derived from startTime and
	// defaultEventLength.
	endTime *time.Time

	// whenChannelGoesLive is an optional time that describes when an event's channel goes live.  If left
	// unspecified, it means the channel never goes live.
	whenChannelGoesLive *time.Time
}

// testOutput encapsulates the results of a test run.
type testOutput struct {
	createdEventIDs []string
	scheduledEvents []string
}

// doRun performs a scheduler test run.  It configures the injectables based on the test params, creates a set of
// events, invokes the scheduler, waits for it to complete, and returns the events that the scheduler queues.
func doRun(ts *testSetup, injectables *testInjectables, tc *testCase) *testOutput {
	output := &testOutput{}

	// Empty the queue of notification jobs that may have been populated by other tests.
	injectables.hypemanClient.clear()

	// Create the events and configure when jax will report each event as being live.
	channelIDToLiveTime := make(map[string]time.Time, len(tc.eventProps))
	for _, arg := range tc.eventProps {
		event := addSingleEventForSchedulerTest(ts, arg)
		logTest("eventID: %v", event.ID)
		logTest("ChannelID: %v", event.ChannelID)
		logTest("StartTime: %v", event.StartTime)
		logTest("EndTime: %v", event.EndTime)

		output.createdEventIDs = append(output.createdEventIDs, event.ID)

		// Set when each channel goes live.
		if arg.whenChannelGoesLive != nil {
			channelIDToLiveTime[event.ChannelID] = *arg.whenChannelGoesLive
			logTest("LiveTime: %v", *arg.whenChannelGoesLive)
		}
	}

	injectables.jaxClient.addWhenChannelsGoLive(channelIDToLiveTime)

	// Set what the scheduler will think the current time is, which is used to determine the
	// events that it thinks are starting.
	injectables.schedulerTimeProvider.setTime(tc.timeSchedulerChecksForEvents)

	// Trigger creating notification jobs.
	numInvocations := tc.numInvocations
	if numInvocations == 0 {
		numInvocations = 1
	}
	for i := 0; i < numInvocations; i++ {
		triggerCreateNotificationJobs(ts)
		injectables.wg.Add(1)
	}
	injectables.wg.Wait()

	// Get the events that notification jobs were created for.
	output.scheduledEvents = injectables.hypemanClient.getEventIDs()

	return output
}

func triggerCreateNotificationJobs(ts *testSetup) {
	err := request(ts, "POST", "/v1/create_notification_jobs", nil, nil)
	So(err, ShouldBeNil)
}

func addSingleEventForSchedulerTest(ts *testSetup, args *eventProperties) *geaclient.SingleEvent {
	endTime := args.startTime.Add(defaultEventLength)
	if args.endTime != nil {
		endTime = *args.endTime
	}

	channelID := timestampUser()
	title := "TestIntegration_Single title " + timestamp()
	description := "TestIntegration_Single description" + timestamp()

	event, err := ts.client.CreateSingleEvent(ts.ctx, geaclient.CreateSingleEventParams{
		OwnerID:     channelID,
		StartTime:   args.startTime,
		EndTime:     endTime,
		Language:    languageEN,
		Title:       title,
		Description: description,
		ChannelID:   channelID,
		GameID:      overwatchGameID,
	}, channelID, nil)

	So(err, ShouldBeNil)
	So(event, ShouldNotBeNil)

	return event
}

// stubJaxClient allows our tests to set when an event's channel goes live, and to check which channels are live based
// on this info.
type stubJaxClient struct {
	channelIDToLiveTime map[string]time.Time
	timeProvider        *timeProvider
	rwMutex             sync.RWMutex
}

var _ jax.Client = &stubJaxClient{}

// GetStreamsByChannelIDs is the call that the Scheduler makes to check whether channels are live.
func (c *stubJaxClient) GetStreamsByChannelIDs(ctx net_context.Context, channelIDs []string, opts *jaxclient.StreamsOptions, reqOpts *twitchclient.ReqOpts) (*jaxclient.Streams, error) {
	c.rwMutex.RLock()
	defer c.rwMutex.RUnlock()

	streams := make([]*jaxclient.Stream, 0, len(channelIDs))
	for _, id := range channelIDs {
		if !c.isChannelLive(id) {
			logTest("channel %s was not live", id)
			continue
		}

		idInt, err := strconv.Atoi(id)
		if err != nil {
			return nil, err
		}

		stream := &jaxclient.Stream{
			Channel: "channel_" + id,
			Properties: jaxclient.StreamProperties{
				Rails: &jaxclient.StreamRailsProperties{
					ChannelID: &idInt,
				},
			},
		}

		streams = append(streams, stream)
	}

	return &jaxclient.Streams{
		TotalCount: len(streams),
		Hits:       streams,
	}, nil
}

func (c *stubJaxClient) isChannelLive(channelID string) bool {
	liveTime, ok := c.channelIDToLiveTime[channelID]
	if !ok {
		return false
	}

	now := c.timeProvider.getTime()
	return !liveTime.After(now)
}

// addWhenChannelsGoLive allows our tests to specify when a set of channels go live.
func (c *stubJaxClient) addWhenChannelsGoLive(channelIDToLiveTime map[string]time.Time) {
	c.rwMutex.Lock()
	defer c.rwMutex.Unlock()

	if c.channelIDToLiveTime == nil {
		c.channelIDToLiveTime = make(map[string]time.Time, len(channelIDToLiveTime))
	}
	for k, v := range channelIDToLiveTime {
		c.channelIDToLiveTime[k] = v
	}
}

// stubHypemanClient allows our tests to determine the events that the scheduler creates notification jobs for.
type stubHypemanClient struct {
	eventIDs []string
	rwMutex  sync.RWMutex
}

var _ hypeman.Client = &stubHypemanClient{}

// AddJobs is the method that the Scheduler calls to queue notification jobs to SQS.
func (c *stubHypemanClient) AddJobs(ctx context.Context, eventIDs []string) error {
	c.rwMutex.Lock()
	defer c.rwMutex.Unlock()

	c.eventIDs = append(c.eventIDs, eventIDs...)
	return nil
}

func (c *stubHypemanClient) clear() {
	c.rwMutex.Lock()
	defer c.rwMutex.Unlock()

	c.eventIDs = make([]string, 0)
}

func (c *stubHypemanClient) getEventIDs() []string {
	c.rwMutex.RLock()
	defer c.rwMutex.RUnlock()

	eventIDs := make([]string, 0, len(c.eventIDs))
	return append(eventIDs, c.eventIDs...)
}

// timeProvider allows our tests to inject what the scheduler thinks the current time is.
type timeProvider struct {
	val     time.Time
	rwMutex sync.RWMutex
}

// getTime is the method that the scheduler calls to get the "current" time.
func (p *timeProvider) getTime() time.Time {
	p.rwMutex.RLock()
	defer p.rwMutex.RUnlock()

	return p.val
}

func (p *timeProvider) setTime(val time.Time) {
	p.rwMutex.Lock()
	defer p.rwMutex.Unlock()

	p.val = val
}

func logTest(format string, a ...interface{}) {
	// Comment out return to log values to the console.
	return

	fmt.Println(fmt.Sprintf(format, a...))
}
