// +build integration

package main

import (
	"testing"
	"time"

	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/hypeman-scheduler"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	. "github.com/smartystreets/goconvey/convey"
)

func TestJobStatusClient(t *testing.T) {
	Convey("jobStatusClient", t, func() {
		injectables := newDefaultInjectables()
		ts := startServer(t, injectables)
		if ts == nil {
			t.Error("Unable to setup testing server")
			return
		}

		dc := ts.thisInstance.serviceCommon.Config
		clientConfig := hypemanscheduler.JobStatusClientConfig{}
		err := clientConfig.Load(dc)
		So(err, ShouldBeNil)

		session, awsConf := service_common.CreateAWSSession(dc)
		dynamoClient := dynamodb.New(session, awsConf...)

		client := hypemanscheduler.JobStatusClient{
			Dynamo: dynamoClient,
			Config: &clientConfig,
			Log:    &log.ElevatedLog{},
		}

		Convey("addJob", func() {
			So(ts.Setup(), ShouldBeNil)
			ctx := ts.ctx
			eventID := db.NewID()

			Convey("should store a job status when given an event that does not have a notifications job", func() {
				added, err := client.AddJob(ctx, eventID)
				So(err, ShouldBeNil)
				So(added, ShouldBeTrue)

				jobStatus, err := client.GetJobStatus(ctx, eventID)
				So(err, ShouldBeNil)
				So(jobStatus, ShouldNotBeNil)
				So(jobStatus.EventID, ShouldEqual, eventID)
				So(jobStatus.ExpiresAt, ShouldNotEqual, time.Time{})
			})

			Convey("should NOT store a job status when given an event that already has a notifications job", func() {
				_, err := client.AddJob(ctx, eventID)
				So(err, ShouldBeNil)

				added, err := client.AddJob(ctx, eventID)
				So(err, ShouldBeNil)
				So(added, ShouldBeFalse)
			})
		})

		Convey("needsJobs should return events that do NOT have notifications jobs", func() {
			So(ts.Setup(), ShouldBeNil)
			ctx := ts.ctx
			// Given an event that has a notifications job,
			eventID1 := db.NewID()
			_, err := client.AddJob(ctx, eventID1)
			So(err, ShouldBeNil)

			// and an event that does not have a notifications job,
			eventID2 := db.NewID()

			// needsJobs should return the event that does not have a notifications job.
			eventsThatNeedJobs, err := client.NeedsJobs(ctx, []string{eventID1, eventID2})
			So(err, ShouldBeNil)
			So(eventsThatNeedJobs, ShouldHaveLength, 1)
			So(eventsThatNeedJobs[0], ShouldEqual, eventID2)
		})
	})
}
