// +build integration

package main

import (
	"context"
	"encoding/json"
	"fmt"
	"sync"
	"testing"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/gea/internal/db"
	hypemanworker "code.justin.tv/twitch-events/gea/internal/hypeman-worker"
	"code.justin.tv/twitch-events/gea/internal/images"
	testutils "code.justin.tv/twitch-events/gea/internal/test-utils"
	. "code.justin.tv/twitch-events/gea/internal/test-utils/assertions"
	"code.justin.tv/twitch-events/gea/lib/geaclient"
	"github.com/satori/go.uuid"
	. "github.com/smartystreets/goconvey/convey"
)

const seoulTimeZoneID = "Asia/Seoul"

func TestWorker(t *testing.T) {
	snsClient := &stubNotificationSNSClient{}
	clock := &testutils.StubClock{}
	injectables := newDefaultInjectables()
	injectables.notificationsSNSClient = snsClient
	injectables.clock = clock

	ts := startServer(t, injectables, map[string][]byte{
		// Lower the limit used when getting followers so the tests exercise the paging logic.
		"gea.hypeman_worker.get_followers_limit": []byte("2"),

		// Set the image URL templates so that the tests can verify that we populate the suffix properly.
		"gea.images.gea_template":     []byte("https://static-cdn.jtvnw.net/twitch-event-images-dev-v2/%s-{width}x{height}"),
		"gea.images.default_template": []byte("https://static-cdn.jtvnw.net/twitch-event-images-dev-v2/default/%s-{width}x{height}"),
	})
	if ts == nil {
		t.Fatalf("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		ctx := ts.ctx
		Reset(snsClient.reset)

		Convey("Hypeman worker send a notification job to the SNS topic for every user that subscribed", func() {
			doTestWorkerRun(ctx, ts, snsClient, 3)
		})

		Convey("Hypeman worker NOT send any notification jobs to SNS if there are no subscriber", func() {
			doTestWorkerRun(ctx, ts, snsClient, 0)
		})

		Convey("Hypeman worker sends notifications for Segment events", func() {
			Reset(snsClient.reset)
			geaClient := ts.client
			dbClient := ts.thisInstance.oracleDB

			// Create a parent event and a child event that takes place in Korea.
			ownerID := timestampUser()
			timetableEventParams := timetableParams(ownerID)
			timetableEventParams.TimeZoneID = seoulTimeZoneID
			parentEvent, err := geaClient.CreateTimetableEvent(ctx, timetableEventParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(parentEvent, ShouldNotBeNil)
			parentID := parentEvent.ID

			startTime := time.Now().UTC()
			segmentEventParams := segmentParams(parentID, ownerID, overwatchGameID, startTime, startTime.Add(time.Hour))
			segmentEvent, err := geaClient.CreateSegmentEvent(ctx, segmentEventParams, ownerID, nil)
			So(err, ShouldBeNil)
			So(segmentEvent, ShouldNotBeNil)

			// Have users follow each event.
			timetableFollower := timestampUser()
			err = geaClient.FollowEvent(ctx, parentEvent.ID, timetableFollower, nil)
			So(err, ShouldBeNil)

			segment1Follower := timestampUser()
			err = geaClient.FollowEvent(ctx, segmentEvent.ID, segment1Follower, nil)
			So(err, ShouldBeNil)

			Convey("When the first segment of a timetable starts, "+
				"notifications should be sent to both the timetable's and segment's followers", func() {

				// Let's say that a segment event goes live at some arbitrary time.
				// UTC:        01/31/2018 11:30 PM
				// Korea time: 02/01/2018  8:30 AM
				firstEventLiveTime := time.Date(2018, 1, 31, 23, 30, 0, 0, time.UTC)
				clock.SetNow(firstEventLiveTime)

				expectedUsers := []string{segment1Follower, timetableFollower}
				snsClient.setExpectedNumberOfCallsToPublish(len(expectedUsers))

				err := triggerProcessNotificationJobs(ts, segmentEvent.ID)
				So(err, ShouldBeNil)

				// Wait until hypeman worker publishes to SNS for each follower.
				err = snsClient.waitUntilCompleteOrTimeout(time.Minute)
				So(err, ShouldBeNil)

				// Verify that the messages have the expected properties.
				messages := snsClient.getMessages()

				dbEvent, err := dbClient.GetEvent(ctx, segmentEvent.ID, false)
				So(err, ShouldBeNil)
				So(dbEvent, ShouldNotBeNil)
				assertThatMessagesArePopulatedWithDBEventData(dbEvent, messages)
				assertThatMessagesSentToExpectedUsers(messages, expectedUsers)

				Convey("When a segment starts, but notifications have already been sent for its timetable that day, "+
					"notifications should be sent only the segment's followers", func() {
					snsClient.reset()

					// Let's say that another segment event goes live after the first on the same day using Korean time,
					// but different on a different day if we were using UTC.  (We choose these times to verify that
					// the worker uses the event's timezone when determining if notifications should be sent.)
					// UTC:        02/01/2018 12:30 AM
					// Korea time: 02/01/2018  9:30 AM
					secondEventLiveTime := time.Date(2018, 2, 1, 0, 30, 0, 0, time.UTC)
					clock.SetNow(secondEventLiveTime)

					expectedUsers := []string{segment1Follower}
					snsClient.setExpectedNumberOfCallsToPublish(len(expectedUsers))

					err := triggerProcessNotificationJobs(ts, segmentEvent.ID)
					So(err, ShouldBeNil)

					// Wait until hypeman worker publishes to SNS for each follower.
					err = snsClient.waitUntilCompleteOrTimeout(time.Minute)
					So(err, ShouldBeNil)

					// Verify that the messages have the expected properties.
					messages := snsClient.getMessages()

					dbEvent, err := dbClient.GetEvent(ctx, segmentEvent.ID, false)
					So(err, ShouldBeNil)
					So(dbEvent, ShouldNotBeNil)
					assertThatMessagesArePopulatedWithDBEventData(dbEvent, messages)
					assertThatMessagesSentToExpectedUsers(messages, expectedUsers)
				})

				Convey("When a segment starts, and no notifications have been sent for its timetable that day "+
					"notifications should be sent to both the timetable's and segment's followers", func() {
					snsClient.reset()

					// Let's say that another segment event goes live after the first on a different day.
					// UTC:        02/01/2018  9:30 PM
					// Korea time: 02/02/2018  6:30 AM
					secondEventLiveTime := time.Date(2018, 2, 1, 21, 30, 0, 0, time.UTC)
					clock.SetNow(secondEventLiveTime)

					expectedUsers := []string{segment1Follower, timetableFollower}
					snsClient.setExpectedNumberOfCallsToPublish(len(expectedUsers))

					err := triggerProcessNotificationJobs(ts, segmentEvent.ID)
					So(err, ShouldBeNil)

					// Wait until hypeman worker publishes to SNS for each follower.
					err = snsClient.waitUntilCompleteOrTimeout(time.Minute)
					So(err, ShouldBeNil)

					// Verify that the messages have the expected properties.
					messages := snsClient.getMessages()

					dbEvent, err := dbClient.GetEvent(ctx, segmentEvent.ID, false)
					So(err, ShouldBeNil)
					So(dbEvent, ShouldNotBeNil)
					assertThatMessagesArePopulatedWithDBEventData(dbEvent, messages)
					assertThatMessagesSentToExpectedUsers(messages, expectedUsers)
				})
			})
		})
	})
}

// createEventUsingDBClient uses the db client to to create an event that is a child of another event.  Ideally we'd
// use GeaClient for this, but none of our events support child events yet.
func createEventUsingDBClient(ctx context.Context, ts *testSetup, parentID *string) (*db.Event, error) {
	dbClient := ts.thisInstance.oracleDB

	startTime := time.Now()
	endTime := startTime.Add(defaultEventLength)

	channelID := timestampUser()
	title := "TestIntegration_Single title " + timestamp()
	description := "TestIntegration_Single description" + timestamp()

	txCtx, createdTx, err := dbClient.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer dbClient.RollbackTxIfNotCommitted(txCtx, createdTx)

	legacy := "legacy"
	event, err := dbClient.CreateEvent(ctx, &db.CreateDBEventParams{
		OwnerID:     channelID,
		Type:        "single",
		ParentID:    parentID,
		StartTime:   &startTime,
		EndTime:     &endTime,
		Language:    &legacy,
		Title:       &title,
		Description: &description,
		ChannelID:   &channelID,
		GameID:      &overwatchGameID,
		CoverImageID: &images.ImageID{
			Type: "gea",
			ID:   uuid.NewV4().String(),
		},
	})
	if err != nil {
		return nil, errors.Wrap(err, "error creating event")
	}

	err = dbClient.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}

	return event, nil
}

func assertThatMessagesArePopulatedWithDBEventData(event *db.Event, messages []*hypemanworker.NotificationMessage) {
	for _, msg := range messages {
		So(msg, ShouldNotBeNil)

		So(msg.EventID, ShouldEqual, event.ID)
		So(msg.ChannelID, ShouldEqual, *event.ChannelID)
		So(msg.EventTitle, ShouldEqual, *event.Title)

		So(msg.EventDescription, ShouldNotBeNil)
		So(*msg.EventDescription, ShouldEqual, *event.Description)

		So(msg.EventGame, ShouldNotBeBlank)
		So(msg.EventURL, ShouldBeFullHTTPURL)
		So(msg.EventImage, ShouldNotBeNil)
		So(*msg.EventImage, ShouldBeFullHTTPURL)
	}
}

func assertThatMessagesSentToExpectedUsers(messages []*hypemanworker.NotificationMessage, expectedUserIDs []string) {
	sentUserIDs := make([]string, 0, len(messages))
	for _, msg := range messages {
		sentUserIDs = append(sentUserIDs, msg.UserID)
	}

	So(sentUserIDs, ShouldHaveLength, len(expectedUserIDs))
	So(sentUserIDs, ShouldNotContainDuplicates)

	for _, expectedUserID := range expectedUserIDs {
		So(expectedUserID, ShouldBeIn, sentUserIDs)
	}
}

func doTestWorkerRun(ctx context.Context, ts *testSetup, snsClient *stubNotificationSNSClient, numFollowers int) {
	snsClient.setExpectedNumberOfCallsToPublish(numFollowers)
	geaClient := ts.client

	// Add an event.
	event := addSingleEventForWorkerTest(ctx, geaClient)
	eventID := event.ID

	// Have a set of users follow the event.
	userIDs := make([]string, 0, numFollowers)
	for i := 0; i < numFollowers; i++ {
		userID := timestampUser()
		userIDs = append(userIDs, userID)

		err := geaClient.FollowEvent(ctx, eventID, userID, nil)
		So(err, ShouldBeNil)
	}

	// Make hypeman worker process notification jobs for the event we created,
	err := triggerProcessNotificationJobs(ts, eventID)
	So(err, ShouldBeNil)

	// Wait until hypeman worker publishes to SNS for each follower.
	err = snsClient.waitUntilCompleteOrTimeout(time.Minute)
	So(err, ShouldBeNil)

	// Verify that the messages have the expected properties.
	messages := snsClient.getMessages()
	So(messages, ShouldHaveLength, numFollowers)

	for _, msg := range messages {
		So(msg, ShouldNotBeNil)
		So(msg.EventID, ShouldEqual, eventID)
		So(msg.ChannelID, ShouldEqual, event.ChannelID)
		So(msg.EventDescription, ShouldNotBeNil)
		So(*msg.EventDescription, ShouldEqual, event.Description)
		So(msg.EventGame, ShouldNotBeBlank)
		So(msg.EventImage, ShouldNotBeNil)
		So(*msg.EventImage, ShouldBeFullHTTPURL)
		So(msg.EventTime, ShouldNotBeBlank)
		So(msg.EventTitle, ShouldEqual, event.Title)
		So(msg.EventURL, ShouldBeFullHTTPURL)

		b, err := json.MarshalIndent(msg, "", "  ")
		So(err, ShouldBeNil)
		logTest(string(b))
	}

	assertThatMessagesSentToExpectedUsers(messages, userIDs)
}

func addSingleEventForWorkerTest(ctx context.Context, geaClient geaclient.Client) *geaclient.SingleEvent {
	startTime := time.Now()
	endTime := startTime.Add(defaultEventLength)

	channelID := timestampUser()
	title := "TestIntegration_Single title " + timestamp()
	description := "TestIntegration_Single description" + timestamp()

	event, err := geaClient.CreateSingleEvent(ctx, geaclient.CreateSingleEventParams{
		OwnerID: channelID,

		StartTime:   startTime,
		EndTime:     endTime,
		Language:    languageEN,
		Title:       title,
		Description: description,
		ChannelID:   channelID,
		GameID:      overwatchGameID,
	}, channelID, nil)

	So(err, ShouldBeNil)
	So(event, ShouldNotBeNil)

	return event
}

func triggerProcessNotificationJobs(ts *testSetup, eventID string) error {
	url := fmt.Sprintf("/test/process_notification_job?event_id=%s", eventID)
	return request(ts, "POST", url, nil, nil)
}

// stubNotificationSNSClient allows our tests to inspect that notification jobs that the worker sends to the SNS
// topic.
type stubNotificationSNSClient struct {
	messages        []*hypemanworker.NotificationMessage
	messagesRWMutex sync.RWMutex

	wg sync.WaitGroup
}

var _ hypemanworker.NotificationsSNSClient = &stubNotificationSNSClient{}

func (c *stubNotificationSNSClient) Publish(ctx context.Context, notificationMsg *hypemanworker.NotificationMessage) error {
	c.messagesRWMutex.Lock()
	c.messages = append(c.messages, notificationMsg)
	c.messagesRWMutex.Unlock()

	c.wg.Done()
	return nil
}

func (c *stubNotificationSNSClient) getMessages() []*hypemanworker.NotificationMessage {
	c.messagesRWMutex.RLock()
	defer c.messagesRWMutex.RUnlock()

	messages := make([]*hypemanworker.NotificationMessage, 0, len(c.messages))
	return append(messages, c.messages...)
}

func (c *stubNotificationSNSClient) setExpectedNumberOfCallsToPublish(numCalls int) {
	c.wg.Add(numCalls)
}

func (c *stubNotificationSNSClient) waitUntilCompleteOrTimeout(timeout time.Duration) error {
	done := make(chan struct{})

	go func() {
		c.wg.Wait()
		done <- struct{}{}
	}()

	select {
	case <-done:
	case <-time.After(timeout):
		return errors.New("waiting on wait group timed out")
	}

	return nil
}

func (c *stubNotificationSNSClient) reset() {
	c.messagesRWMutex.Lock()
	c.messages = []*hypemanworker.NotificationMessage{}
	c.messagesRWMutex.Unlock()
}
