package cache

import (
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/foundation/gomemcache/memcache"
)

type Config struct {
	ConfigServer     *distconf.Str
	PollInterval     *distconf.Duration
	ReadWriteTimeout *distconf.Duration
	MaxIdleConns     *distconf.Int
}

func (c *Config) Load(dconf *distconf.Distconf) error {
	c.ConfigServer = dconf.Str("gea.cache.config_server", "")
	if c.ConfigServer.Get() == "" {
		return errors.New("unable to find cache config server")
	}
	c.PollInterval = dconf.Duration("gea.cache.poll_interval", time.Minute)
	c.ReadWriteTimeout = dconf.Duration("gea.cache.rw_timeout", 350*time.Millisecond)
	c.MaxIdleConns = dconf.Int("gea.cache.idle_conn_count", 1000)
	return nil
}

type Service struct {
	*memcache.Client
	Config       *Config
	onClose      chan struct{}
	stopPollFunc func()
}

func (s *Service) Setup() error {
	s.Client, s.stopPollFunc = memcache.Elasticache(s.Config.ConfigServer.Get(), s.Config.PollInterval.Get())
	s.Client.Timeout = s.Config.ReadWriteTimeout.Get()
	s.Client.MaxIdleConns(int(s.Config.MaxIdleConns.Get()))
	s.onClose = make(chan struct{})
	return nil
}

func (s *Service) Start() error {
	// Wait for close signal
	<-s.onClose
	return nil
}

func (s *Service) Close() error {
	s.stopPollFunc()
	close(s.onClose)
	return nil
}
