package channelevent

import (
	"context"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/gomemcache/memcache"
	"code.justin.tv/twitch-events/gea/internal/cache"
)

// CacheVersion should be updated whenever we make breaking changes to how we use this cache
const CacheVersion = "v1:"

// CacheConfig contains configurable timeouts for cached channelIDs
type CacheConfig struct {
	ChannelEventEnqueueCacheDuration *distconf.Duration
	ChannelEventMessageCacheDuration *distconf.Duration
}

// Load links config values to distconf
func (c *CacheConfig) Load(dconf *distconf.Distconf) error {
	c.ChannelEventEnqueueCacheDuration = dconf.Duration("gea.channel_event_enqueue_cache_duration", 45*time.Second)
	c.ChannelEventMessageCacheDuration = dconf.Duration("gea.channel_event_message_cache_duration", 5*time.Minute)
	return nil
}

// Cache represents a decorated cache with logging, stats, and conf variables
type Cache struct {
	Config *CacheConfig
	Cache  *cache.Service
	Stats  *service_common.StatSender
	Log    *log.ElevatedLog
}

func (c *Cache) channelKey(channelID string) string {
	return "gea:ce:" + CacheVersion + channelID
}

// AddChannelIfNotThrottled executes an atomic Add operation, caching the ID if it does not already exist
// and returning an error if it does
func (c *Cache) AddChannelIfNotThrottled(ctx context.Context, channelID string) error {
	key := c.channelKey(channelID)
	cacheErr := c.Cache.Add(ctx, &memcache.Item{
		Key:        key,
		Expiration: int32(c.Config.ChannelEventEnqueueCacheDuration.Get().Seconds()),
	})
	if cacheErr != nil {
		if cacheErr != memcache.ErrNotStored {
			c.Log.LogCtx(ctx, cacheErr)
		}
	}
	// cacheErr == nil if item DNE in cache and was just written
	// else if cacheErr == ErrNotStored, item exists in cache
	// else if cacheErr != nil, failure occurred
	return cacheErr
}

// SetChannel caches the given channel ID.  This is meant to prevent additional messages being sent for the
// channel until the TTL expires.
func (c *Cache) SetChannel(ctx context.Context, channelID string) error {
	key := c.channelKey(channelID)
	return c.Cache.Set(ctx, &memcache.Item{
		Key:        key,
		Expiration: int32(c.Config.ChannelEventMessageCacheDuration.Get().Seconds()),
	})
}
