package channelevent

import (
	"context"
	"strconv"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/twitch-events/gea/internal/clock"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/types"
	"code.justin.tv/web/users-service/client/channels"
	usersmodels "code.justin.tv/web/users-service/models"
)

// liveEventTypes contains the types of events that we report as being live on a channel.
//
// Premiere events are not included in this list because Premieres has its own live Premiere functionality,
// and reporting Premiere events as being live interferes with this functionality.  E.g. A channel that is broadcasting
// a premiere renders two banners - one stating that the video is premiere, and one stating that a premiere event
// is live.
var liveEventTypes = []string{types.EventTypeSingle, types.EventTypeSegment}

// GetChannelClient serves as an interface for mocking the channel client
type GetChannelClient interface {
	GetByIDAndParams(ctx context.Context, id string, params *usersmodels.ChannelFilterParams, reqOpts *twitchclient.ReqOpts) (*channels.Channel, error)
}

// LiveEventLoader encapsulates the logic for loading a channel's current event.  This allows the logic to be
// shared by the get_live_event handler and the ChannelEventPublisher sqs queue processor.
type LiveEventLoader struct {
	EventHandlers    *types.EventHandlers
	OracleDB         db.DB
	Clock            clock.Clock
	GetChannelClient GetChannelClient
}

func (s *LiveEventLoader) GetLiveEvent(ctx context.Context, channelID string, getDeleted, skipCache bool) (types.TypedEvent, error) {
	now := s.Clock.NowUTC()

	currentGameID, err := s.getCurrentGameID(ctx, channelID)
	if err != nil || currentGameID == nil {
		return nil, err
	}

	filter := &db.BroadcastFilter{
		ChannelIDs: []string{channelID},
		GameIDs:    []string{strconv.Itoa(*currentGameID)},
		Types:      liveEventTypes,
		StartTimeWindow: &db.TimeWindow{
			End: &now,
		},
		EndTimeWindow: &db.TimeWindow{
			Start: &now,
		},
	}

	eventIDs, err := s.OracleDB.GetEventIDsSortedByStartTime(ctx, filter, true, "", 10)
	if err != nil || len(eventIDs.EventIDs) == 0 {
		return nil, err
	}

	events, err := s.EventHandlers.GetEvents(ctx, eventIDs.EventIDs, getDeleted, skipCache)
	if err != nil || len(events) == 0 {
		return nil, err
	}

	return events[0], nil
}

func (s *LiveEventLoader) getCurrentGameID(ctx context.Context, channelID string) (*int, error) {
	channelProps, err := s.GetChannelClient.GetByIDAndParams(ctx, channelID, nil, nil)
	if err != nil {
		return nil, errors.Wrapf(err, "loading channel properties for channel ID, \"%s\" failed", channelID)
	} else if channelProps == nil {
		return nil, nil
	}

	gameID := channelProps.GameID
	return &gameID, nil
}
