package channelevent

import (
	"context"
	"encoding/json"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/service-common/feedsqs"
	"code.justin.tv/twitch-events/gea/internal/pubsub"
	"code.justin.tv/twitch-events/gea/internal/types"
	"github.com/aws/aws-sdk-go/service/sqs"
	net_context "golang.org/x/net/context"
)

type PublisherConfig struct {
	feedsqs.SQSQueueProcessorConfig
}

// Load sets up PublisherConfig to read configuration values from the given distconf.
func (c *PublisherConfig) Load(dconf *distconf.Distconf) error {
	return c.SQSQueueProcessorConfig.Verify(dconf, "gea.channel_event", time.Second*3, 2, time.Minute*15)
}

type Publisher struct {
	feedsqs.SQSQueueProcessor
	Config *PublisherConfig

	EventHandlers   *types.EventHandlers
	PubsubClient    pubsub.Client
	Throttler       *Cache
	LiveEventLoader *LiveEventLoader
}

func (w *Publisher) Setup() error {
	w.SQSQueueProcessor.ProcessMessage = w.processMessageWithStats
	return w.SQSQueueProcessor.Setup()
}

func (w *Publisher) processMessageWithStats(ctx net_context.Context, m *sqs.Message) error {
	defer w.recordDuration("process_message.duration", time.Now())
	w.Stats.IncC("process_message.start", 1, 1)

	err := w.processMessage(m)
	if err != nil {
		w.Stats.IncC("process_message.failed", 1, 1)
		return err
	}

	w.Stats.IncC("process_message.succeeded", 1, 1)
	return nil
}

func (w *Publisher) processMessage(m *sqs.Message) error {
	if m.Body == nil {
		return errors.New("received empty body message")
	}

	msg := &SQSMessage{}
	if err := json.NewDecoder(strings.NewReader(*m.Body)).Decode(msg); err != nil {
		return errors.Wrap(err, "unable to parse SQS message body")
	}

	return w.PublishChannelEvent(context.Background(), msg)
}

func (w *Publisher) PublishChannelEvent(ctx context.Context, msg *SQSMessage) error {
	if msg.Version != SQSVersion {
		return errors.Errorf("expected message version, \"%s\", but received \"%s\"", SQSVersion, msg.Version)
	}

	channelID := msg.ChannelID
	if channelID == "" {
		return errors.New("channelID was empty")
	}

	currentEvent, err := w.getCurrentChannelEvent(ctx, channelID)
	if err != nil {
		return err
	}

	err = w.PubsubClient.PublishCurrentChannelEvent(ctx, channelID, currentEvent)
	if err != nil {
		return err
	}

	// Throttle this channel so that we don't send another update for it till the TTL (5 min) expires.
	return w.Throttler.SetChannel(ctx, channelID)
}

func (w *Publisher) getCurrentChannelEvent(ctx context.Context, channelID string) (*pubsub.CurrentEvent, error) {
	liveEvent, err := w.LiveEventLoader.GetLiveEvent(ctx, channelID, false, false)
	if err != nil || liveEvent == nil {
		return nil, err
	}

	parentID := liveEvent.GetParentID()
	parentEvent, err := w.EventHandlers.GetEvent(ctx, parentID, false, false)
	if err != nil {
		return nil, err
	}

	currentEvent := w.convertToCurrentEvent(liveEvent, parentEvent)
	return currentEvent, nil
}

func (w *Publisher) convertToCurrentEvent(event, parentEvent types.TypedEvent) *pubsub.CurrentEvent {
	if event == nil {
		return nil
	}

	return &pubsub.CurrentEvent{
		EventID: event.GetID(),
		Type:    event.GetType(),
		Title:   event.GetTitle(),
		Parent:  w.convertToCurrentEvent(parentEvent, nil),
	}
}

func (w *Publisher) recordDuration(stat string, start time.Time) {
	w.Stats.TimingDurationC(stat, time.Since(start), 1.0)
}
