package channelevent

import (
	"context"
	"sync"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/types"

	"golang.org/x/sync/errgroup"
)

const (
	// DefaultTimeLowerBound is how far back in time we want to look for an event that has started or ended
	DefaultTimeLowerBound = time.Minute * 5
)

// SchedulerConfig contains config values for Scheduler.
type SchedulerConfig struct {
	TimeLowerBound *distconf.Duration
}

// Load initializes SchedulerConfig to read configuration values from the given distconf.
func (c *SchedulerConfig) Load(dconf *distconf.Distconf) error {
	c.TimeLowerBound = dconf.Duration("gea.channel_event_window_lower_bound", DefaultTimeLowerBound)
	return nil
}

// Scheduler facilitates queueing up notification jobs for events that are started.  These notification jobs are then
// fleshed out by hypeman worker and set to Pushy.
type Scheduler struct {
	Cache         *Cache
	Config        *SchedulerConfig
	EventHandlers *types.EventHandlers
	Log           *log.ElevatedLog
	OracleDB      db.DB
	SQS           SQSClient

	// NowFunc is an optional field that contains a function that returns the current time.  It is meant to
	// facilitate testing.
	NowFunc func() time.Time

	// CreateChannelEventUpdatesDoneFunc is an optional field that contains a function that is called at the end of
	// CreateChannelEventUpdates.  It is meant to allow test code to wait on CreateChannelEventUpdates.
	CreateChannelEventUpdatesDoneFunc func()
}

// CreateChannelEventUpdates identifies all channels that have had an event start or end within some time window
// and sends them off to be enqueued
func (s *Scheduler) CreateChannelEventUpdates(ctx context.Context) {
	err := s.innerCreateChannelEventUpdates(ctx)
	if err != nil {
		s.Log.LogCtx(ctx, err)
		return
	}

	if s.CreateChannelEventUpdatesDoneFunc != nil {
		s.CreateChannelEventUpdatesDoneFunc()
	}
}

// Enqueue the channel ids corresponding to all events that have started or ended recently
func (s *Scheduler) innerCreateChannelEventUpdates(ctx context.Context) error {
	now := s.now().UTC()

	timeWindowLowerBound := now.Add((-1) * s.Config.TimeLowerBound.Get())
	timeWindow := &db.TimeWindow{
		Start: &timeWindowLowerBound,
		End:   &now,
	}

	// Find all event ids for events that have either gone live or ended within the specified time window
	g, gCtx := errgroup.WithContext(ctx)
	g.Go(func() error {
		filter := &db.BroadcastFilter{
			StartTimeWindow: timeWindow,
			Types:           liveEventTypes,
		}
		return s.enqueueAllChannelIDsByBroadcastFilter(gCtx, filter, false)
	})
	g.Go(func() error {
		filter := &db.BroadcastFilter{
			EndTimeWindow: timeWindow,
			Types:         liveEventTypes,
		}
		return s.enqueueAllChannelIDsByBroadcastFilter(gCtx, filter, false)
	})
	return g.Wait()
}

// Get ids of all channels that own an event that is returned by the broadcast filter
// Then filter out all cached channel ids and enqueue the remainders
func (s *Scheduler) enqueueAllChannelIDsByBroadcastFilter(ctx context.Context, filter *db.BroadcastFilter, skipCache bool) error {
	cursor := ""
	for ok := true; ok; ok = (cursor != "") {
		result, err := s.OracleDB.GetEventIDsSortedByStartTime(ctx, filter, false, cursor, 100)
		if err != nil {
			return err
		}
		candidateEventIDs := result.EventIDs
		cursor = result.Cursor

		candidateEvents, err := s.EventHandlers.GetEvents(ctx, candidateEventIDs, false, false)
		if err != nil {
			return err
		}

		candidateChannelIDs := s.getCandidateChannelIDs(candidateEvents)
		s.EnqueueChannelIDs(ctx, candidateChannelIDs, skipCache)
	}
	return nil
}

// EnqueueChannelIDs takes in a list of channelIDs and checks if each channel has already been accounted for as "updated" (by pinging memcache) and if not, this
// will send the channelID to an SQS queue for further processing
func (s *Scheduler) EnqueueChannelIDs(ctx context.Context, candidateChannelIDs []string, skipCache bool) {
	// Filter out all channel ids that exist in our cache
	if !skipCache {
		candidateChannelIDs = s.filterCachedIDs(ctx, candidateChannelIDs)
	}

	// Enqueue each new channel id for which there has been a recently started or ended event
	wg := sync.WaitGroup{}
	for _, channelID := range candidateChannelIDs {
		wg.Add(1)
		go func(channelID string) {
			defer wg.Done()
			err := s.SQS.SendChannelToSQS(ctx, channelID)
			if err != nil {
				s.Log.LogCtx(ctx, err)
			}
		}(channelID)
	}
	wg.Wait()
}

func (s *Scheduler) filterCachedIDs(ctx context.Context, channelIDs []string) []string {
	filteredChannelIDs := make([]string, 0)
	for _, channelID := range channelIDs {
		err := s.Cache.AddChannelIfNotThrottled(ctx, channelID)
		if err == nil {
			// Item did not exist in cache and was thus added
			filteredChannelIDs = append(filteredChannelIDs, channelID)
		}
	}
	return filteredChannelIDs
}

func (s *Scheduler) getCandidateChannelIDs(candidateEvents []types.TypedEvent) []string {
	candidateChannelIDMap := make(map[string]bool)
	for _, event := range candidateEvents {
		if event == nil {
			continue
		}
		for _, channelID := range event.GetChannelIDs() {
			candidateChannelIDMap[channelID] = true
		}
	}
	candidateChannelIDs := make([]string, len(candidateChannelIDMap))
	index := 0
	for channelID := range candidateChannelIDMap {
		candidateChannelIDs[index] = channelID
		index++
	}
	return candidateChannelIDs
}

func (s *Scheduler) now() time.Time {
	if s.NowFunc != nil {
		return s.NowFunc()
	}

	return time.Now()
}
