package channelevent

import (
	"context"
	"encoding/json"

	"code.justin.tv/feeds/ctxlog/ctxlogaws"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
)

// SQSVersion should be updated whenever we make breaking changes to how we use this queue
const SQSVersion = "1"

type SQSClientConfig struct {
	ChannelEventQueueURL *distconf.Str
}

func (c *SQSClientConfig) Load(dconf *distconf.Distconf) error {
	c.ChannelEventQueueURL = dconf.Str("gea.channel_event.sqssource.queue_url", "")
	if c.ChannelEventQueueURL.Get() == "" {
		return errors.New("unable to find a URL for the channel event SQS queue")
	}

	return nil
}

// SQSClient sets up functions for mocking
type SQSClient interface {
	SendChannelToSQS(ctx context.Context, channelID string) error
}

var _ SQSClient = &SQSClientImpl{}

type SQSClientImpl struct {
	Log    *log.ElevatedLog
	Config *SQSClientConfig
	SQS    *sqs.SQS
}

// SQSMessage is our payload into the SQS queue
type SQSMessage struct {
	ChannelID string `json:"channel_id"`
	Version   string `json:"version"`
}

// SendChannelToSQS enqueues a message containing the channelID and the version number
func (s *SQSClientImpl) SendChannelToSQS(ctx context.Context, channelID string) error {
	channelEventStruct := SQSMessage{
		ChannelID: channelID,
		Version:   SQSVersion,
	}

	sqsMessage, err := json.Marshal(channelEventStruct)
	if err != nil {
		return err
	}

	message := &sqs.SendMessageInput{
		QueueUrl:    aws.String(s.Config.ChannelEventQueueURL.Get()),
		MessageBody: aws.String(string(sqsMessage)),
	}

	req, _ := s.SQS.SendMessageRequest(message)
	req.HTTPRequest = req.HTTPRequest.WithContext(ctx)
	return ctxlogaws.DoAWSSend(req, s.Log)
}
