package cursor

import (
	"encoding/base64"
	"encoding/json"

	"code.justin.tv/feeds/errors"
)

// Encode takes the address of a JSON serializable struct, and convert's the struct's values into a cursor string.
func Encode(v interface{}) (string, error) {
	b, err := json.Marshal(v)
	if err != nil {
		return "", errors.Wrap(err, "cursor encode error")
	}

	encoded := base64.StdEncoding.EncodeToString(b)
	return encoded, nil
}

// Decode takes a cursor string, and the address of an allocated JSON serializable struct, and populates the struct
// using the values encoded in the string.
func Decode(cursor string, into interface{}) error {
	if cursor == "" {
		return errors.New("cannot decode empty cursor")
	}

	b, err := base64.StdEncoding.DecodeString(cursor)
	if err != nil {
		return errors.Wrap(err, "cursor decode error")
	}

	err = json.Unmarshal(b, into)
	if err != nil {
		return errors.Wrap(err, "cursor decode error")
	}

	return nil
}

// DecodeVersioned takes a cursor string, and the address of an allocated JSON serializable struct, populates the struct
// using the values encoded in the string, and verifies that the version encoded in the string matches the expected.
// DecodeVersioned returns 'true' is the version check failed, and 'false' otherwise.
func DecodeVersioned(cursor string, expectedVersion int, into VersionedCursor) (bool, error) {
	err := Decode(cursor, into)
	if err != nil {
		return false, err
	}

	if currentVersion := into.GetVersion(); currentVersion != expectedVersion {
		return true, errors.Errorf("expected cursor version %d, but got %d instead", expectedVersion, currentVersion)
	}

	return false, nil
}

type VersionedCursor interface {
	GetVersion() int
}

type OffsetCursor struct {
	Version int `json:"v"`
	Offset  int `json:"o"`
}

func (c *OffsetCursor) GetVersion() int {
	return c.Version
}

type IndexCursor struct {
	Version int `json:"v"`
	Index   int `json:"i"`
}

func (c *IndexCursor) GetVersion() int {
	return c.Version
}
