package db

import (
	"context"
	"fmt"

	"code.justin.tv/feeds/errors"
)

func (db *Impl) GetEventAttributesForEventIDs(ctx context.Context, eventIDs []string, keys []string) (map[string]map[string]string, error) {
	params := make([]interface{}, 0, len(eventIDs)+len(keys))
	ret := make(map[string]map[string]string)
	for _, eventID := range eventIDs {
		params = append(params, eventID)
		ret[eventID] = map[string]string{}
	}
	for _, key := range keys {
		params = append(params, key)
	}
	query := fmt.Sprintf(
		"SELECT event_node_id, key, value FROM event_node_attributes WHERE event_node_id IN (%s) AND key IN (%s)",
		generatePlaceholders(1, len(eventIDs)),
		generatePlaceholders(len(eventIDs)+1, len(keys)))
	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, query, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not get attributes for events")
	}
	defer db.closeRows(rows)

	for rows.Next() {
		var eventID string
		var key string
		var value string
		err := rows.Scan(&eventID, &key, &value)
		if err != nil {
			return nil, errors.Wrap(err, "could not get attributes for event")
		}
		ret[eventID][key] = value
	}

	return ret, nil
}

func (db *Impl) GetEventAttributes(ctx context.Context, eventID string) (map[string]string, error) {
	query := `
		SELECT key, value
		FROM event_node_attributes
		WHERE event_node_id = $1`

	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, query, eventID)
	if err != nil {
		return nil, errors.Wrap(err, "could not get attributes for event")
	}
	defer db.closeRows(rows)

	ret := make(map[string]string)
	for rows.Next() {
		var key string
		var value string
		err := rows.Scan(&key, &value)
		if err != nil {
			return nil, errors.Wrap(err, "could not get attributes for event")
		}

		ret[key] = value
	}

	return ret, nil
}

func (db *Impl) SetEventAttributes(ctx context.Context, eventID string, attributes map[string]string) error {
	statement := "INSERT INTO event_node_attributes (event_node_id, key, value) VALUES "
	params := make([]interface{}, 0, len(attributes)*3)
	i := 1
	for key, value := range attributes {
		if i > 1 {
			statement += ", "
		}
		statement += fmt.Sprintf("(%s)", generatePlaceholders(i, 3))
		i += 3
		params = append(params, eventID, key, value)
	}
	statement += " ON CONFLICT (event_node_id, key) DO UPDATE SET value = EXCLUDED.value"
	_, err := db.getTxIfJoined(ctx).ExecContext(ctx, statement, params...)
	if err != nil {
		return errors.Wrapf(err, "could not set attributes for event %s", eventID)
	}
	return nil
}
