package db

import (
	"context"
	"fmt"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/feeds/errors"
)

type Broadcast struct {
	ID        string
	EventID   string
	Language  string
	ChannelID string
	GameID    string
	StartTime time.Time
	EndTime   time.Time
}

type CreateDBBroadcastParams struct {
	EventID   string
	Language  string
	ChannelID string
	GameID    string
	StartTime time.Time
	EndTime   time.Time
}

type UpdateDBBroadcastParams struct {
	Language  string
	ChannelID string
	GameID    string
	StartTime time.Time
	EndTime   time.Time
}

type OrderField int

const (
	OrderByStartTime OrderField = iota
	OrderByHype
)

type BroadcastFilter struct {
	Types           []string
	GameIDs         []string
	Languages       []string
	ChannelIDs      []string
	OwnerIDs        []string
	ParentEventIDs  []string
	StartTimeWindow *TimeWindow
	EndTimeWindow   *TimeWindow
}

func (filter *BroadcastFilter) generateSQL(placeholderStart int) ([]string, []interface{}, error) {
	whereFilters := make([]string, 0, 5)
	queryArgs := make([]interface{}, 0, 10)

	if filter.Types != nil {
		f, args, err := membershipTest("e.type", filter.Types, placeholderStart)
		if err != nil {
			return nil, nil, err
		}
		whereFilters = append(whereFilters, f)
		queryArgs = append(queryArgs, args...)
		placeholderStart += len(args)
	}
	if filter.GameIDs != nil {
		f, args, err := membershipTest("b.game_id", filter.GameIDs, placeholderStart)
		if err != nil {
			return nil, nil, err
		}
		whereFilters = append(whereFilters, f)
		queryArgs = append(queryArgs, args...)
		placeholderStart += len(args)
	}
	if filter.Languages != nil {
		f, args, err := membershipTest("b.language", filter.Languages, placeholderStart)
		if err != nil {
			return nil, nil, err
		}
		whereFilters = append(whereFilters, f)
		queryArgs = append(queryArgs, args...)
		placeholderStart += len(args)
	}
	if filter.ChannelIDs != nil {
		f, args, err := membershipTest("b.channel_id", filter.ChannelIDs, placeholderStart)
		if err != nil {
			return nil, nil, err
		}
		whereFilters = append(whereFilters, f)
		queryArgs = append(queryArgs, args...)
		placeholderStart += len(args)
	}
	if filter.OwnerIDs != nil {
		f, args, err := membershipTest("e.owner_id", filter.OwnerIDs, placeholderStart)
		if err != nil {
			return nil, nil, err
		}
		whereFilters = append(whereFilters, f)
		queryArgs = append(queryArgs, args...)
		placeholderStart += len(args)
	}
	if filter.ParentEventIDs != nil {
		f, args, err := membershipTest("e.parent_id", filter.ParentEventIDs, placeholderStart)
		if err != nil {
			return nil, nil, err
		}
		whereFilters = append(whereFilters, f)
		queryArgs = append(queryArgs, args...)
		placeholderStart += len(args)
	}

	f, args := timeWindowTest("b.start_time", filter.StartTimeWindow, placeholderStart)
	if len(args) > 0 {
		whereFilters = append(whereFilters, f)
		queryArgs = append(queryArgs, args...)
		placeholderStart += len(args)
	}

	f, args = timeWindowTest("b.end_time", filter.EndTimeWindow, placeholderStart)
	if len(args) > 0 {
		whereFilters = append(whereFilters, f)
		queryArgs = append(queryArgs, args...)
	}

	return whereFilters, queryArgs, nil
}

func (db *Impl) CreateBroadcast(ctx context.Context, params *CreateDBBroadcastParams) (*Broadcast, error) {
	statement := `
    INSERT INTO broadcasts (
      id,
      event_node_id,
      language,
      channel_id,
      game_id,
      start_time,
      end_time
    ) VALUES (
      $1, $2, $3, $4, $5, $6, $7
    ) RETURNING
      id, event_node_id, language, channel_id, game_id, start_time, end_time`

	startTimeUTC := params.StartTime.UTC()
	endTimeUTC := params.EndTime.UTC()

	row := db.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		statement,
		NewID(),
		params.EventID,
		params.Language,
		params.ChannelID,
		params.GameID,
		startTimeUTC,
		endTimeUTC,
	)

	var broadcast Broadcast
	err := row.Scan(
		&broadcast.ID,
		&broadcast.EventID,
		&broadcast.Language,
		&broadcast.ChannelID,
		&broadcast.GameID,
		&broadcast.StartTime,
		&broadcast.EndTime,
	)

	if err != nil {
		return nil, errors.Wrap(err, "could not create broadcast")
	}

	return &broadcast, err
}

func (db *Impl) UpdateBroadcast(ctx context.Context, eventID, language string, params *UpdateDBBroadcastParams) (*Broadcast, error) {
	statement := `
    UPDATE broadcasts
    SET
      channel_id = $1,
      game_id = $2,
      start_time = $3,
      end_time = $4,
      language = $5
    WHERE event_node_id = $6 AND language = $7
    RETURNING
      id, event_node_id, language, channel_id, game_id, start_time, end_time`

	startTimeUTC := params.StartTime.UTC()
	endTimeUTC := params.EndTime.UTC()

	row := db.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		statement,
		params.ChannelID,
		params.GameID,
		startTimeUTC,
		endTimeUTC,
		params.Language,
		eventID,
		language,
	)

	var broadcast Broadcast
	err := row.Scan(
		&broadcast.ID,
		&broadcast.EventID,
		&broadcast.Language,
		&broadcast.ChannelID,
		&broadcast.GameID,
		&broadcast.StartTime,
		&broadcast.EndTime,
	)

	if err != nil {
		return nil, errors.Wrap(err, "could not update broadcast")
	}

	return &broadcast, err
}

func (db *Impl) DeleteBroadcastsByEventID(ctx context.Context, eventID string) ([]string, error) {
	broadcastIDs, err := db.DeleteBroadcastsByEventIDs(ctx, []string{eventID})
	if err != nil {
		return nil, err
	}
	return broadcastIDs, nil
}

func (db *Impl) DeleteBroadcastsByEventIDs(ctx context.Context, eventIDs []string) ([]string, error) {
	if len(eventIDs) == 0 {
		return nil, errors.New("no event IDs given")
	}
	statement := fmt.Sprintf(`
    DELETE FROM broadcasts
    WHERE event_node_id in (%s)
    RETURNING id`, generatePlaceholders(1, len(eventIDs)))
	params := make([]interface{}, len(eventIDs))
	for i, eventID := range eventIDs {
		params[i] = eventID
	}

	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, statement, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not delete broadcasts")
	}
	defer db.closeRows(rows)

	broadcastIDs := make([]string, 0, 1)
	for rows.Next() {
		var broadcastID string
		err = rows.Scan(&broadcastID)
		if err != nil {
			return nil, errors.Wrap(err, "could not read deleted broadcast ids")
		}
		broadcastIDs = append(broadcastIDs, broadcastID)
	}

	return broadcastIDs, nil
}

func (db *Impl) DeleteBroadcastByEventIDAndLanguage(ctx context.Context, eventID string, language string) ([]string, error) {
	statement := `
    DELETE FROM broadcasts
    WHERE event_node_id = $1 and language = $2
    RETURNING id`
	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, statement, eventID, language)
	if err != nil {
		return nil, errors.Wrap(err, "could not delete broadcast")
	}
	defer db.closeRows(rows)

	broadcastIDs := make([]string, 0, 1)
	for rows.Next() {
		var broadcastID string
		err = rows.Scan(&broadcastID)
		if err != nil {
			return nil, errors.Wrap(err, "could not read deleted broadcast ids")
		}
		broadcastIDs = append(broadcastIDs, broadcastID)
	}

	return broadcastIDs, nil
}

type EventIDs struct {
	EventIDs []string
	Cursor   string
}

func (db *Impl) GetEventIDsSortedByStartTime(ctx context.Context, filter *BroadcastFilter, desc bool, cursor string, limit int) (*EventIDs, error) {
	var (
		order  string
		offset int
	)
	if desc {
		order = "DESC"
	} else {
		order = "ASC"
	}

	if cursor != "" {
		intCursor, err := strconv.Atoi(cursor)
		if err != nil {
			db.Log.Log("err", err, "cursor", cursor, "could not covert cursor to offset")
		}
		offset = intCursor
	}

	queryArgs := make([]interface{}, 2)
	queryArgs[0] = offset
	queryArgs[1] = limit

	whereFilters, optionalArgs, err := filter.generateSQL(3)
	if err != nil {
		return nil, errors.Wrap(err, "unable to generate query for broadcasts")
	}
	queryArgs = append(queryArgs, optionalArgs...)

	whereClause := ""
	if len(whereFilters) > 0 {
		whereClause = "WHERE " + strings.Join(whereFilters, " AND ")
	}

	statement := fmt.Sprintf(`
		SELECT b.event_node_id, b.start_time
		FROM broadcasts b INNER JOIN event_nodes e ON b.event_node_id = e.id
		%s
		GROUP BY b.event_node_id, b.start_time
		ORDER BY b.start_time %s
		OFFSET $1 LIMIT $2`, whereClause, order)

	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, statement, queryArgs...)
	if err != nil {
		return nil, errors.Wrap(err, "could not get event ids")
	}
	defer db.closeRows(rows)

	eventIDs := make([]string, 0, limit)
	for rows.Next() {
		var eventID string
		var startTime time.Time
		err = rows.Scan(&eventID, &startTime)
		if err != nil {
			return nil, errors.Wrap(err, "could not read event ids")
		}
		eventIDs = append(eventIDs, eventID)
	}

	var newCursor string
	if len(eventIDs) == limit {
		newCursor = strconv.Itoa(offset + limit)
	}

	return &EventIDs{EventIDs: eventIDs, Cursor: newCursor}, nil
}

func (db *Impl) GetEventIDsSortedByHype(ctx context.Context, filter *BroadcastFilter, desc bool, cursor string, limit int) (*EventIDs, error) {
	order := "ASC"
	if desc {
		order = "DESC"
	}

	var offset int
	if cursor != "" {
		intCursor, err := strconv.Atoi(cursor)
		if err != nil {
			db.Log.Log("err", err, "cursor", cursor, "could not covert cursor to offset")
		}
		offset = intCursor
	}

	queryArgs := make([]interface{}, 2)
	queryArgs[0] = offset
	queryArgs[1] = limit

	whereFilters, optionalArgs, err := filter.generateSQL(3)
	if err != nil {
		return nil, errors.Wrap(err, "unable to generate query for broadcasts")
	}
	queryArgs = append(queryArgs, optionalArgs...)

	whereClause := ""
	if len(whereFilters) > 0 {
		whereClause = "WHERE " + strings.Join(whereFilters, " AND ")
	}

	// If filter requires fields on the event_nodes table, include a join caluse
	joinEventNodesClause := ""
	if filter.Types != nil || filter.ParentEventIDs != nil {
		joinEventNodesClause = "INNER JOIN event_nodes e ON b.event_node_id = e.id"
	}

	statement := fmt.Sprintf(`
	  SELECT b.event_node_id, b.start_time
	  FROM broadcasts b
	  LEFT JOIN event_node_stats s ON b.event_node_id = s.event_node_id
	  %s
	  %s
	  GROUP BY b.event_node_id, b.start_time, s.follow_count
	  ORDER BY COALESCE(s.follow_count, 0) DESC, b.start_time %s
	  OFFSET $1 LIMIT $2`, joinEventNodesClause, whereClause, order)

	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, statement, queryArgs...)
	if err != nil {
		return nil, errors.Wrap(err, "could not get event ids")
	}
	defer db.closeRows(rows)

	eventIDs := make([]string, 0, limit)
	for rows.Next() {
		var eventID string
		var startTime time.Time
		err = rows.Scan(&eventID, &startTime)
		if err != nil {
			return nil, errors.Wrap(err, "could not read event ids")
		}
		eventIDs = append(eventIDs, eventID)
	}

	var newCursor string
	if len(eventIDs) == limit {
		newCursor = strconv.Itoa(offset + limit)
	}

	return &EventIDs{EventIDs: eventIDs, Cursor: newCursor}, nil
}

func (db *Impl) GetEventIDsSortedByID(ctx context.Context, filter *BroadcastFilter, cursor string, limit int) (*EventIDs, error) {
	queryArgs := make([]interface{}, 1)
	queryArgs[0] = limit

	whereFilters, optionalArgs, err := filter.generateSQL(2)
	if err != nil {
		return nil, errors.Wrap(err, "unable to generate query for broadcasts")
	}
	queryArgs = append(queryArgs, optionalArgs...)

	if cursor != "" {
		queryArgs = append(queryArgs, cursor)
		whereFilters = append(whereFilters, "b.event_node_id > $"+strconv.Itoa(len(queryArgs)))
	}

	statement := fmt.Sprintf(`
	  SELECT b.event_node_id
	  FROM broadcasts b INNER JOIN event_nodes e ON b.event_node_id = e.id
	  WHERE %s
	  GROUP BY b.event_node_id
	  ORDER BY b.event_node_id ASC
	  LIMIT $1`, strings.Join(whereFilters, " AND "))

	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, statement, queryArgs...)
	if err != nil {
		return nil, errors.Wrap(err, "could not get event ids")
	}
	defer db.closeRows(rows)

	eventIDs := make([]string, 0, limit)
	for rows.Next() {
		var eventID string
		err = rows.Scan(&eventID)
		if err != nil {
			return nil, errors.Wrap(err, "could not read event ids")
		}
		eventIDs = append(eventIDs, eventID)
	}

	var newCursor string
	if len(eventIDs) == limit {
		newCursor = eventIDs[len(eventIDs)-1]
	}

	return &EventIDs{EventIDs: eventIDs, Cursor: newCursor}, nil
}

func (db *Impl) GetBroadcastsByHype(ctx context.Context, filter *BroadcastFilter, desc bool, cursor string, limit int) ([]*Broadcast, error) {
	var (
		order  string
		offset int
	)
	if desc {
		order = "DESC"
	} else {
		order = "ASC"
	}

	if cursor != "" {
		intCursor, err := strconv.Atoi(cursor)
		if err != nil {
			db.Log.Log("err", err, "cursor", cursor, "could not covert cursor to offset")
		}
		offset = intCursor
	}

	queryArgs := make([]interface{}, 2)
	queryArgs[0] = offset
	queryArgs[1] = limit

	whereFilters, optionalArgs, err := filter.generateSQL(3)
	if err != nil {
		return nil, errors.Wrap(err, "unable to generate query for broadcasts")
	}
	queryArgs = append(queryArgs, optionalArgs...)

	// If filter requires fields on the event_nodes table, include a join caluse
	joinEventNodesClause := ""
	if filter.Types != nil || filter.ParentEventIDs != nil {
		joinEventNodesClause = "INNER JOIN event_nodes e ON b.event_node_id = e.id"
	}

	statement := fmt.Sprintf(`
	  SELECT b.id, b.event_node_id, b.language, b.channel_id, b.game_id, b.start_time, b.end_time
	  FROM broadcasts b LEFT JOIN event_node_stats s ON b.event_node_id = s.event_node_id
	  %s
	  WHERE %s
	  ORDER BY COALESCE(s.follow_count, 0) DESC, b.start_time %s
	  OFFSET $1 LIMIT $2`, joinEventNodesClause, strings.Join(whereFilters, " AND "), order)

	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, statement, queryArgs...)
	if err != nil {
		return nil, errors.Wrap(err, "could not get event ids")
	}
	defer db.closeRows(rows)

	broadcasts := make([]*Broadcast, 0, limit)
	for rows.Next() {
		var broadcast Broadcast
		err = rows.Scan(
			&broadcast.ID,
			&broadcast.EventID,
			&broadcast.Language,
			&broadcast.ChannelID,
			&broadcast.GameID,
			&broadcast.StartTime,
			&broadcast.EndTime,
		)
		if err != nil {
			return nil, errors.Wrap(err, "could not read event ids")
		}
		broadcasts = append(broadcasts, &broadcast)
	}

	return broadcasts, nil
}
