package db

import (
	"time"

	"code.justin.tv/twitch-events/gea/internal/cursor"
)

const currentCursorVersion = 2

// v2Cursor allows paging across events sorted by start time.  Unlike the v1 cursor (which is an offset that points
// to the event ID that the next page starts with), the v2 cursor points to an event ID in the current page.  Each
// event ID in a page CAN have a v2 cursor that points to that event ID.
type v2Cursor struct {
	// Version is meant to allow us to distinguish between multiple versions of the cursor.  When the shape of
	// v2Cursor changes, the value that Version is set to should be incremented.
	Version   int        `json:"v"`
	EventID   string     `json:"i,omitempty"`
	StartTime *time.Time `json:"s,omitempty"`
}

func (c *v2Cursor) GetVersion() int {
	return c.Version
}

// Create a version 2 cursor string that allows paging across events sorted by start time.
func EncodeNewV2Cursor(eventID string, eventStartTime time.Time) (string, error) {
	c := &v2Cursor{
		Version:   currentCursorVersion,
		EventID:   eventID,
		StartTime: &eventStartTime,
	}
	return cursor.Encode(c)
}

// Extract the event ID and start time that is encoded in a version 2 cursor string.
func unpackV2Cursor(cursorStr string) (string, *time.Time, error) {
	c := v2Cursor{}
	_, err := cursor.DecodeVersioned(cursorStr, currentCursorVersion, &c)
	if err != nil {
		return "", nil, err
	}

	return c.EventID, c.StartTime, nil
}
