package db

import (
	"context"
	"database/sql"
)

type DB interface {
	CreateLocalization(ctx context.Context, params *UpdateDBLocalizationParams) (*Localization, error)
	GetLocalization(ctx context.Context, eventID string, language string) (*Localization, error)
	GetLocalizationsByEventID(ctx context.Context, eventID string) ([]*Localization, error)
	UpdateLocalization(ctx context.Context, eventID string, language string, params *UpdateDBLocalizationParams) (*Localization, error)
	DeleteLocalization(ctx context.Context, eventID string, language string) (*Localization, error)

	CreateEvent(ctx context.Context, params *CreateDBEventParams) (*Event, error)
	GetEvent(ctx context.Context, id string, getDeleted bool) (*Event, error)
	GetEvents(ctx context.Context, ids []string, getDeleted bool) ([]*Event, error)
	GetEventsByParentIDs(ctx context.Context, parentIDs []string, getDeleted bool) ([]*Event, error)
	UpdateEvent(ctx context.Context, eventID string, params *UpdateDBEventParams) (*Event, error)
	DeleteEvent(ctx context.Context, eventID string, params *DeleteDBEventParams) (*Event, error)
	DeleteEvents(ctx context.Context, eventIDs []string, deleteParams *DeleteDBEventParams) ([]*Event, error)
	HardDeleteEventsByOwnerID(ctx context.Context, ownerID string) ([]*Event, error)

	GetEventIDsOrderedByAscStartTime(ctx context.Context, filter *BroadcastFilter, offset *EventIDsOrderedByAscTimeOffset, limit int) ([]EventIDItem, bool, error)
	HasEventIDsOrderedByAscStartTime(
		ctx context.Context, filter *BroadcastFilter, offset *HasEventIDsOrderedByAscTimeOffset) (bool, error)

	CreateBroadcast(ctx context.Context, params *CreateDBBroadcastParams) (*Broadcast, error)
	UpdateBroadcast(ctx context.Context, eventID, language string, params *UpdateDBBroadcastParams) (*Broadcast, error)
	DeleteBroadcastsByEventID(ctx context.Context, eventID string) ([]string, error)
	DeleteBroadcastsByEventIDs(ctx context.Context, eventIDs []string) ([]string, error)
	DeleteBroadcastByEventIDAndLanguage(ctx context.Context, eventID string, language string) ([]string, error)

	GetEventIDsSortedByStartTime(ctx context.Context, filter *BroadcastFilter, desc bool, cursor string, limit int) (*EventIDs, error)
	GetEventIDsSortedByHype(ctx context.Context, filter *BroadcastFilter, desc bool, cursor string, limit int) (*EventIDs, error)
	GetEventIDsSortedByID(ctx context.Context, filter *BroadcastFilter, cursor string, limit int) (*EventIDs, error)
	GetBroadcastsByHype(ctx context.Context, filter *BroadcastFilter, desc bool, cursor string, limit int) ([]*Broadcast, error)

	GetCollectionIDsByOwner(ctx context.Context, ownerID string, desc bool, cursor string, limit int) (*EventIDs, error)

	GetEventAttributesForEventIDs(ctx context.Context, eventIDs []string, keys []string) (map[string]map[string]string, error)
	GetEventAttributes(ctx context.Context, eventID string) (map[string]string, error)
	SetEventAttributes(ctx context.Context, eventID string, attributes map[string]string) error

	GetEventStats(ctx context.Context, eventIDs []string) ([]*EventStats, error)
	IncrementEventFollowCount(ctx context.Context, eventID string) (int64, error)
	DecrementEventFollowCount(ctx context.Context, eventID string) (int64, error)

	StartOrJoinTx(ctx context.Context, opts *sql.TxOptions) (context.Context, bool, error)
	CommitTx(ctx context.Context, createdTx bool) error
	RollbackTxIfNotCommitted(ctx context.Context, createdTx bool)
}
