package db

import (
	"context"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/feeds/errors"
)

type EventIDItem struct {
	ID     string
	Cursor string
}

// EventIDsOrderedByAscTimeOffset contains information required to determine where the current page of IDs starts.
// Only one of EventIDsOrderedByAscTimeOffset's fields should be set.
type EventIDsOrderedByAscTimeOffset struct {
	// When AtEventID is set, it means that the page should start with the given event.
	AtEventID string

	// When EndsAfter is set, it means that the page should start with the earliest event that has an end-time that
	// is after the given time.
	EndsAfter *time.Time

	// When AfterCursor is set, it means that the page should start with the event that comes after the event
	// that the given cursor points to.
	AfterCursor string

	// When BeforeCursor is set, it means that the page should end with the event that comes before the event
	// that the given cursor points to.
	BeforeCursor string
}

func (o *EventIDsOrderedByAscTimeOffset) getIDAndStartTime() (string, *time.Time, error) {
	if o.EndsAfter != nil {
		return "", nil, errors.New("cannot get ID and start time from EndsAtOrBefore offset")
	}

	if o.AfterCursor == "" && o.BeforeCursor == "" {
		return o.AtEventID, nil, nil
	}

	cursorStr := o.AfterCursor
	if cursorStr == "" {
		cursorStr = o.BeforeCursor
	}

	return unpackV2Cursor(cursorStr)
}

func (db *Impl) GetEventIDsOrderedByAscStartTime(
	ctx context.Context, filter *BroadcastFilter, offset *EventIDsOrderedByAscTimeOffset, limit int) ([]EventIDItem, bool, error) {

	allQueryParams := newQueryParams()

	// Use a larger limit than given so that we can check whether there will be another page of results.
	actualLimit := limit + 1
	allQueryParams.add(actualLimit)

	// Generate the filter clauses that allows us to select all event IDs that are in this list.
	whereFilters, queryParams, err := filter.generateSQL(allQueryParams.nextPlaceholder())
	if err != nil {
		return nil, false, errors.Wrap(err, "unable to generate query for broadcasts")
	}
	allQueryParams.add(queryParams...)

	// Generate the filter clause that allows us to select the current page of event IDs from the overall list.
	offsetWhereFilter, queryParams, err := db.getFilterClauseForEventIDsOrderedByAscStartTime(ctx, offset, allQueryParams.nextPlaceholder())
	if err != nil {
		return nil, false, err
	}
	if offsetWhereFilter != "" {
		whereFilters = append(whereFilters, offsetWhereFilter)
		allQueryParams.add(queryParams...)
	}

	// Combine the filter clauses to make a WHERE clause.
	whereClause := ""
	if len(whereFilters) > 0 {
		whereClause = "WHERE " + strings.Join(whereFilters, " AND ")
	}

	order := "ASC"
	if offset.BeforeCursor != "" {
		// When selecting a previous page, order the events by descending start time, so that we can select the events
		// with the latest start times that come before the event specified in the offset.
		order = "DESC"
	}

	statement := fmt.Sprintf(`
		SELECT b.event_node_id, b.start_time
		FROM broadcasts b INNER JOIN event_nodes e ON b.event_node_id = e.id
		%s
		GROUP BY b.event_node_id, b.start_time
		ORDER BY b.start_time %s, b.event_node_id %s
		LIMIT $1`, whereClause, order, order)
	params := allQueryParams.params
	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, statement, params...)
	if err != nil {
		return nil, false, errors.Wrap(err, "could not get event ids")
	}
	defer db.closeRows(rows)

	eventIDItems := make([]EventIDItem, 0, limit)
	hasNextPage := false
	i := 0
	for rows.Next() {
		if i >= limit {
			hasNextPage = true
			break
		}

		var eventID string
		var startTime time.Time
		err = rows.Scan(&eventID, &startTime)
		if err != nil {
			return nil, false, errors.Wrap(err, "could not read event ids")
		}

		cursor, err := EncodeNewV2Cursor(eventID, startTime)
		if err != nil {
			return nil, false, errors.Wrap(err, "could not encode cursor")
		}

		eventIDItems = append(eventIDItems, EventIDItem{
			ID:     eventID,
			Cursor: cursor,
		})
		i++
	}

	if offset.BeforeCursor != "" {
		// When selecting a previous page, reverse the eventIDItems, so that they will be in ascending order by
		// start time.
		reverseEventIDItems(eventIDItems)
	}

	return eventIDItems, hasNextPage, nil
}

// getFilterClauseForEventIDsOrderedByAscStartTime returns a filter clause that can be used to page through event IDs where the events are ordered
// by start time and event ID.
//
// For example, suppose our listing has an event with ID, 'myeventid' and start time, 2016-11-04T05:00.
//
// If we want to get the page of events after our event, we could filter using
//     (b.start_time, b.event_node_id) > (2016-11-04T05:00, 'myeventid')
//
// If we want to get the page of events before our event, we could filter using
//     (b.start_time, b.event_node_id) < (2016-11-04T05:00, 'myeventid')
//
// If we want to start the page of events with our event, we could filter using,
//     (b.start_time, b.event_node_id) >= (2016-11-04T05:00, 'myeventid')
//
// If we want to start the page of events with an event with an end-time after the given time, we can filter with,
//      b.end_time > 2016-11-04T07:00
func (db *Impl) getFilterClauseForEventIDsOrderedByAscStartTime(
	ctx context.Context, offset *EventIDsOrderedByAscTimeOffset, placeholderStart int) (string, []interface{}, error) {

	// Generate a filter clause for the EndsAfter strategy.
	if offset.EndsAfter != nil {
		whereClause := fmt.Sprintf("b.end_time > $%d", placeholderStart)
		queryParams := []interface{}{*offset.EndsAfter}
		return whereClause, queryParams, nil
	}

	// Generate a filter clause for the AtEventID, BeforeCursor and AfterCursor strategies.
	comparison := ""
	switch {
	case offset.AtEventID != "":
		comparison = ">="
	case offset.BeforeCursor != "":
		comparison = "<"
	case offset.AfterCursor != "":
		comparison = ">"
	}

	eventID, startTimePtr, err := offset.getIDAndStartTime()
	if err != nil {
		return "", nil, err
	}

	var startTime time.Time
	if startTimePtr != nil {
		startTime = *startTimePtr
	} else {
		event, err := db.GetEvent(ctx, eventID, false)
		if err != nil || event == nil || event.StartTime == nil {
			return "", nil, err
		}

		startTime = *event.StartTime
	}

	whereClause := fmt.Sprintf(
		"(b.start_time, b.event_node_id) %s ($%d, $%d)", comparison, placeholderStart, placeholderStart+1)
	queryParams := []interface{}{startTime, eventID}
	return whereClause, queryParams, nil
}

// HasEventIDsOrderedByAscTimeOffset contains information required to determine where the current page of IDs starts.
// Only one of HasEventIDsOrderedByAscTimeOffset's fields should be set.
type HasEventIDsOrderedByAscTimeOffset struct {
	BeforeEventID    string
	EndsAtOrBefore   *time.Time
	AtOrAfterCursor  string
	AtOrBeforeCursor string
}

// InvertEventIDsOrderedByAscTimeOffset returns an offset object that can be used to query whether there are events
// in the opposite direction from the given offset object.
func InvertEventIDsOrderedByAscTimeOffset(o *EventIDsOrderedByAscTimeOffset) *HasEventIDsOrderedByAscTimeOffset {
	return &HasEventIDsOrderedByAscTimeOffset{
		BeforeEventID:    o.AtEventID,
		EndsAtOrBefore:   o.EndsAfter,
		AtOrAfterCursor:  o.BeforeCursor,
		AtOrBeforeCursor: o.AfterCursor,
	}
}

func (o *HasEventIDsOrderedByAscTimeOffset) isEmpty() bool {
	return o.BeforeEventID == "" && o.EndsAtOrBefore == nil && o.AtOrAfterCursor == "" && o.AtOrBeforeCursor == ""
}

func (o *HasEventIDsOrderedByAscTimeOffset) getIDAndStartTime() (string, *time.Time, error) {
	if o.EndsAtOrBefore != nil {
		return "", nil, errors.New("cannot get ID and start time from EndsAtOrBefore offset")
	}

	if o.AtOrAfterCursor == "" && o.AtOrBeforeCursor == "" {
		return o.BeforeEventID, nil, nil
	}

	cursorStr := o.AtOrAfterCursor
	if cursorStr == "" {
		cursorStr = o.AtOrBeforeCursor
	}

	return unpackV2Cursor(cursorStr)
}

// HasEventIDsOrderedByAscStartTime returns true if the page of events created by applying the given filter, and paging
// using the given offset has events.
func (db *Impl) HasEventIDsOrderedByAscStartTime(
	ctx context.Context, filter *BroadcastFilter, offset *HasEventIDsOrderedByAscTimeOffset) (bool, error) {
	if offset.isEmpty() {
		return false, nil
	}

	allQueryParams := newQueryParams()

	// Generate the filter clauses that allows us to select all event IDs that are in this list.
	whereFilters, queryParams, err := filter.generateSQL(allQueryParams.nextPlaceholder())
	if err != nil {
		return false, errors.Wrap(err, "unable to generate query for broadcasts")
	}
	allQueryParams.add(queryParams...)

	// Generate the filter clause that allows us to select the current page of event IDs from the overall list.
	offsetWhereFilter, queryParams, err := db.getHasOffsetWhereFilter(ctx, offset, allQueryParams.nextPlaceholder())
	if err != nil {
		return false, err
	}
	if offsetWhereFilter == "" {
		return false, nil
	}
	whereFilters = append(whereFilters, offsetWhereFilter)
	allQueryParams.add(queryParams...)

	// Combine the filter clauses to make a WHERE clause.
	whereClause := ""
	if len(whereFilters) > 0 {
		whereClause = "WHERE " + strings.Join(whereFilters, " AND ")
	}

	var count int
	statement := fmt.Sprintf(`
		SELECT COUNT(*)
		FROM broadcasts b INNER JOIN event_nodes e ON b.event_node_id = e.id
		%s
		LIMIT 1`, whereClause)
	params := allQueryParams.params
	row := db.getTxIfJoined(ctx).QueryRowContext(ctx, statement, params...)
	err = row.Scan(&count)
	if err != nil {
		return false, errors.Wrap(err, "could not check if there are pages before the current page")
	}

	return count > 0, nil
}

func (db *Impl) getHasOffsetWhereFilter(
	ctx context.Context,
	offset *HasEventIDsOrderedByAscTimeOffset,
	placeholderStart int) (string, []interface{}, error) {

	// Generate a filter clause for the EndsAtOrBefore strategy.
	if offset.EndsAtOrBefore != nil {
		whereClause := fmt.Sprintf("b.end_time <= $%d", placeholderStart)
		queryParams := []interface{}{*offset.EndsAtOrBefore}
		return whereClause, queryParams, nil
	}

	// Generate a filter clause for the AtEventID, BeforeCursor and AfterCursor strategies.
	comparison := ""
	switch {
	case offset.BeforeEventID != "":
		comparison = "<"
	case offset.AtOrBeforeCursor != "":
		comparison = "<="
	case offset.AtOrAfterCursor != "":
		comparison = ">="
	}

	eventID, startTimePtr, err := offset.getIDAndStartTime()
	if err != nil {
		return "", nil, err
	}

	var startTime time.Time
	if startTimePtr != nil {
		startTime = *startTimePtr
	} else {
		event, err := db.GetEvent(ctx, eventID, false)
		if err != nil || event == nil || event.StartTime == nil {
			return "", nil, err
		}

		startTime = *event.StartTime
	}

	whereClause := fmt.Sprintf(
		"(b.start_time, b.event_node_id) %s ($%d, $%d)", comparison, placeholderStart, placeholderStart+1)
	queryParams := []interface{}{startTime, eventID}
	return whereClause, queryParams, nil
}

func reverseEventIDItems(items []EventIDItem) {
	if len(items) == 0 {
		return
	}

	last := len(items) - 1
	for i := 0; i < len(items)/2; i++ {
		items[i], items[last-i] = items[last-i], items[i]
	}
}
