package db

import (
	"context"
	"fmt"

	"code.justin.tv/feeds/errors"
)

type EventStats struct {
	EventID     string
	FollowCount int
}

func (db *Impl) GetEventStats(ctx context.Context, eventIDs []string) ([]*EventStats, error) {
	if len(eventIDs) == 0 {
		return nil, nil
	}
	placeholders := generatePlaceholders(1, len(eventIDs))
	statement := fmt.Sprintf(`
		SELECT e.id, s.follow_count
		FROM event_nodes e LEFT OUTER JOIN event_node_stats s ON e.id = s.event_node_id
		WHERE e.deleted_at IS NULL and e.id IN (%v)
	`, placeholders)

	args := make([]interface{}, 0, len(eventIDs))
	for _, v := range eventIDs {
		args = append(args, v)
	}
	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, statement, args...)
	if err != nil {
		return nil, errors.Wrap(err, "could not get event stats")
	}
	defer db.closeRows(rows)

	res := make([]*EventStats, 0, len(eventIDs))
	for rows.Next() {
		stats := EventStats{}
		var followCount *int
		err = rows.Scan(&stats.EventID, &followCount)
		if err != nil {
			return nil, errors.Wrap(err, "could not read event stats")
		}
		if followCount != nil {
			stats.FollowCount = *followCount
		}
		res = append(res, &stats)
	}

	return res, nil
}

func (db *Impl) IncrementEventFollowCount(ctx context.Context, eventID string) (int64, error) {
	statement := `
	INSERT INTO event_node_stats (event_node_id, follow_count) VALUES ($1, 1)
	ON CONFLICT (event_node_id)
	DO UPDATE SET follow_count = event_node_stats.follow_count + 1
	RETURNING follow_count`

	followCount := int64(-1)
	row := db.getTxIfJoined(ctx).QueryRowContext(ctx, statement, eventID)
	err := row.Scan(&followCount)
	if err != nil {
		return -1, errors.Wrap(err, "could not increment event follow count")
	}
	return followCount, nil
}

func (db *Impl) DecrementEventFollowCount(ctx context.Context, eventID string) (int64, error) {
	statement := `
	INSERT INTO event_node_stats (event_node_id, follow_count) VALUES ($1, 0)
	ON CONFLICT (event_node_id)
	DO UPDATE SET follow_count = event_node_stats.follow_count - 1 WHERE event_node_stats.follow_count > 0
	RETURNING follow_count`

	followCount := int64(-1)
	row := db.getTxIfJoined(ctx).QueryRowContext(ctx, statement, eventID)
	err := row.Scan(&followCount)
	if err != nil {
		return -1, errors.Wrap(err, "could not decrement event follow count")
	}
	return followCount, nil
}
