package db

import (
	"context"
	"database/sql"
	"fmt"
	"time"

	"code.justin.tv/feeds/errors"
	c "code.justin.tv/twitch-events/gea/internal/cursor"
	"code.justin.tv/twitch-events/gea/internal/images"
)

const (
	// MaxGetEventsIDs is the maximum number of event IDs that can be passed to GetEvents.
	MaxGetEventsIDs = inClauseLimit

	cursorVersion = 1
)

type CreateDBEventParams struct {
	OwnerID  string
	Type     string
	ParentID *string

	StartTime  *time.Time
	EndTime    *time.Time
	TimeZoneID *string

	CoverImageID *images.ImageID
	Language     *string
	Title        *string
	Description  *string
	ChannelID    *string
	GameID       *string

	Timestamp time.Time
}

type UpdateDBEventParams struct {
	OwnerID  string
	Type     string
	ParentID *string

	StartTime  *time.Time
	EndTime    *time.Time
	TimeZoneID *string

	CoverImageID *images.ImageID
	Language     *string
	Title        *string
	Description  *string
	ChannelID    *string
	GameID       *string

	Timestamp time.Time
}

type DeleteDBEventParams struct {
	Timestamp time.Time
}

type Event struct {
	ID       string
	OwnerID  string
	Type     string
	ParentID *string

	CreatedAt time.Time
	UpdatedAt time.Time
	DeletedAt *time.Time

	StartTime  *time.Time
	EndTime    *time.Time
	TimeZoneID *string

	CoverImageID *images.ImageID
	Language     *string
	Title        *string
	Description  *string
	ChannelID    *string
	GameID       *string
}

func (db *Impl) CreateEvent(ctx context.Context, params *CreateDBEventParams) (*Event, error) {
	now := ConvertToDBTime(params.Timestamp)

	statement := `
    INSERT INTO event_nodes (
      id,
      owner_id,
      type,
      parent_id,

      created_at,
      updated_at,

      start_time,
      end_time,
      time_zone_id,

      cover_image_id,
      language,
      title,
      description,
      channel_id,
      game_id

    ) VALUES (
      $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15
    ) RETURNING
      id, owner_id, type, parent_id, created_at, updated_at, deleted_at, start_time, end_time, time_zone_id, cover_image_id, language, title, description, channel_id, game_id
  `
	var startTimeUTC *time.Time
	if params.StartTime != nil {
		startTime := params.StartTime.UTC()
		startTimeUTC = &startTime
	}
	var endTimeUTC *time.Time
	if params.EndTime != nil {
		endTime := params.EndTime.UTC()
		endTimeUTC = &endTime
	}

	row := db.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		statement,
		NewID(),
		params.OwnerID,
		params.Type,
		params.ParentID,
		now,
		now,
		startTimeUTC,
		endTimeUTC,
		params.TimeZoneID,
		params.CoverImageID,
		params.Language,
		params.Title,
		params.Description,
		params.ChannelID,
		params.GameID,
	)

	var event Event
	err := row.Scan(
		&event.ID,
		&event.OwnerID,
		&event.Type,
		&event.ParentID,

		&event.CreatedAt,
		&event.UpdatedAt,
		&event.DeletedAt,

		&event.StartTime,
		&event.EndTime,
		&event.TimeZoneID,

		&event.CoverImageID,
		&event.Language,
		&event.Title,
		&event.Description,
		&event.ChannelID,
		&event.GameID,
	)

	if err != nil {
		return nil, errors.Wrap(err, "could not create event")
	}

	return &event, err
}

func (db *Impl) GetEvent(ctx context.Context, id string, getDeleted bool) (*Event, error) {
	query := `
		SELECT id, owner_id, type, parent_id, created_at, updated_at, deleted_at,
			cover_image_id, language, title, description, start_time, end_time,
			time_zone_id, channel_id, game_id
		FROM event_nodes
		WHERE id = $1`
	if !getDeleted {
		query += " AND deleted_at IS NULL"
	}
	row := db.getTxIfJoined(ctx).QueryRowContext(ctx, query, id)

	var event Event
	err := row.Scan(
		&event.ID,
		&event.OwnerID,
		&event.Type,
		&event.ParentID,
		&event.CreatedAt,
		&event.UpdatedAt,
		&event.DeletedAt,
		&event.CoverImageID,
		&event.Language,
		&event.Title,
		&event.Description,
		&event.StartTime,
		&event.EndTime,
		&event.TimeZoneID,
		&event.ChannelID,
		&event.GameID,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not read values for event %v", id)
	}

	return &event, nil
}

func (db *Impl) GetEvents(ctx context.Context, ids []string, getDeleted bool) ([]*Event, error) {
	if len(ids) == 0 {
		return nil, errors.New("not enough values to do a in clause on id")
	}
	if len(ids) > MaxGetEventsIDs {
		return nil, errors.Errorf("too many arguments provided for IN on id, args received: %d, limit: %d", len(ids), MaxGetEventsIDs)
	}
	query := fmt.Sprintf(`
		SELECT id, owner_id, type, parent_id, created_at, updated_at, deleted_at,
			cover_image_id, language, title, description, start_time, end_time,
			time_zone_id, channel_id, game_id
		FROM event_nodes
		WHERE id IN (%s)`, generatePlaceholders(1, len(ids)))
	if !getDeleted {
		query += " AND deleted_at IS NULL"
	}

	params := make([]interface{}, len(ids))
	for i, id := range ids {
		params[i] = id
	}

	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, query, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not get events")
	}
	defer db.closeRows(rows)

	events := make([]*Event, 0, len(ids))
	for rows.Next() {
		var event Event
		err := rows.Scan(
			&event.ID,
			&event.OwnerID,
			&event.Type,
			&event.ParentID,
			&event.CreatedAt,
			&event.UpdatedAt,
			&event.DeletedAt,
			&event.CoverImageID,
			&event.Language,
			&event.Title,
			&event.Description,
			&event.StartTime,
			&event.EndTime,
			&event.TimeZoneID,
			&event.ChannelID,
			&event.GameID,
		)
		if err != nil {
			return nil, errors.Wrap(err, "could not read event")
		}
		events = append(events, &event)
	}

	return events, nil
}

func (db *Impl) GetEventsByParentIDs(ctx context.Context, parentIDs []string, getDeleted bool) ([]*Event, error) {
	if len(parentIDs) == 0 {
		return nil, errors.New("not enough values to do an in clause on parent_id")
	}
	if len(parentIDs) > MaxGetEventsIDs {
		return nil, errors.Errorf("too many arguments provided for IN on parent_id, args received: %d, limit: %d", len(parentIDs), MaxGetEventsIDs)
	}
	query := fmt.Sprintf(`
		SELECT id, owner_id, type, parent_id, created_at, updated_at, deleted_at,
			cover_image_id, language, title, description, start_time, end_time,
			time_zone_id, channel_id, game_id
		FROM event_nodes
		WHERE parent_id IN (%s)`, generatePlaceholders(1, len(parentIDs)))
	if !getDeleted {
		query += " AND deleted_at IS NULL"
	}

	params := make([]interface{}, len(parentIDs))
	for i, id := range parentIDs {
		params[i] = id
	}

	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, query, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not get events")
	}
	defer db.closeRows(rows)

	var events []*Event
	for rows.Next() {
		var event Event
		err := rows.Scan(
			&event.ID,
			&event.OwnerID,
			&event.Type,
			&event.ParentID,
			&event.CreatedAt,
			&event.UpdatedAt,
			&event.DeletedAt,
			&event.CoverImageID,
			&event.Language,
			&event.Title,
			&event.Description,
			&event.StartTime,
			&event.EndTime,
			&event.TimeZoneID,
			&event.ChannelID,
			&event.GameID,
		)
		if err != nil {
			return nil, errors.Wrap(err, "could not read event")
		}
		events = append(events, &event)
	}

	return events, nil
}

func (db *Impl) UpdateEvent(ctx context.Context, eventID string, params *UpdateDBEventParams) (*Event, error) {
	now := ConvertToDBTime(params.Timestamp)

	statement := `
    UPDATE event_nodes
    SET
      owner_id = $1,
      type = $2,
      parent_id = $3,
      updated_at = $4,
      start_time = $5,
      end_time = $6,
      time_zone_id = $7,
      cover_image_id = $8,
      language = $9,
      title = $10,
      description = $11,
      channel_id = $12,
      game_id = $13
    WHERE id = $14
    RETURNING 
      id, owner_id, type, parent_id, created_at, updated_at, deleted_at, start_time, end_time, time_zone_id, cover_image_id, language, title, description, channel_id, game_id
  `

	var startTimeUTC *time.Time
	if params.StartTime != nil {
		startTime := params.StartTime.UTC()
		startTimeUTC = &startTime
	}
	var endTimeUTC *time.Time
	if params.EndTime != nil {
		endTime := params.EndTime.UTC()
		endTimeUTC = &endTime
	}

	row := db.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		statement,
		params.OwnerID,
		params.Type,
		params.ParentID,
		now,
		startTimeUTC,
		endTimeUTC,
		params.TimeZoneID,
		params.CoverImageID,
		params.Language,
		params.Title,
		params.Description,
		params.ChannelID,
		params.GameID,
		eventID,
	)

	var event Event
	err := row.Scan(
		&event.ID,
		&event.OwnerID,
		&event.Type,
		&event.ParentID,
		&event.CreatedAt,
		&event.UpdatedAt,
		&event.DeletedAt,
		&event.StartTime,
		&event.EndTime,
		&event.TimeZoneID,
		&event.CoverImageID,
		&event.Language,
		&event.Title,
		&event.Description,
		&event.ChannelID,
		&event.GameID,
	)

	if err != nil {
		return nil, errors.Wrap(err, "could not update event")
	}

	return &event, err
}

func (db *Impl) DeleteEvent(ctx context.Context, eventID string, params *DeleteDBEventParams) (*Event, error) {
	events, err := db.DeleteEvents(ctx, []string{eventID}, params)
	if err != nil {
		return nil, err
	}
	if len(events) == 0 {
		return nil, nil
	}
	return events[0], nil
}

func (db *Impl) DeleteEvents(ctx context.Context, eventIDs []string, deleteParams *DeleteDBEventParams) ([]*Event, error) {
	if len(eventIDs) == 0 {
		return nil, errors.New("no event IDs given")
	}
	now := ConvertToDBTime(deleteParams.Timestamp)

	statement := fmt.Sprintf(`
		UPDATE event_nodes
		SET deleted_at = $1, updated_at = $2
		WHERE id in (%s)
		RETURNING id, owner_id, type, parent_id, created_at, updated_at, deleted_at,
			cover_image_id, language, title, description, start_time, end_time,
			time_zone_id, channel_id, game_id
	`, generatePlaceholders(3, len(eventIDs)))

	numOfNonIDParams := 2
	params := make([]interface{}, len(eventIDs)+numOfNonIDParams)
	params[0] = now
	params[1] = now
	for i, eventID := range eventIDs {
		params[i+numOfNonIDParams] = eventID
	}

	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, statement, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not delete events")
	}
	defer db.closeRows(rows)

	var events []*Event
	for rows.Next() {
		var event Event
		err = rows.Scan(
			&event.ID,
			&event.OwnerID,
			&event.Type,
			&event.ParentID,
			&event.CreatedAt,
			&event.UpdatedAt,
			&event.DeletedAt,
			&event.CoverImageID,
			&event.Language,
			&event.Title,
			&event.Description,
			&event.StartTime,
			&event.EndTime,
			&event.TimeZoneID,
			&event.ChannelID,
			&event.GameID,
		)
		if err != nil {
			return nil, errors.Wrap(err, "could not read deleted event")
		}
		events = append(events, &event)
	}

	return events, err
}

func (db *Impl) HardDeleteEventsByOwnerID(ctx context.Context, ownerID string) ([]*Event, error) {
	if ownerID == "" {
		return nil, errors.New("no owner ID given")
	}
	params := []interface{}{ownerID}
	statsStatement := `
		DELETE FROM event_node_stats
		USING event_nodes
		WHERE event_nodes.owner_id = $1
		AND event_nodes.id = event_node_stats.event_node_id
	`
	_, err := db.getTxIfJoined(ctx).ExecContext(ctx, statsStatement, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not hard delete event stats")
	}

	attributesStatement := `
		DELETE FROM event_node_attributes
		USING event_nodes
		WHERE event_nodes.owner_id = $1
		AND event_nodes.id = event_node_attributes.event_node_id
	`
	_, err = db.getTxIfJoined(ctx).ExecContext(ctx, attributesStatement, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not hard delete event attributes")
	}

	localizationsStatement := `
		DELETE FROM event_node_localizations
		USING event_nodes
		WHERE event_nodes.owner_id = $1
		AND event_nodes.id = event_node_localizations.event_node_id
	`
	_, err = db.getTxIfJoined(ctx).ExecContext(ctx, localizationsStatement, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not hard delete event localizations")
	}

	broadcastsStatement := `
		DELETE FROM broadcasts
		USING event_nodes
		WHERE event_nodes.owner_id = $1
		AND event_nodes.id = broadcasts.event_node_id
	`
	_, err = db.getTxIfJoined(ctx).ExecContext(ctx, broadcastsStatement, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not hard delete broadcasts")
	}

	eventsStatement := `
		DELETE FROM event_nodes
		WHERE owner_id = $1
		RETURNING id, owner_id, type, parent_id, created_at, updated_at, deleted_at,
			cover_image_id, language, title, description, start_time, end_time,
			time_zone_id, channel_id, game_id
	`

	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, eventsStatement, params...)
	if err != nil {
		return nil, errors.Wrap(err, "could not hard delete events")
	}
	defer db.closeRows(rows)

	var events []*Event
	for rows.Next() {
		var event Event
		err = rows.Scan(
			&event.ID,
			&event.OwnerID,
			&event.Type,
			&event.ParentID,
			&event.CreatedAt,
			&event.UpdatedAt,
			&event.DeletedAt,
			&event.CoverImageID,
			&event.Language,
			&event.Title,
			&event.Description,
			&event.StartTime,
			&event.EndTime,
			&event.TimeZoneID,
			&event.ChannelID,
			&event.GameID,
		)
		if err != nil {
			return nil, errors.Wrap(err, "could not read hard deleted event")
		}
		events = append(events, &event)
	}

	return events, err
}

func (db *Impl) GetCollectionIDsByOwner(ctx context.Context, ownerID string, desc bool, cursor string, limit int) (*EventIDs, error) {
	if ownerID == "" {
		return nil, errors.New("no channelID provided")
	}

	var (
		order  string
		offset int
	)
	if desc {
		order = "DESC"
	} else {
		order = "ASC"
	}

	if cursor != "" {
		var decodedCursor c.OffsetCursor
		err := c.Decode(cursor, &decodedCursor)
		if err != nil {
			db.Log.Log("err", err, "cursor", cursor, "invalid cursor supplied")
			return nil, err
		}
		offset = decodedCursor.Offset
	}

	queryArgs := make([]interface{}, 3)
	queryArgs[0] = ownerID
	queryArgs[1] = offset
	queryArgs[2] = limit

	query := fmt.Sprintf(`
		SELECT id
		FROM event_nodes
		WHERE type = 'timetable' AND deleted_at IS NULL AND owner_id = $1
		ORDER BY title %s
		OFFSET $2 LIMIT $3`, order)

	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, query, queryArgs...)
	if err != nil {
		return nil, errors.Wrap(err, "could not get collection event ids")
	}
	defer db.closeRows(rows)

	eventIDs := make([]string, 0, limit)
	for rows.Next() {
		var eventID string
		err = rows.Scan(&eventID)
		if err != nil {
			return nil, errors.Wrap(err, "could not read event ids")
		}
		eventIDs = append(eventIDs, eventID)
	}

	var encodedCursor string
	if len(eventIDs) == limit {
		newOffset := offset + limit
		newCursor := c.OffsetCursor{
			Offset:  newOffset,
			Version: cursorVersion,
		}
		encodedCursor, err = c.Encode(newCursor)
		if err != nil {
			return nil, errors.Wrap(err, "unable to encode resulting cursor")
		}
	}
	return &EventIDs{EventIDs: eventIDs, Cursor: encodedCursor}, nil
}
