package db

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
)

type UpdateDBLocalizationParams struct {
	EventID  string
	Language string

	Title       *string
	Description *string
	ChannelID   *string
}

type Localization struct {
	EventID  string
	Language string

	CreatedAt time.Time
	UpdatedAt time.Time

	Title       *string
	Description *string
	ChannelID   *string
}

func scanLocalizationRow(row *sql.Row) (*Localization, error) {
	var res Localization
	err := row.Scan(
		&res.EventID,
		&res.Language,

		&res.CreatedAt,
		&res.UpdatedAt,

		&res.Title,
		&res.Description,
		&res.ChannelID,
	)
	if err != nil {
		return nil, err
	}
	return &res, nil
}

func scanLocalizationRowFromRows(rows *sql.Rows) (*Localization, error) {
	var res Localization
	err := rows.Scan(
		&res.EventID,
		&res.Language,

		&res.CreatedAt,
		&res.UpdatedAt,

		&res.Title,
		&res.Description,
		&res.ChannelID,
	)
	if err != nil {
		return nil, err
	}
	return &res, nil
}

func (db *Impl) GetLocalization(ctx context.Context, eventID string, language string) (*Localization, error) {
	row := db.getTxIfJoined(ctx).QueryRowContext(ctx, `
		SELECT event_node_id, language, created_at, updated_at, title, description, channel_id
		FROM event_node_localizations
		WHERE event_node_id = $1 and language = $2
	`, eventID, language)

	res, err := scanLocalizationRow(row)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not read values for localization for event %v language %v", eventID, language)
	}

	return res, nil
}

func (db *Impl) GetLocalizationsByEventID(ctx context.Context, eventID string) ([]*Localization, error) {
	rows, err := db.getTxIfJoined(ctx).QueryContext(ctx, `
		SELECT event_node_id, language, created_at, updated_at, title, description, channel_id
		FROM event_node_localizations
		WHERE event_node_id = $1
	`, eventID)
	if err != nil {
		return nil, errors.Wrap(err, "could not get localizations")
	}
	defer db.closeRows(rows)

	res := []*Localization{}
	for rows.Next() {
		localization, err := scanLocalizationRowFromRows(rows)
		if err != nil {
			return nil, errors.Wrap(err, "could not read localization")
		}
		res = append(res, localization)
	}
	return res, nil
}

func (db *Impl) CreateLocalization(ctx context.Context, params *UpdateDBLocalizationParams) (*Localization, error) {
	now := now()

	statement := `
    INSERT INTO event_node_localizations (
      event_node_id,
      language,

      created_at,
      updated_at,

      title,
      description,
      channel_id
    ) VALUES (
      $1, $2, $3, $4, $5, $6, $7
    ) RETURNING
      event_node_id, language, created_at, updated_at, title, description, channel_id
  `

	row := db.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		statement,
		params.EventID,
		params.Language,
		now,
		now,
		params.Title,
		params.Description,
		params.ChannelID,
	)
	res, err := scanLocalizationRow(row)
	if err != nil {
		return nil, errors.Wrap(err, "could not create localization")
	}
	return res, nil
}

func (db *Impl) UpdateLocalization(ctx context.Context, eventID string, language string, params *UpdateDBLocalizationParams) (*Localization, error) {
	now := now()

	statement := `
    UPDATE event_node_localizations
    SET
      updated_at = $1,
      title = $2,
      description = $3,
      channel_id = $4
    WHERE event_node_id = $5 and language = $6
    RETURNING 
      event_node_id, language, created_at, updated_at, title, description, channel_id
  `

	row := db.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		statement,
		now,
		params.Title,
		params.Description,
		params.ChannelID,
		eventID,
		language,
	)

	res, err := scanLocalizationRow(row)
	if err != nil {
		return nil, errors.Wrap(err, "could not update localization")
	}
	return res, nil
}

func (db *Impl) DeleteLocalization(ctx context.Context, eventID string, language string) (*Localization, error) {
	statement := `
    DELETE FROM event_node_localizations
    WHERE event_node_id = $1 and language = $2
    RETURNING 
      event_node_id, language, created_at, updated_at, title, description, channel_id
  `

	row := db.getTxIfJoined(ctx).QueryRowContext(ctx, statement, eventID, language)

	res, err := scanLocalizationRow(row)
	if err != nil {
		return nil, errors.Wrap(err, "could not delete localization")
	}
	return res, nil
}
