package filter

import (
	"context"

	"code.justin.tv/feeds/log"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/twitch-events/gea/internal/db"
	usersmodels "code.justin.tv/web/users-service/models"
)

// GetUsersClient contains the only method on the internal users service client that Filter uses.  It is meant to
// facilitate stubbing during tests.
type GetUsersClient interface {
	GetUsers(ctx context.Context, params *usersmodels.FilterParams, reqOpts *twitchclient.ReqOpts) (*usersmodels.PropertiesResult, error)
}

// Filter facilitates removing events that should not be served because they are owned by users
// who have Terms of Service violations, or users that have been deleted.
type Filter struct {
	GetUsersClient GetUsersClient
	Log            *log.ElevatedLog
}

func (f *Filter) FilterEvent(ctx context.Context, event *db.Event) *db.Event {
	if event == nil {
		return nil
	}

	validEvents := f.FilterEvents(ctx, []*db.Event{event})
	if len(validEvents) == 0 {
		return nil
	}

	return validEvents[0]
}

func (f *Filter) FilterEvents(ctx context.Context, events []*db.Event) []*db.Event {
	userIDs := stringSet{}
	for _, event := range events {
		userIDs.add(event.OwnerID)
	}

	validUserIDs := f.loadValidUsers(ctx, userIDs)
	validEvents := make([]*db.Event, 0, len(events))
	for _, event := range events {
		if !validUserIDs.contains(event.OwnerID) {
			continue
		}

		validEvents = append(validEvents, event)
	}

	return validEvents
}

func (f *Filter) loadValidUsers(ctx context.Context, userIDs stringSet) stringSet {
	if userIDs.isEmpty() {
		return stringSet{}
	}

	params := &usersmodels.FilterParams{
		NotDeleted:      true,
		NoTOSViolation:  true,
		NoDMCAViolation: true,
		IDs:             userIDs.toSlice(),
	}
	res, err := f.GetUsersClient.GetUsers(ctx, params, nil)
	if err != nil {
		// If user service is down, just assume all the users are valid
		f.Log.LogCtx(ctx, "err", err, "checking whether an event's owner is valid failed")
		return userIDs
	}

	validUserIDs := stringSet{}
	for _, result := range res.Results {
		validUserIDs.add(result.ID)
	}

	return validUserIDs
}

type stringSet map[string]struct{}

func (u stringSet) add(userID string) {
	u[userID] = struct{}{}
}

func (u stringSet) contains(userID string) bool {
	_, contains := u[userID]
	return contains
}

func (u stringSet) isEmpty() bool {
	return len(u) == 0
}

func (u stringSet) toSlice() []string {
	ret := make([]string, 0, len(u))
	for s := range u {
		ret = append(ret, s)
	}
	return ret
}
