package follows

import (
	"context"
	"encoding/base64"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedsdynamo"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

type EventFollowsConfig struct {
	EventFollowsTableName *distconf.Str
}

func (c *EventFollowsConfig) Load(dconf *distconf.Distconf) error {
	c.EventFollowsTableName = dconf.Str("gea.event_follows_table", "")
	if c.EventFollowsTableName.Get() == "" {
		return errors.New("unable to find event follows table")
	}

	return nil
}

type EventFollows struct {
	Dynamo                 *dynamodb.DynamoDB
	Config                 *EventFollowsConfig
	Log                    *log.ElevatedLog
	RequireConsistentReads bool
}

func (f *EventFollows) HealthCheck() error {
	_, err := f.Dynamo.DescribeTable(&dynamodb.DescribeTableInput{TableName: aws.String(f.Config.EventFollowsTableName.Get())})
	return err
}

func now() time.Time {
	t := time.Now().UTC().Truncate(time.Microsecond)
	return t
}

type PagedUserIDs struct {
	UserIDs []string
	Cursor  string
}

type PagedEventIDs struct {
	EventIDs []string
	Cursor   string
}

func decodeCursor(encoded string) (string, error) {
	b, err := base64.StdEncoding.DecodeString(encoded)
	if err != nil {
		return "", errors.Wrap(err, "cursor decode error")
	}
	return string(b), nil
}

func encodeCursor(cursor string) string {
	return base64.StdEncoding.EncodeToString([]byte(cursor))
}

func (f *EventFollows) GetFollowersByEventID(ctx context.Context, eventID string, limit int, cursor string) (*PagedUserIDs, error) {
	var startKey map[string]*dynamodb.AttributeValue
	if cursor != "" {
		userID, err := decodeCursor(cursor)
		if err != nil {
			return nil, err
		}
		startKey = map[string]*dynamodb.AttributeValue{
			"event_id": {S: &eventID},
			"user_id":  {S: &userID},
		}
	}

	input := &dynamodb.QueryInput{
		TableName:              aws.String(f.Config.EventFollowsTableName.Get()),
		KeyConditionExpression: aws.String("event_id = :event_id"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":event_id": {S: &eventID},
		},
		ProjectionExpression: aws.String("user_id"),
		ExclusiveStartKey:    startKey,
		Limit:                aws.Int64(int64(limit)),
		ConsistentRead:       &f.RequireConsistentReads,
	}
	req, output := f.Dynamo.QueryRequest(input)
	if err := service_common.ContextSend(ctx, req, f.Log); err != nil {
		return nil, errors.Wrapf(err, "could not get followers for event %v", eventID)
	}

	ret := PagedUserIDs{
		UserIDs: make([]string, 0, len(output.Items)),
	}
	for _, item := range output.Items {
		s, err := feedsdynamo.AwsStrings(item, []string{"user_id"})
		if err != nil {
			return nil, err
		}
		ret.UserIDs = append(ret.UserIDs, s["user_id"])
	}

	if output.LastEvaluatedKey != nil {
		s, err := feedsdynamo.AwsStrings(output.LastEvaluatedKey, []string{"user_id"})
		if err != nil {
			return nil, err
		}
		ret.Cursor = encodeCursor(s["user_id"])
	}

	return &ret, nil
}

func (f *EventFollows) GetFollowedEventsByUserID(ctx context.Context, userID string, limit int, cursor string) (*PagedEventIDs, error) {
	var startKey map[string]*dynamodb.AttributeValue
	if cursor != "" {
		eventID, err := decodeCursor(cursor)
		if err != nil {
			return nil, err
		}
		startKey = map[string]*dynamodb.AttributeValue{
			"event_id": {S: &eventID},
			"user_id":  {S: &userID},
		}
	}

	input := &dynamodb.QueryInput{
		TableName:              aws.String(f.Config.EventFollowsTableName.Get()),
		KeyConditionExpression: aws.String("user_id = :user_id"),
		IndexName:              aws.String("user_id-event_id-index"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":user_id": {S: &userID},
		},
		ProjectionExpression: aws.String("event_id"),
		ExclusiveStartKey:    startKey,
		Limit:                aws.Int64(int64(limit)),
	}
	req, output := f.Dynamo.QueryRequest(input)
	if err := service_common.ContextSend(ctx, req, f.Log); err != nil {
		return nil, errors.Wrapf(err, "could not get followed events for user %v", userID)
	}

	ret := PagedEventIDs{
		EventIDs: make([]string, 0, len(output.Items)),
	}
	for _, item := range output.Items {
		s, err := feedsdynamo.AwsStrings(item, []string{"event_id"})
		if err != nil {
			return nil, err
		}
		ret.EventIDs = append(ret.EventIDs, s["event_id"])
	}

	if output.LastEvaluatedKey != nil {
		s, err := feedsdynamo.AwsStrings(output.LastEvaluatedKey, []string{"event_id"})
		if err != nil {
			return nil, err
		}
		ret.Cursor = encodeCursor(s["event_id"])
	}

	return &ret, nil
}

func (f *EventFollows) GetFollowedEventsByUserIDAndEventIDs(ctx context.Context, userID string, eventIDs []string) ([]string, error) {
	getter := feedsdynamo.KeyedBatchGet{
		Dynamo:                 f.Dynamo,
		RequireConsistentReads: f.RequireConsistentReads,
		ElevatedLog:            f.Log,
	}

	items, err := getter.BatchGet(ctx, eventIDs, f.Config.EventFollowsTableName.Get(), func(eventID string) map[string]*dynamodb.AttributeValue {
		return map[string]*dynamodb.AttributeValue{
			"event_id": {S: &eventID},
			"user_id":  {S: &userID},
		}
	})
	if err != nil {
		return nil, err
	}

	ret := make([]string, 0, len(items))
	for _, item := range items {
		s, err := feedsdynamo.AwsStrings(item, []string{"event_id"})
		if err != nil {
			return nil, err
		}
		ret = append(ret, s["event_id"])
	}

	return ret, nil
}

// Follow returns true if the following relationship didn't exist previously and was established successfully.  False if
// the operation failed or if the user was already following the event.
func (f *EventFollows) Follow(ctx context.Context, eventID string, userID string) (bool, error) {
	createdAt := now()
	input := &dynamodb.PutItemInput{
		TableName: aws.String(f.Config.EventFollowsTableName.Get()),
		Item: map[string]*dynamodb.AttributeValue{
			"event_id":   {S: &eventID},
			"user_id":    {S: &userID},
			"created_at": timeToAttr(createdAt),
		},
		ConditionExpression: aws.String("attribute_not_exists(event_id)"),
	}

	req, _ := f.Dynamo.PutItemRequest(input)
	if err := service_common.ContextSend(ctx, req, f.Log); err != nil {
		if aerr, ok := errors.Cause(err).(awserr.Error); ok && aerr.Code() == dynamodb.ErrCodeConditionalCheckFailedException {
			return false, nil
		}
		return false, errors.Wrapf(err, "could not follow event %v by user %v", eventID, userID)
	}

	return true, nil
}

// Unfollow returns true if the event was previously followed and is not unfollowed.  False if the operation failed or
// if the user was never following the event.
func (f *EventFollows) Unfollow(ctx context.Context, eventID string, userID string) (bool, error) {
	input := &dynamodb.DeleteItemInput{
		TableName: aws.String(f.Config.EventFollowsTableName.Get()),
		Key: map[string]*dynamodb.AttributeValue{
			"event_id": {S: &eventID},
			"user_id":  {S: &userID},
		},
		ConditionExpression: aws.String("attribute_exists(event_id)"),
	}

	req, _ := f.Dynamo.DeleteItemRequest(input)
	if err := service_common.ContextSend(ctx, req, f.Log); err != nil {
		if strings.Contains(err.Error(), "ConditionalCheckFailedException") {
			return false, nil
		}
		return false, errors.Wrapf(err, "could not unfollow event %v by user %v", eventID, userID)
	}

	return true, nil
}

func timeToAttr(t time.Time) *dynamodb.AttributeValue {
	nano := strconv.FormatInt(t.UTC().UnixNano(), 10)
	return &dynamodb.AttributeValue{N: aws.String(nano)}
}
