package hypemanscheduler

import (
	"context"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/hypeman"
	jax "code.justin.tv/twitch-events/gea/internal/jax-client"
	"code.justin.tv/twitch-events/gea/internal/types"
)

const (
	DefaultOffsetStartMustAtOrAfter = time.Minute * time.Duration(30)
	DefaultOffsetStartMustBefore    = time.Second * time.Duration(10)
)

// SchedulerConfig contains config values for Scheduler.
type SchedulerConfig struct {
	// DBLimit defines the number of rows to load from the postgres db at a time.
	DBLimit *distconf.Int

	// OffsetStartMustAtOrAfter defines the max (inclusive) amount of time in the past an event's start time can be to
	// be considered relevant
	OffsetStartMustAtOrAfter *distconf.Duration

	// OffsetStartMustBefore defines the max (exclusive) amount of time in the future an event's start time can be to be
	// considered relevant
	OffsetStartMustBefore *distconf.Duration
}

// Load initializes SchedulerConfig to read configuration values from the given distconf.
func (c *SchedulerConfig) Load(dconf *distconf.Distconf) error {
	c.DBLimit = dconf.Int("hypeman.db_limit", db.MaxGetEventsIDs)
	c.OffsetStartMustAtOrAfter = dconf.Duration("hypeman.start_must_at_or_after", DefaultOffsetStartMustAtOrAfter)
	c.OffsetStartMustBefore = dconf.Duration("hypeman.start_must_before_or_after", DefaultOffsetStartMustBefore)

	return nil
}

// Scheduler facilitates queueing up notification jobs for events that are started.  These notification jobs are then
// fleshed out by hypeman worker and set to Pushy.
type Scheduler struct {
	OracleDB      db.DB
	JobStatus     *JobStatusClient
	Log           *log.ElevatedLog
	Config        *SchedulerConfig
	JaxClient     jax.Client
	HypemanClient hypeman.Client

	// NowFunc is an optional field that contains a function that returns the current time.  It is meant to
	// facilitate testing.
	NowFunc func() time.Time

	// CreateNotificationJobsDoneFunc is an optional field that contains a function that is called at the end of
	// CreateNotificationJobs.  It is meant to allow test code to wait on CreateNotificationJobs.
	CreateNotificationJobsDoneFunc func()
}

// CreateNotificationJobs finds live events that have recently started and queues up notification jobs for them.
func (s *Scheduler) CreateNotificationJobs(ctx context.Context) {
	err := s.innerCreateNotificationJobs(ctx)
	if err != nil {
		s.Log.LogCtx(ctx, err)
	}

	if s.CreateNotificationJobsDoneFunc != nil {
		s.CreateNotificationJobsDoneFunc()
	}
}

func (s *Scheduler) innerCreateNotificationJobs(ctx context.Context) error {
	var cursor string
	var err error
	var events []*db.Event

	hasMore := true

	for hasMore {
		events, cursor, err = s.getEventsThatStartedRecently(ctx, cursor)

		if err != nil {
			return err
		}
		hasMore = cursor != ""

		events, err = s.getEventsThatNeedNotifications(ctx, events)
		if err != nil {
			return err
		}

		eventIDs := make([]string, len(events))
		for i, event := range events {
			eventIDs[i] = event.ID
		}

		err = s.EnqueueNotificationJobs(ctx, eventIDs)
		if err != nil {
			return err
		}
	}

	return nil
}

func (s *Scheduler) getEventsThatStartedRecently(ctx context.Context, cursor string) ([]*db.Event, string, error) {
	limit := min(int(s.Config.DBLimit.Get()), db.MaxGetEventsIDs)

	now := s.now()
	start := now.Add(-s.Config.OffsetStartMustAtOrAfter.Get())
	end := now.Add(s.Config.OffsetStartMustBefore.Get())
	filter := db.BroadcastFilter{
		Types: []string{types.EventTypeSingle, types.EventTypeSegment},
		StartTimeWindow: &db.TimeWindow{
			Start: &start,
			End:   &end,
		},
		EndTimeWindow: &db.TimeWindow{
			Start: &now,
		},
	}

	response, err := s.OracleDB.GetEventIDsSortedByID(ctx, &filter, cursor, limit)
	if err != nil {
		return nil, "", err
	}

	eventIDs := response.EventIDs
	if len(eventIDs) == 0 {
		return []*db.Event{}, response.Cursor, nil
	}

	events, err := s.OracleDB.GetEvents(ctx, eventIDs, false)
	if err != nil {
		return nil, "", err
	}

	return events, response.Cursor, nil
}

func (s *Scheduler) getEventsThatNeedNotifications(ctx context.Context, events []*db.Event) ([]*db.Event, error) {
	eventIDs, eventIDToEvent := s.getEventIDsAndMap(events)

	eventIDs, err := s.JobStatus.NeedsJobs(ctx, eventIDs)
	if err != nil {
		return nil, err
	}

	eventIDs, err = s.getLiveEvents(ctx, eventIDs, eventIDToEvent)
	if err != nil {
		return nil, err
	}

	filteredEvents := make([]*db.Event, 0, len(eventIDs))
	for _, eventID := range eventIDs {
		filteredEvents = append(filteredEvents, eventIDToEvent[eventID])
	}

	return filteredEvents, nil
}

func (s *Scheduler) getLiveEvents(ctx context.Context, eventIDs []string, eventIDToEvent map[string]*db.Event) ([]string, error) {
	channelIDToEventIDs := make(map[string][]string, len(eventIDs))
	for _, eventID := range eventIDs {
		channelID := eventIDToEvent[eventID].ChannelID
		if channelID == nil || *channelID == "" {
			continue
		}

		channelIDToEventIDs[*channelID] = append(channelIDToEventIDs[*channelID], eventID)
	}

	channelIDs := make([]string, 0, len(channelIDToEventIDs))
	for channelID := range channelIDToEventIDs {
		channelIDs = append(channelIDs, channelID)
	}

	streams, err := s.JaxClient.GetStreamsByChannelIDs(ctx, channelIDs, nil, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could get get streams fom jax")
	}

	liveEvents := make([]string, 0, len(eventIDs))
	for _, stream := range streams.Hits {
		channelIDInt := stream.Properties.Rails.ChannelID
		channelID := strconv.Itoa(*channelIDInt)

		channelEvents, ok := channelIDToEventIDs[channelID]
		if !ok {
			s.Log.LogCtx(ctx, "got stream info on a channel we didn't request", "channelID", channelID)
			continue
		}

		liveEvents = append(liveEvents, channelEvents...)
	}

	return liveEvents, nil
}

func (s *Scheduler) EnqueueNotificationJobs(ctx context.Context, eventIDs []string) error {
	jobCreated := make([]bool, len(eventIDs))
	var wg sync.WaitGroup

	for i, eventID := range eventIDs {
		wg.Add(1)
		j := i
		innerEventID := eventID
		go func() {
			defer wg.Done()

			added, err := s.JobStatus.AddJob(ctx, innerEventID)
			if err != nil {
				s.Log.LogCtx(ctx,
					"error", err,
					"message", "could not add notification job to dynamo",
					"eventID", innerEventID)
				return
			}

			jobCreated[j] = added
		}()
	}
	wg.Wait()

	eventIDsToAdd := make([]string, 0, len(eventIDs))
	for i, eventID := range eventIDs {
		if jobCreated[i] {
			eventIDsToAdd = append(eventIDsToAdd, eventID)
		}
	}

	return s.HypemanClient.AddJobs(ctx, eventIDsToAdd)
}

func (s *Scheduler) getEventIDsAndMap(events []*db.Event) ([]string, map[string]*db.Event) {
	eventIDs := make([]string, 0, len(events))
	eventIDToEvent := make(map[string]*db.Event, len(events))

	for _, event := range events {
		eventIDs = append(eventIDs, event.ID)
		eventIDToEvent[event.ID] = event
	}

	return eventIDs, eventIDToEvent
}

func (s *Scheduler) now() time.Time {
	if s.NowFunc != nil {
		return s.NowFunc()
	}

	return time.Now()
}

func min(values ...int) int {
	if len(values) == 0 {
		return 0
	}

	minValue := values[0]
	for _, value := range values {
		if value < minValue {
			minValue = value
		}
	}

	return minValue
}
