package hypemanscheduler

import (
	"context"
	"strconv"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedsdynamo"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

// JobStatusClientConfig contains configuration for JobStatusClient.
type JobStatusClientConfig struct {
	notificationsTableName *distconf.Str
	ttl                    *distconf.Duration
}

// Load sets up JobStatusClientConfig to read configuration values from the given distconf.
func (c *JobStatusClientConfig) Load(dconf *distconf.Distconf) error {
	c.notificationsTableName = dconf.Str("gea.notifications_table", "")
	if c.notificationsTableName.Get() == "" {
		return errors.New("unable to find event notifications table")
	}

	week := time.Hour * time.Duration(24*7)
	c.ttl = dconf.Duration("gea.ttl", week)
	return nil
}

// JobStatusClient facilitates keeping track of whether notification jobs have been created for events that that are
// starting now.
type JobStatusClient struct {
	Dynamo *dynamodb.DynamoDB
	Config *JobStatusClientConfig
	Log    *log.ElevatedLog
}

// NeedsJobs takes a slice of event IDs and returns the event IDs that don't have notifications jobs.
func (c *JobStatusClient) NeedsJobs(ctx context.Context, eventIDs []string) ([]string, error) {
	getter := feedsdynamo.KeyedBatchGet{
		Dynamo:                 c.Dynamo,
		RequireConsistentReads: true,
		ElevatedLog:            c.Log,
	}

	items, err := getter.BatchGet(ctx, eventIDs, c.Config.notificationsTableName.Get(), func(eventID string) map[string]*dynamodb.AttributeValue {
		return map[string]*dynamodb.AttributeValue{
			"event_id": {S: &eventID},
		}
	})
	if err != nil {
		return nil, err
	}

	eventsThatHaveJobs := make(map[string]struct{}, len(items))
	for _, item := range items {
		s, err := feedsdynamo.AwsStrings(item, []string{"event_id"})
		if err != nil {
			return nil, err
		}

		eventsThatHaveJobs[s["event_id"]] = struct{}{}
	}

	eventsThatNeedJobs := make([]string, 0, len(eventIDs)-len(items))
	for _, eventID := range eventIDs {
		if _, ok := eventsThatHaveJobs[eventID]; !ok {
			eventsThatNeedJobs = append(eventsThatNeedJobs, eventID)
		}
	}

	return eventsThatNeedJobs, nil
}

// AddJob records that a notifications job will be started for the given event.  It returns false if a notifications
// job has already been started, and true otherwise.
func (c *JobStatusClient) AddJob(ctx context.Context, eventID string) (bool, error) {
	expiresAt := time.Now().Add(c.Config.ttl.Get())

	input := dynamodb.PutItemInput{
		TableName: aws.String(c.Config.notificationsTableName.Get()),
		Item: map[string]*dynamodb.AttributeValue{
			"event_id": {S: &eventID},
			"expires":  timeToAttributeValue(expiresAt),
		},
		ConditionExpression: aws.String("attribute_not_exists(event_id)"),
	}
	req, _ := c.Dynamo.PutItemRequest(&input)

	err := service_common.ContextSend(ctx, req, c.Log)
	if err != nil {
		aerr, ok := errors.Cause(err).(awserr.Error)
		if ok && aerr.Code() == dynamodb.ErrCodeConditionalCheckFailedException {
			return false, nil
		}
		return false, err
	}

	return true, nil
}

// GetLastSent returns when a notification was sent for the given container event, or nil if no entry for the
// container event exists.
func (c *JobStatusClient) GetLastSent(ctx context.Context, containerEventID string) (*time.Time, error) {
	input := dynamodb.GetItemInput{
		ConsistentRead: aws.Bool(true),
		Key: map[string]*dynamodb.AttributeValue{
			"event_id": {S: aws.String(containerEventID)},
		},
		TableName: aws.String(c.Config.notificationsTableName.Get()),
	}

	req, resp := c.Dynamo.GetItemRequest(&input)
	err := service_common.ContextSend(ctx, req, c.Log)
	if err != nil || resp.Item == nil {
		return nil, err
	}

	return getOptionalTime("last_sent", resp.Item)
}

// SetLastSent updates when a notification was sent for the given container event.  SetLastSent takes the last time
// that the table should currently have for the event, and does a compare and swap.  It returns true if the times
// match, and false otherwise.
func (c *JobStatusClient) SetLastSent(ctx context.Context, containerEventID string, prevLastSentPtr *time.Time, newLastSent time.Time) (bool, error) {
	expiresAt := time.Now().Add(c.Config.ttl.Get())

	input := dynamodb.PutItemInput{
		TableName: aws.String(c.Config.notificationsTableName.Get()),
		Item: map[string]*dynamodb.AttributeValue{
			"event_id":  {S: aws.String(containerEventID)},
			"expires":   timeToAttributeValue(expiresAt),
			"last_sent": timeToAttributeValue(newLastSent),
		},
	}

	if prevLastSentPtr == nil {
		input.ConditionExpression = aws.String("attribute_not_exists(last_sent)")
	} else {
		input.ConditionExpression = aws.String("last_sent = :prev_last_sent")
		input.ExpressionAttributeValues = map[string]*dynamodb.AttributeValue{
			":prev_last_sent": timeToAttributeValue(*prevLastSentPtr),
		}
	}

	req, _ := c.Dynamo.PutItemRequest(&input)

	err := service_common.ContextSend(ctx, req, c.Log)
	if err != nil {
		aerr, ok := errors.Cause(err).(awserr.Error)
		if ok && aerr.Code() == dynamodb.ErrCodeConditionalCheckFailedException {
			return false, nil
		}
		return false, err
	}

	return true, nil
}

// JobStatus contains the info in an item in the table.
type JobStatus struct {
	EventID   string
	ExpiresAt time.Time
	LastSent  *time.Time
}

func (c *JobStatusClient) GetJobStatus(ctx context.Context, eventID string) (*JobStatus, error) {
	input := dynamodb.GetItemInput{
		ConsistentRead: aws.Bool(true),
		Key: map[string]*dynamodb.AttributeValue{
			"event_id": {S: &eventID},
		},
		TableName: aws.String(c.Config.notificationsTableName.Get()),
	}

	req, resp := c.Dynamo.GetItemRequest(&input)
	err := service_common.ContextSend(ctx, req, c.Log)
	if err != nil {
		return nil, err
	} else if resp.Item == nil {
		return nil, nil
	}

	strAttributes, err := feedsdynamo.AwsStrings(resp.Item, []string{"event_id"})
	if err != nil {
		return nil, err
	}
	storedEventID := strAttributes["event_id"]

	intAttributes, err := feedsdynamo.AwsInts(resp.Item, []string{"expires"})
	if err != nil {
		return nil, err
	}
	expiresAtInt := intAttributes["expires"]

	lastSent, err := getOptionalTime("last_sent", resp.Item)
	if err != nil {
		return nil, err
	}

	return &JobStatus{
		EventID:   storedEventID,
		LastSent:  lastSent,
		ExpiresAt: time.Unix(expiresAtInt, 0),
	}, nil
}

func getOptionalTime(attributeName string, attributeValues map[string]*dynamodb.AttributeValue) (*time.Time, error) {
	value := attributeValues[attributeName]
	if value == nil {
		return nil, nil
	}

	if value.N == nil {
		return nil, errors.Errorf("key \"%s\" isn't a number", attributeName)
	}

	unixSecondsStr := *value.N
	unixSeconds, err := strconv.ParseInt(unixSecondsStr, 10, 64)
	if err != nil {
		return nil, errors.Wrapf(err, "value, \"%s\" associated with key, \"%s\" could not be converted to an integer", unixSecondsStr, attributeName)
	}

	t := time.Unix(unixSeconds, 0)
	return &t, nil
}

func timeToAttributeValue(t time.Time) *dynamodb.AttributeValue {
	s := strconv.FormatInt(t.Unix(), 10)
	return &dynamodb.AttributeValue{
		N: &s,
	}
}
