package hypemanworker

import (
	"context"
	"encoding/json"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
)

const (
	pushyEventStartName = "event_start"
)

type NotificationsSNSClient interface {
	Publish(ctx context.Context, notificationMsg *NotificationMessage) error
}

type NotificationsSNSClientConfig struct {
	SNSTopic *distconf.Str
}

func (c *NotificationsSNSClientConfig) Load(dconf *distconf.Distconf) error {
	c.SNSTopic = dconf.Str("gea.hypeman_worker.notifications_sns_topic", "")
	if c.SNSTopic.Get() == "" {
		return errors.New("unable to find a valid notifications SNS topic for hypeman worker")
	}

	return nil
}

type NotificationsSNSClientImpl struct {
	Config    *NotificationsSNSClientConfig
	SNSClient *sns.SNS
	Log       *log.ElevatedLog
}

func (c *NotificationsSNSClientImpl) Publish(ctx context.Context, notificationMsg *NotificationMessage) error {
	msgJSON, err := json.Marshal(notificationMsg)
	if err != nil {
		return err
	}

	messageAttributes := map[string]*sns.MessageAttributeValue{
		"event": {
			DataType:    aws.String("String"),
			StringValue: aws.String(pushyEventStartName),
		},
	}

	snsTopic := c.Config.SNSTopic.Get()
	publishInput := &sns.PublishInput{
		TopicArn:          aws.String(snsTopic),
		Message:           aws.String(string(msgJSON)),
		MessageAttributes: messageAttributes,
	}

	req, _ := c.SNSClient.PublishRequest(publishInput)
	if err := service_common.ContextSend(ctx, req, c.Log); err != nil {
		return errors.Wrapf(
			err,
			"could not publish notification msg to sns topic.  eventID: %s, snsTopic: %s",
			notificationMsg.EventID, snsTopic)
	}
	return nil
}
