package recipient

import (
	"context"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/gea/internal/follows"
	"code.justin.tv/twitch-events/gea/internal/types"
)

// TypedRecipientLoader facilitates fetching the users that should get notifications when a particular type
// of event goes live.
//
// Each event type so far has had unique business logic that dictate things like "whether the event's parent's
// subscribers should also get notifications".  This business logic is encapsulated in a struct that implements
// TypedRecipientLoader for a specific event type.  An instance of this struct is expected to be added to
// Loaders.TypedLoaders on service initialization.
type TypedRecipientLoader interface {
	Handles(eventType string) bool
	LoadRecipients(ctx context.Context, event types.TypedEvent) ([]string, error)
}

// Loaders facilitates fetching the users that should get notifications when a given event goes live.
// It defers loading to a event-type specific loader
type Loaders struct {
	TypedEventHandlers    *types.EventHandlers
	TypedRecipientLoaders []TypedRecipientLoader
}

func (l *Loaders) LoadRecipients(ctx context.Context, eventID string) ([]string, error) {

	event, err := l.TypedEventHandlers.GetEvent(ctx, eventID, false, false)
	if err != nil || event == nil {
		return nil, err
	}

	loader := l.getLoader(event.GetType())
	if loader == nil {
		return nil, errors.Errorf("could not find recipient loader for event type: %s, id: %s", event.GetType(), event.GetID())
	}

	return loader.LoadRecipients(ctx, event)
}

func (l *Loaders) getLoader(eventType string) TypedRecipientLoader {
	for _, typedLoader := range l.TypedRecipientLoaders {
		if typedLoader.Handles(eventType) {
			return typedLoader
		}
	}

	return nil
}

type BaseRecipientLoaderConfig struct {
	GetFollowersLimit *distconf.Int
}

// Load sets up BaseRecipientLoaderConfig to read configuration values from the given distconf.
func (c *BaseRecipientLoaderConfig) Load(dconf *distconf.Distconf) error {
	c.GetFollowersLimit = dconf.Int("gea.hypeman_worker.get_followers_limit", 1000)
	return nil
}

// BaseRecipientLoader contains common functionality that is useful to loading recipieints.
type BaseRecipientLoader struct {
	EventFollowsClient *follows.EventFollows
	Config             *BaseRecipientLoaderConfig
}

// Load the given event's followers into the given user ID map.
func (b *BaseRecipientLoader) loadEventFollowers(ctx context.Context, eventID string) ([]string, error) {
	userIDs := make([]string, 0)
	cursor := ""
	hasMore := true

	for hasMore && ctx.Err() == nil {
		limit := int(b.Config.GetFollowersLimit.Get())
		pagedUserIDs, err := b.EventFollowsClient.GetFollowersByEventID(ctx, eventID, limit, cursor)
		if err != nil {
			return nil, err
		}

		userIDs = append(userIDs, pagedUserIDs.UserIDs...)
		cursor = pagedUserIDs.Cursor
		hasMore = cursor != ""
	}

	return userIDs, nil
}

type stringSet struct {
	values map[string]struct{}
}

func newStringSet() *stringSet {
	return &stringSet{
		values: make(map[string]struct{}),
	}
}

func (s *stringSet) add(items ...string) {
	for _, item := range items {
		s.values[item] = struct{}{}
	}
}

func (s *stringSet) toSlice() []string {
	strSlice := make([]string, 0, len(s.values))
	for value := range s.values {
		strSlice = append(strSlice, value)
	}
	return strSlice
}
