package recipient

import (
	"context"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/twitch-events/gea/internal/clock"
	hypemanscheduler "code.justin.tv/twitch-events/gea/internal/hypeman-scheduler"
	"code.justin.tv/twitch-events/gea/internal/types"
)

type SegmentRecipientLoader struct {
	BaseRecipientLoader
	Log                log.Logger
	StatusClient       *hypemanscheduler.JobStatusClient
	Clock              clock.Clock
	TypedEventHandlers *types.EventHandlers
}

var _ TypedRecipientLoader = &SegmentRecipientLoader{}

func (s *SegmentRecipientLoader) Handles(eventType string) bool {
	return eventType == types.EventTypeSegment
}

// In general, when a segment event goes live, we want to send notifications to its followers.
// In addition, we want to send notification to the timetable's followers if the segment is the timetable's
// first segment to go live that day.  We want to restrict sending to a timetable's followers to once
// a day.
func (s *SegmentRecipientLoader) LoadRecipients(ctx context.Context, segmentTypedEvent types.TypedEvent) ([]string, error) {
	segmentEvent, ok := segmentTypedEvent.(*types.SegmentEvent)
	if !ok {
		return nil, errors.Errorf("expected a SegmentEvent but was given \"%T\"", segmentTypedEvent)
	}

	segmentFollowers, err := s.loadEventFollowers(ctx, segmentEvent.ID)
	if err != nil {
		return nil, err
	}

	shouldSendToParentFollowers, err := s.shouldSendToParentFollowers(ctx, segmentEvent)
	if err != nil {
		s.Log.Log(err)
		return segmentFollowers, nil
	}
	if !shouldSendToParentFollowers {
		return segmentFollowers, nil
	}

	timetableFollowers, err := s.loadEventFollowers(ctx, segmentEvent.ParentID)
	if err != nil {
		s.Log.Log(err)
		return segmentFollowers, nil
	}

	set := newStringSet()
	set.add(segmentFollowers...)
	set.add(timetableFollowers...)
	return set.toSlice(), nil
}

func (s *SegmentRecipientLoader) shouldSendToParentFollowers(ctx context.Context, segmentEvent *types.SegmentEvent) (bool, error) {
	parentID := segmentEvent.ParentID
	if parentID == "" {
		return false, nil
	}

	lastSentUTCPtr, err := s.StatusClient.GetLastSent(ctx, parentID)
	if err != nil {
		return false, err
	}

	nowUTC := s.Clock.NowUTC()
	first, err := s.isFirstSegmentOfTheDay(ctx, segmentEvent.TimeZoneID, nowUTC, lastSentUTCPtr)
	if err != nil || !first {
		return false, err
	}

	return s.StatusClient.SetLastSent(ctx, parentID, lastSentUTCPtr, nowUTC)
}

func (s *SegmentRecipientLoader) isFirstSegmentOfTheDay(ctx context.Context, timeZoneID string, nowUTC time.Time, lastSentUTCPtr *time.Time) (bool, error) {
	if lastSentUTCPtr == nil {
		// If there was no previous segment, then this is the first segment of the day.
		return true, nil
	}

	lastSentUTC := *lastSentUTCPtr
	if nowUTC.Before(lastSentUTC) {
		return false, nil
	}

	// Convert the times to the event's time zone.  This will affect whether two times fall on the same day.
	loc, err := time.LoadLocation(timeZoneID)
	if err != nil {
		return false, errors.Wrapf(err, "loading location for \"%s\" failed", timeZoneID)
	}
	lastSentLocal := lastSentUTC.In(loc)
	nowLocal := nowUTC.In(loc)

	startOfThatDay := time.Date(lastSentLocal.Year(), lastSentLocal.Month(), lastSentLocal.Day(), 0, 0, 0, 0, loc)
	startOfNextDay := startOfThatDay.Add(time.Hour * 24)

	nowAtOrAfterNextDay := !nowLocal.Before(startOfNextDay)

	return nowAtOrAfterNextDay, nil
}
