package hypemanworker

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/twitch-events/gea/internal/hypeman-worker/recipient"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/service-common/feedsqs"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/follows"
	"code.justin.tv/twitch-events/gea/internal/hypeman"
	"code.justin.tv/twitch-events/gea/internal/images"
	discovery "code.justin.tv/web/discovery/client"
	"github.com/aws/aws-sdk-go/service/sqs"
	net_context "golang.org/x/net/context"
)

const (
	twitchEventURLFormat = "https://www.twitch.tv/events/%s"
)

type WorkerConfig struct {
	feedsqs.SQSQueueProcessorConfig
}

// Load sets up WorkerConfig to read configuration values from the given distconf.
func (c *WorkerConfig) Load(dconf *distconf.Distconf) error {
	return c.SQSQueueProcessorConfig.Verify(dconf, "gea.hypeman_worker", time.Second*3, 2, time.Minute*15)
}

type Worker struct {
	feedsqs.SQSQueueProcessor
	Config *WorkerConfig

	DBClient               db.DB
	DiscoveryClient        discovery.Client
	EventFollowsClient     *follows.EventFollows
	ImageURLClient         *images.ImageURLClient
	NotificationsSNSClient NotificationsSNSClient
	RecipientLoaders       *recipient.Loaders
}

func (w *Worker) Setup() error {
	w.SQSQueueProcessor.ProcessMessage = w.processMessageWithStats
	return w.SQSQueueProcessor.Setup()
}

func (w *Worker) processMessageWithStats(ctx net_context.Context, m *sqs.Message) error {
	defer w.recordDuration("worker.process_message.duration", time.Now())
	w.Stats.IncC("worker.process_message.start", 1, 1)

	err := w.processMessage(m)
	if err != nil {
		w.Stats.IncC("worker.process_message.failed", 1, 1)
		return err
	}

	w.Stats.IncC("worker.process_message.succeeded", 1, 1)
	return nil
}

func (w *Worker) processMessage(m *sqs.Message) error {
	if m.Body == nil {
		return errors.New("received empty body message")
	}

	msg := &hypeman.NotificationJob{}
	if err := json.NewDecoder(strings.NewReader(*m.Body)).Decode(msg); err != nil {
		return errors.Wrap(err, "unable to parse SQS message body")
	}

	return w.ProcessNotificationJob(context.Background(), msg)
}

func (w *Worker) ProcessNotificationJob(ctx context.Context, notificationJob *hypeman.NotificationJob) error {
	eventID := notificationJob.EventID
	notificationMsg, err := w.hydrateNotificationMessage(ctx, eventID)
	if err != nil {
		return err
	}

	userIDs, err := w.RecipientLoaders.LoadRecipients(ctx, eventID)
	if err != nil {
		return err
	}

	for _, userID := range userIDs {
		go w.sendNotificationJobToPushy(context.Background(), *notificationMsg, userID)
	}

	return nil
}

func (w *Worker) hydrateNotificationMessage(ctx context.Context, eventID string) (*NotificationMessage, error) {
	event, err := w.DBClient.GetEvent(ctx, eventID, false)
	if err != nil {
		return nil, err
	} else if event == nil {
		return nil, nil
	}

	err = w.validateEvent(ctx, event)
	if err != nil {
		return nil, err
	}

	gameName, err := w.getGameName(ctx, event)
	if err != nil {
		return nil, err
	}

	imageURL := w.ImageURLClient.GetImageURL(event.CoverImageID)
	imageURL = strings.Replace(imageURL, "{width}", "338", 1)
	imageURL = strings.Replace(imageURL, "{height}", "190", 1)

	eventURL := fmt.Sprintf(twitchEventURLFormat, eventID)

	return &NotificationMessage{
		EventID:          event.ID,
		EventURL:         eventURL,
		EventImage:       &imageURL,
		EventTitle:       *event.Title,
		EventTime:        event.StartTime.Format(time.RFC3339),
		EventDescription: event.Description,
		EventGame:        gameName,
		ChannelID:        *event.ChannelID,

		NotificationType: NotificationType{
			Email:  true,
			Onsite: true,
			Mobile: true,
		},
	}, nil
}

func (w *Worker) validateEvent(ctx context.Context, event *db.Event) error {
	if event.Title == nil || *event.Title == "" {
		return errors.Errorf("cannot create notification for event, \"%s\", because title is empty", event.ID)
	}

	if event.StartTime == nil {
		return errors.Errorf("cannot create notification for event, \"%s\", because start time is nil", event.ID)
	}

	if event.ChannelID == nil {
		return errors.Errorf("cannot create notification for event, \"%s\", because channel ID is nil", event.ID)
	}

	return nil
}

func (w *Worker) getGameName(ctx context.Context, event *db.Event) (string, error) {
	if event.GameID == nil {
		return "", nil
	}

	game, err := w.DiscoveryClient.Get(ctx, *event.GameID, "", nil)
	if err != nil {
		return "", errors.Wrap(err, fmt.Sprintf("getting game associated with event failed; eventID: %s, gameID: %s", event.ID, *event.GameID))
	} else if game == nil {
		return "", nil
	}

	return game.Name, nil
}

func (w *Worker) sendNotificationJobToPushy(ctx context.Context, notificationMsg NotificationMessage, userID string) {
	notificationMsg.UserID = userID

	err := w.NotificationsSNSClient.Publish(ctx, &notificationMsg)
	if err != nil {
		w.Log.LogCtx(ctx, err)
		w.Stats.IncC("worker.send_to_pushy.failed", 1, 1)
		return
	}

	w.Stats.IncC("worker.send_to_pushy.succeeded", 1, 1)
}

func (w *Worker) recordDuration(stat string, start time.Time) {
	w.Stats.TimingDurationC(stat, time.Since(start), 1.0)
}
