package hypeman

import (
	"bytes"
	"context"
	"encoding/json"
	"strconv"

	"code.justin.tv/feeds/ctxlog"
	"code.justin.tv/feeds/ctxlog/ctxlogaws"
	"code.justin.tv/feeds/ctxlog/ctxlogaws/ctxlogsqs"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
)

// sqsBatchSize is the maximum number of messages that can be sent in a batch.  Value based on
// http://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessageBatch.html
const sqsBatchSize = 10

// Config holds configuration values for Client.
type Config struct {
	QueueURL *distconf.Str
}

// Load sets up Config to read configuration values from the given distconf.
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.QueueURL = dconf.Str("gea.hypeman_worker.sqssource.queue_url", "")
	if c.QueueURL.Get() == "" {
		return errors.New("unable to find a URL for the notifications SQS queue")
	}

	return nil
}

// NotificationJob is the payload that gets written to the SQS queue.
type NotificationJob struct {
	EventID string `json:"event_id"`
}

// Client facilitates enqueueing notification jobs.
type Client interface {
	AddJobs(ctx context.Context, eventIDs []string) error
}

// ClientImpl facilitates enqueueing notification jobs.
type ClientImpl struct {
	Sqs    *sqs.SQS
	Config *Config
	Log    *log.ElevatedLog
	CtxLog *ctxlog.Ctxlog
}

var _ Client = &ClientImpl{}

// AddJobs enqueues notification jobs for the given event IDs.
func (c *ClientImpl) AddJobs(ctx context.Context, eventIDs []string) error {
	jobs := make([]*NotificationJob, 0, len(eventIDs))
	for _, eventID := range eventIDs {
		jobs = append(jobs, &NotificationJob{
			EventID: eventID,
		})
	}

	return c.sendToSQS(ctx, jobs)
}

func (c *ClientImpl) sendToSQS(ctx context.Context, jobs []*NotificationJob) error {
	for start := 0; start < len(jobs); start += sqsBatchSize {
		nextStart := start + sqsBatchSize
		if nextStart > len(jobs) {
			nextStart = len(jobs)
		}

		err := c.sendBatchToSQS(ctx, jobs, start, nextStart)
		if err != nil {
			return nil
		}
	}
	return nil
}

func (c *ClientImpl) sendBatchToSQS(ctx context.Context, jobs []*NotificationJob, start, nextStart int) error {
	msgAttributes := ctxlogsqs.ModifyRequest(ctx, nil, c.CtxLog)

	requestEntries := make([]*sqs.SendMessageBatchRequestEntry, 0, len(jobs))
	for i := start; i < nextStart; i++ {
		msgBuffer := &bytes.Buffer{}
		if err := json.NewEncoder(msgBuffer).Encode(jobs[i]); err != nil {
			return err
		}
		msgBody := msgBuffer.String()

		requestEntries = append(requestEntries, &sqs.SendMessageBatchRequestEntry{
			Id:                aws.String(strconv.Itoa(i)),
			MessageBody:       &msgBody,
			MessageAttributes: msgAttributes,
		})
	}

	input := &sqs.SendMessageBatchInput{
		QueueUrl: aws.String(c.Config.QueueURL.Get()),
		Entries:  requestEntries,
	}
	req, _ := c.Sqs.SendMessageBatchRequest(input)
	req.HTTPRequest = req.HTTPRequest.WithContext(ctx)
	return ctxlogaws.DoAWSSend(req, c.Log)
}
