package images

import (
	"database/sql/driver"
	"encoding/json"
	"strings"

	"code.justin.tv/feeds/errors"
)

type ImageID struct {
	Type string
	ID   string
}

func (i *ImageID) String() string {
	return i.Type + ":" + i.ID
}

func (i ImageID) MarshalJSON() ([]byte, error) {
	return json.Marshal(i.String())
}

func (i *ImageID) UnmarshalJSON(b []byte) error {
	var s string
	if err := json.Unmarshal(b, &s); err != nil {
		return err
	}
	return i.fromString(s)
}

func (i *ImageID) Value() (driver.Value, error) {
	return i.String(), nil
}

func (i *ImageID) Scan(src interface{}) error {
	s, ok := src.(string)
	if !ok {
		return errors.Errorf("Cannot convert non string type to ImageID got %+v", src)
	}

	return i.fromString(s)
}

func GeaID(str string) *ImageID {
	return &ImageID{
		Type: "gea",
		ID:   str,
	}
}

func DefaultID(str string) *ImageID {
	return &ImageID{
		Type: "default",
		ID:   str,
	}
}

func IDFromString(str string) (*ImageID, error) {
	i := &ImageID{}
	err := i.fromString(str)
	if err != nil {
		return nil, err
	}

	return i, nil
}

func (i *ImageID) fromString(str string) error {
	parts := strings.SplitN(str, ":", 2)
	if len(parts) < 2 {
		return errors.Errorf("Unable to parse valid ImageID from string: %s", str)
	}
	i.Type = parts[0]
	i.ID = parts[1]
	return nil
}
