package images

import (
	"context"
	"encoding/json"
	"io"
	"net/http"
	"time"

	"code.justin.tv/common/golibs/pubsubclient"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/web/upload-service/rpc/uploader"
)

type SynchronousUploadClientConfig struct {
	Timeout      *distconf.Duration
	PubsubURL    *distconf.Str
	PubsubPrefix *distconf.Str
}

func (c *SynchronousUploadClientConfig) Load(dconf *distconf.Distconf) error {
	c.Timeout = dconf.Duration("gea.uploads.upload_timeout", 2*time.Second)
	c.PubsubURL = dconf.Str("gea.uploads.pubsub.url", "")
	if c.PubsubURL.Get() == "" {
		return errors.New("unable to find a valid pubsub url")
	}
	c.PubsubPrefix = dconf.Str("gea.uploads.pubsub.prefix", "pubsubtest.upload.")

	return nil
}

type SynchronousUploadClient struct {
	Config       *SynchronousUploadClientConfig
	S3HTTPClient *http.Client
	Log          *log.ElevatedLog
}

func (s *SynchronousUploadClient) UploadImage(ctx context.Context, uploadID, uploadURL string, imageData io.Reader) error {
	ctx, cancel := context.WithTimeout(ctx, s.Config.Timeout.Get())
	defer cancel()

	req, err := http.NewRequest(http.MethodPut, uploadURL, imageData)
	if err != nil {
		return errors.Wrap(err, "unable to create request for image upload to s3")
	}
	req = req.WithContext(ctx)

	// Create new pubsub client PER request so we don't eat messages for other uploads.
	// This is a cheap solution instead of creating a demuxer here.
	client := pubsubclient.New(s.Config.PubsubURL.Get(), nil)
	defer func() {
		retErr := client.Close()
		if retErr != nil {
			s.Log.LogCtx(ctx, "err", errors.Wrap(retErr, "closing the pubsub client failed"))
		}
	}()

	// Subscribe to pubsub BEFORE we upload
	pubsubTopic := s.Config.PubsubPrefix.Get() + uploadID
	err = client.Subscribe(pubsubTopic, "")
	if err != nil {
		return errors.Wrap(err, "unable to subscribe to topic")
	}

	resp, err := s.S3HTTPClient.Do(req)
	if err != nil {
		return errors.Wrap(err, "error uploading iamge to S3")
	}
	if resp.StatusCode != http.StatusOK {
		return errors.Errorf("error uploading image to S3: %s", resp.Status)
	}

	// Listen to pubsub for updates about upload complete
	for {
		msg, err := getMessage(ctx, client)
		if err != nil {
			return errors.Wrap(err, "error getting upload update")
		}
		resp := &uploader.StatusResponse{}
		if err := json.Unmarshal([]byte(msg.Message), resp); err != nil {
			return errors.Wrap(err, "unable to parse upload status message")
		}

		switch resp.Status {
		case uploader.Status_POSTPROCESS_COMPLETE:
			fallthrough
		case uploader.Status_COMPLETE:
			return nil
		case uploader.Status_IS_IMAGE_VALIDATION_FAILED:
			fallthrough
		case uploader.Status_IMAGE_FORMAT_VALIDATION_FAILED:
			return &service_common.CodedError{
				Code: http.StatusBadRequest,
				Err:  errors.New("image is invalid"),
			}
		default:
			if resp.Status >= 100 {
				return errors.New("unhandled status")
			}
		}
	}
}

func getMessage(ctx context.Context, client pubsubclient.Client) (*pubsubclient.Message, error) {
	var (
		msg *pubsubclient.Message
		err error
	)

	ch := make(chan struct{})
	go func() {
		msg, err = client.NextMessage()
		close(ch)
	}()

	select {
	case <-ch:
	case <-ctx.Done():
		return nil, errors.New("context canceled")
	}

	return msg, err
}
