package images

import (
	"encoding/json"
	"net/http"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedsqs"
	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/sqs"

	"golang.org/x/net/context"
)

type snsInputMessage struct {
	Type             string
	MessageID        string `json:"MessageId"`
	TopicArn         string
	Message          string
	Timestamp        string
	SignatureVersion string
	Signature        string
	SigningCertURL   string
	UnsubscribeURL   string
}

type ImageUploadServiceConfig struct {
	feedsqs.SQSQueueProcessorConfig

	uploadServiceURL        *distconf.Str
	uploadNotificationTopic *distconf.Str
	uploadS3Bucket          *distconf.Str
	uploadImageSizeLimit    *distconf.Str
	uploadImageFormat       *distconf.Str
}

func (c *ImageUploadServiceConfig) Load(dconf *distconf.Distconf) error {
	c.uploadServiceURL = dconf.Str("gea.uploads.url", "")
	if c.uploadServiceURL.Get() == "" {
		return errors.New("unable to find a valid upload-service url")
	}

	c.uploadNotificationTopic = dconf.Str("gea.uploads.callback_arn", "")
	if c.uploadServiceURL.Get() == "" {
		return errors.New("unable to find the SNS topic ARN that the upload-service is to publish to")
	}

	c.uploadS3Bucket = dconf.Str("gea.uploads.s3bucket", "")
	if c.uploadServiceURL.Get() == "" {
		return errors.New("unable to find a valid S3 bucket name for uploads")
	}

	c.uploadImageSizeLimit = dconf.Str("gea.uploads.image_size_limit", "1MB")
	c.uploadImageFormat = dconf.Str("gea.uploads.image_format", "image")

	return c.SQSQueueProcessorConfig.Verify(dconf, "gea.uploads", time.Second*3, 2, time.Minute*15)
}

type ImageUploadService struct {
	feedsqs.SQSQueueProcessor

	Config       *ImageUploadServiceConfig
	S3           *s3.S3
	SyncUploader *SynchronousUploadClient

	client uploader.Uploader
}

func (i *ImageUploadService) ReserveImageUpload(ctx context.Context) (*uploader.UploadResponse, error) {
	resp, err := i.client.Create(ctx, &uploader.UploadRequest{
		Outputs: []*uploader.Output{
			{
				Name: "{{UploadID}}",
			},
		},
		PreValidation: &uploader.Validation{
			FileSizeLessThan: i.Config.uploadImageSizeLimit.Get(),
			Format:           i.Config.uploadImageFormat.Get(),
		},
		OutputPrefix: "s3://" + i.Config.uploadS3Bucket.Get() + "/uploads/",
		Callback: &uploader.Callback{
			SnsTopicArn: i.Config.uploadNotificationTopic.Get(),
			PubsubTopic: "default",
		},
		//TODO: Monitoring
	})

	if err != nil {
		return nil, err
	}

	return resp, nil
}

func (i *ImageUploadService) SaveUploadedImage(ctx context.Context, imageID string) (*ImageID, error) {
	bucketName := i.Config.uploadS3Bucket.Get()

	copyReq, _ := i.S3.CopyObjectRequest(&s3.CopyObjectInput{
		ACL:        aws.String("public-read"),
		Bucket:     aws.String(bucketName),
		CopySource: aws.String(bucketName + "/uploads/" + imageID),
		Key:        aws.String("saved/" + imageID),
	})
	if err := service_common.ContextSend(ctx, copyReq, i.Log); err != nil {
		return nil, errors.Wrap(err, "could not copy image to long term storage")
	}

	return GeaID(imageID), nil
}

func (i *ImageUploadService) PickDefaultImage(ctx context.Context) (*ImageID, error) {
	// TODO: Make configurable, or just pick one randomly from the /default/ bucket
	return DefaultID("town"), nil
}

func (i *ImageUploadService) processMessage(ctx context.Context, m *sqs.Message) error {
	if m.Body == nil {
		return errors.New("Received empty body message")
	}

	msg := &snsInputMessage{}
	if err := json.NewDecoder(strings.NewReader(*m.Body)).Decode(msg); err != nil {
		return errors.Wrap(err, "Unable to parse SNS message type")
	}

	if msg.Message == "" {
		return errors.New("Received empty SNS message")
	}

	uploaderCallback := &models.SNSCallback{}
	if err := json.NewDecoder(strings.NewReader(msg.Message)).Decode(uploaderCallback); err != nil {
		return errors.Wrap(err, "Unable to parse message from uploader service")
	}

	if uploaderCallback.Status != int64(uploader.Status_POSTPROCESS_COMPLETE) {
		i.Log.Log("UploadID", uploaderCallback.UploadID, "status", uploaderCallback.Status, "Unhandled status type")
		return nil
	}

	if _, err := i.client.SetStatus(ctx, &uploader.SetStatusRequest{
		UploadId: uploaderCallback.UploadID,
		Status:   uploader.Status_COMPLETE,
	}); err != nil {
		return errors.Wrap(err, "Error settings success status on image upload")
	}

	return nil
}

func (i *ImageUploadService) Setup() error {
	i.client = uploader.NewUploaderProtobufClient(i.Config.uploadServiceURL.Get(), &http.Client{})
	i.SQSQueueProcessor.ProcessMessage = i.processMessage

	return i.SQSQueueProcessor.Setup()
}
