package pubsub

import (
	"encoding/json"
	"errors"
	"time"

	pubsub "code.justin.tv/chat/pubsub-go-pubclient/client"
	"code.justin.tv/foundation/twitchclient"

	"golang.org/x/net/context"
)

const channelEventTopicPrefix = "channel-event-updates."

// PubSub is an interface based on chat's go-pubclient
type PubSub interface {
	Publish(context.Context, []string, string, *twitchclient.ReqOpts) error
}

// Client used within gea
type Client interface {
	PublishCurrentChannelEvent(ctx context.Context, targetChannelID string, currentEvent *CurrentEvent) error
}

type pubsubImpl struct {
	baseClient PubSub
}

// NewPubSubClient creates a new client for use within gea
func NewPubSubClient(host string) (Client, error) {
	clientConf := twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
		Host: host,
	}

	pubsubClient, err := pubsub.NewPubClient(clientConf)
	if err != nil {
		return nil, errors.New("failed to start pubsub client")
	}

	return &pubsubImpl{baseClient: pubsubClient}, nil
}

// CurrentChannelEvent represents a live event message for a channel
type CurrentChannelEvent struct {
	CurrentEvent *CurrentEvent `json:"current_event"`
	Timestamp    time.Time     `json:"timestamp"`
	Type         string        `json:"type"`
}

// CurrentEvent represents the event currently live on a given channel
type CurrentEvent struct {
	EventID string        `json:"event_id"`
	Type    string        `json:"type"`
	Title   string        `json:"title"`
	Parent  *CurrentEvent `json:"parent"`
}

func (p *pubsubImpl) PublishCurrentChannelEvent(ctx context.Context, targetChannelID string, currentEvent *CurrentEvent) error {
	topics := []string{channelEventTopicPrefix + targetChannelID}

	currentChannelEvent := CurrentChannelEvent{
		CurrentEvent: currentEvent,
		Timestamp:    time.Now().UTC(),
		Type:         "live-event",
	}

	pubsubMessage, err := json.Marshal(currentChannelEvent)
	if err != nil {
		return err
	}

	return p.baseClient.Publish(ctx, topics, string(pubsubMessage), nil)
}
