package testutils

import (
	"context"
	"sync"
	"time"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/vod/vodapi/rpc/vodapi"
	"code.justin.tv/web/users-service/client/channels"
	usersmodels "code.justin.tv/web/users-service/models"
)

// StubClock contains a test-defined ControlledNow time which will be returned
type StubClock struct {
	ControlledNow time.Time
	lock          sync.RWMutex
}

// NowUTC as implemented by StubClock returns the ControlledNow time as UTC
func (s *StubClock) NowUTC() time.Time {
	s.lock.RLock()
	defer s.lock.RUnlock()

	return s.ControlledNow.UTC()
}

func (s *StubClock) SetNow(now time.Time) {
	s.lock.Lock()
	s.ControlledNow = now
	s.lock.Unlock()
}

// GetChannelClientStub contains a test-defined GameID which will be returned in the channel struct
type GetChannelClientStub struct {
	GameID int
}

// NewGetChannelClientStub creates a GetChannelClientStub with a defined gameID
func NewGetChannelClientStub(gameID int) *GetChannelClientStub {
	return &GetChannelClientStub{
		GameID: gameID,
	}
}

// GetByIDAndParams returns a channel with the requested GameID
func (c *GetChannelClientStub) GetByIDAndParams(ctx context.Context, id string, params *usersmodels.ChannelFilterParams, reqOpts *twitchclient.ReqOpts) (*channels.Channel, error) {
	return &channels.Channel{
		GameID: c.GameID,
	}, nil
}

// StubVodAPIClient implements video.VodAPIClient and allows getting a specified page of VODs from an internal list.
type StubVodAPIClient struct {
	rwMutex sync.RWMutex
	vods    []*vodapi.Vod
}

// PublicGetVodsByUser returns the specified page of VODs from its internal list.  It does not do any sorting or filtering.
func (m *StubVodAPIClient) PublicGetVodsByUser(ctx context.Context, input *vodapi.PublicGetVodsByUserRequest) (*vodapi.PublicGetVodsByUserResponse, error) {
	m.rwMutex.RLock()
	defer m.rwMutex.RUnlock()

	page := make([]*vodapi.Vod, 0, input.Limit)
	count := 0
	for i := int(input.Offset); i < len(m.vods); i++ {
		if count >= int(input.Limit) {
			break
		}

		page = append(page, m.vods[i])
		count++
	}

	return &vodapi.PublicGetVodsByUserResponse{
		Vods: page,
	}, nil
}

// SetVods updates the internal list of VODs.
func (m *StubVodAPIClient) SetVods(vods []*vodapi.Vod) {
	m.rwMutex.Lock()
	defer m.rwMutex.Unlock()

	m.vods = make([]*vodapi.Vod, len(vods))
	copy(m.vods, vods)
}
