package types

import (
	"context"

	"strings"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/twitch-events/gea/internal/db"
)

type BaseEventHandler struct {
	OracleDB db.DB
	Log      log.Logger
}

func (h *BaseEventHandler) validateLocalizationUpdate(ctx context.Context, event *db.Event, params *LocalizationUpdateParams, isCreate bool) (*Localization, error) {
	if params.Language == "" {
		return nil, invalid("language", "should have a language")
	}
	if params.Title == nil && params.Description == nil && params.ChannelID == nil {
		return nil, invalid("", "must specify title, description, or channel for localization")
	}

	if event.Language != nil && *event.Language == params.Language {
		return nil, invalid("language", "event is already in this language")
	}
	if event.ChannelID != nil && params.ChannelID != nil && *event.ChannelID == *params.ChannelID {
		return nil, invalid("channel", "event is already on this channel")
	}

	localizations, err := h.OracleDB.GetLocalizationsByEventID(ctx, params.EventID)
	if err != nil {
		return nil, err
	}
	var existingLocalization *Localization
	for _, loc := range localizations {
		if loc.Language == params.Language {
			existingLocalization = fromDBLocalization(loc)
		} else if loc.ChannelID != nil && params.ChannelID != nil && *loc.ChannelID == *params.ChannelID {
			return nil, invalid("channel", "localization for this channel already exists")
		}
	}
	if isCreate && existingLocalization != nil {
		return nil, invalid("language", "localization already exists")
	}
	if !isCreate && existingLocalization == nil {
		return nil, invalid("language", "localization does not exist")
	}

	return existingLocalization, nil
}

func (h *BaseEventHandler) AddLocalizationCommon(ctx context.Context, event *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	dbLocalization, err := h.OracleDB.CreateLocalization(ctx, toUpdateDBLocalizationParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error creating localization")
	}
	return fromDBLocalization(dbLocalization), nil
}

func (h *BaseEventHandler) UpdateLocalizationCommon(ctx context.Context, event *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	dbLocalization, err := h.OracleDB.UpdateLocalization(ctx, params.EventID, params.Language, toUpdateDBLocalizationParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error updating localization")
	}
	return fromDBLocalization(dbLocalization), nil
}

func (h *BaseEventHandler) RemoveLocalizationCommon(ctx context.Context, eventID string, language string) (*Localization, error) {
	dbLocalization, err := h.OracleDB.DeleteLocalization(ctx, eventID, language)
	if err != nil {
		return nil, errors.Wrap(err, "error removing localization")
	}
	return fromDBLocalization(dbLocalization), nil
}

const defaultTimeZone = "America/Los_Angeles"

func (h *BaseEventHandler) LoadTimeLocation(ctx context.Context, timeZone string) *time.Location {
	if timeZone == "" {
		timeZone = defaultTimeZone
	}
	if !isTimeZoneValid(timeZone) {
		timeZone = defaultTimeZone
	}

	location, err := time.LoadLocation(timeZone)
	if err != nil {
		h.Log.Log("message", "loading timezone location failed", "timezone", timeZone, "err", err)
		return time.UTC
	}

	return location
}

func isTimeZoneValid(timeZone string) bool {
	upperTimeZone := strings.ToUpper(timeZone)
	if upperTimeZone == "UTC" || upperTimeZone == "LOCAL" {
		return false
	}

	loc, err := time.LoadLocation(timeZone)
	return err == nil && loc != nil
}
