package types

import (
	"bytes"
	"fmt"
	"time"

	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/images"
)

type ValidationError struct {
	Field  string
	Reason string
}

func (e ValidationError) Error() string {
	return fmt.Sprintf("validation error: %v - %v", e.Field, e.Reason)
}

func invalid(field string, reason string) error {
	return ValidationError{Field: field, Reason: reason}
}

type EventUpdateParams struct {
	OwnerID  string
	Type     string
	ParentID *string

	StartTime  *time.Time
	EndTime    *time.Time
	TimeZoneID *string

	CoverImageID *images.ImageID
	Language     *string
	Title        *string
	Description  *string
	ChannelID    *string
	GameID       *string

	// Tracking fields
	Timestamp time.Time

	// Premiere fields
	PremiereID *string
}

type EventDeleteParams struct {
	// Tracking fields
	Timestamp time.Time
}

func toCreateDBEventParams(params *EventUpdateParams) *db.CreateDBEventParams {
	return &db.CreateDBEventParams{
		OwnerID:  params.OwnerID,
		Type:     params.Type,
		ParentID: params.ParentID,

		StartTime:  params.StartTime,
		EndTime:    params.EndTime,
		TimeZoneID: params.TimeZoneID,

		CoverImageID: params.CoverImageID,
		Language:     params.Language,
		Title:        params.Title,
		Description:  params.Description,
		ChannelID:    params.ChannelID,
		GameID:       params.GameID,

		Timestamp: params.Timestamp,
	}
}

func toUpdateDBEventParams(params *EventUpdateParams) *db.UpdateDBEventParams {
	return &db.UpdateDBEventParams{
		OwnerID:  params.OwnerID,
		Type:     params.Type,
		ParentID: params.ParentID,

		StartTime:  params.StartTime,
		EndTime:    params.EndTime,
		TimeZoneID: params.TimeZoneID,

		CoverImageID: params.CoverImageID,
		Language:     params.Language,
		Title:        params.Title,
		Description:  params.Description,
		ChannelID:    params.ChannelID,
		GameID:       params.GameID,

		Timestamp: params.Timestamp,
	}
}

func toDeleteDBEventParams(params *EventDeleteParams) *db.DeleteDBEventParams {
	return &db.DeleteDBEventParams{
		Timestamp: params.Timestamp,
	}
}

type LocalizationUpdateParams struct {
	EventID  string
	Language string

	Title       *string
	Description *string
	ChannelID   *string
}

func toUpdateDBLocalizationParams(params *LocalizationUpdateParams) *db.UpdateDBLocalizationParams {
	return &db.UpdateDBLocalizationParams{
		EventID:  params.EventID,
		Language: params.Language,

		Title:       params.Title,
		Description: params.Description,
		ChannelID:   params.ChannelID,
	}
}

func coalesce(str *string, defaultValue string) string {
	if str != nil {
		return *str
	}
	return defaultValue
}

func keys(m map[string]struct{}) []string {
	keys := make([]string, 0, len(m))
	for k := range m {
		keys = append(keys, k)
	}
	return keys
}

func in(str string, in []string) bool {
	for _, v := range in {
		if v == str {
			return true
		}
	}
	return false
}

func addPrefix(prefix string, data []byte) []byte {
	buf := make([]byte, len(data)+len(prefix))
	copy(buf, prefix)
	copy(buf[len(prefix):], data)
	return buf
}

func removePrefix(prefix string, data []byte) ([]byte, bool) {
	if len(prefix) > len(data) {
		return nil, false
	}
	if !bytes.Equal([]byte(prefix), data[0:len(prefix)]) {
		return nil, false
	}
	return data[len(prefix):], true
}

type Localization struct {
	EventID  string `json:"event_id"`
	Language string `json:"language"`

	CreatedAt time.Time `json:"created_at"`
	UpdatedAt time.Time `json:"updated_at"`

	Title       *string `json:"title,omitempty"`
	Description *string `json:"description,omitempty"`
	ChannelID   *string `json:"channel_id,omitempty"`
}

func fromDBLocalization(dbLocalization *db.Localization) *Localization {
	return &Localization{
		EventID:  dbLocalization.EventID,
		Language: dbLocalization.Language,

		CreatedAt: dbLocalization.CreatedAt,
		UpdatedAt: dbLocalization.UpdatedAt,

		Title:       dbLocalization.Title,
		Description: dbLocalization.Description,
		ChannelID:   dbLocalization.ChannelID,
	}
}
