package types

import (
	"context"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/gomemcache/memcache"
	"code.justin.tv/twitch-events/gea/internal/cache"
)

type EventCacheConfig struct {
	EventCacheDuration *distconf.Duration
}

func (c *EventCacheConfig) Load(dconf *distconf.Distconf) error {
	c.EventCacheDuration = dconf.Duration("gea.event_cache_duration", 5*time.Minute)
	return nil
}

type EventCache struct {
	Config *EventCacheConfig
	Cache  *cache.Service
	Stats  *service_common.StatSender
	Log    *log.ElevatedLog
}

func (c *EventCache) eventKey(eventID string) string {
	return "gea:ev:" + eventID
}

func (c *EventCache) GetEvents(ctx context.Context, eventIDs []string, unmarshaller func(data []byte) (TypedEvent, error)) []TypedEvent {
	startTime := time.Now()
	defer func() {
		c.Stats.TimingDurationC("get.timing", time.Since(startTime), 0.1)
	}()

	keys := make([]string, len(eventIDs))
	for i, id := range eventIDs {
		keys[i] = c.eventKey(id)
	}
	cached, err := c.Cache.GetMulti(ctx, keys)
	if err != nil {
		c.Log.Log("err", err, "keys", keys, "could not get events from cache")
		return nil
	}
	events := make([]TypedEvent, 0, len(cached))
	for key, item := range cached {
		event, err := unmarshaller(item.Value)
		if err != nil {
			c.Log.Log("err", err, "key", key, "could not unmarshall event from cache")
			continue
		}
		if event != nil {
			events = append(events, event)
		}
	}
	return events
}

func (c *EventCache) CacheEvents(ctx context.Context, events []TypedEvent, marshaller func(event TypedEvent) ([]byte, error)) {
	startTime := time.Now()
	defer func() {
		c.Stats.TimingDurationC("set.timing", time.Since(startTime), 0.1)
	}()

	for _, event := range events {
		key := c.eventKey(event.GetID())
		data, err := marshaller(event)
		if err != nil {
			c.Log.Log("err", err, "key", key, "could not marshall event for cache")
			continue
		}
		cacheErr := c.Cache.Set(ctx, &memcache.Item{
			Key:        key,
			Value:      data,
			Expiration: int32(c.Config.EventCacheDuration.Get().Seconds()),
		})
		if cacheErr != nil {
			c.Log.Log("err", cacheErr, "key", key, "could not cache event")
			continue
		}
	}
}

func (c *EventCache) InvalidateEvent(ctx context.Context, eventID string) {
	key := c.eventKey(eventID)
	err := c.Cache.Delete(ctx, key)
	if err != nil {
		if err != memcache.ErrCacheMiss {
			c.Log.Log("err", err, "key", key, "could not invalidate event in cache")
		}
	}
}
