package types

import (
	"context"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/filter"
	"code.justin.tv/twitch-events/gea/internal/video"
)

type TypedEvent interface {
	GetType() string
	GetID() string
	GetChannelIDs() []string
	GetTitle() string
	GetParentID() string
	GetOwnerID() string
}

type EventHandler interface {
	Handles(eventType string) bool

	GetEvents(ctx context.Context, dbEvent []*db.Event) ([]TypedEvent, error)
	GetEventMetadata(ctx context.Context, dbEvent *db.Event) (*EventMetadata, error)

	CreateEvent(ctx context.Context, params *EventUpdateParams) (TypedEvent, error)
	UpdateEvent(ctx context.Context, oldEvent *db.Event, params *EventUpdateParams) (TypedEvent, error)
	DeleteEvent(ctx context.Context, event *db.Event, params *EventDeleteParams) (TypedEvent, error)

	MarshallEvent(event TypedEvent) ([]byte, error)
	UnmarshallEvent(data []byte) TypedEvent

	AddLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error)
	UpdateLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error)
	RemoveLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error)

	GetArchiveVideos(ctx context.Context, event TypedEvent) ([]video.ArchiveVideo, error)
}

type EventHandlers struct {
	Filter   *filter.Filter
	OracleDB db.DB

	Handlers            []EventHandler
	EventCache          *EventCache
	MetadataCache       *MetadataCache
	VideoCache          *video.Cache
	Log                 log.Logger
	LoadEventsThrottler *LoadEventsThrottler
}

func (h *EventHandlers) getHandlerForType(eventType string) (EventHandler, error) {
	for _, handler := range h.Handlers {
		if handler.Handles(eventType) {
			return handler, nil
		}
	}
	return nil, errors.Errorf("unrecognized event type: %v", eventType)
}

type EventMetadata struct {
	ID          string `json:"id"`
	Title       string `json:"title"`
	Description string `json:"description"`

	// Open Graph properties
	OGTitle       string `json:"og_title"`
	OGDescription string `json:"og_description"`
	OGURL         string `json:"og_url"`
	OGImage       string `json:"og_image"`
	OGImageWidth  string `json:"og_image_width"`
	OGImageHeight string `json:"og_image_height"`
	OGType        string `json:"og_type"`

	TwitterCard        string `json:"twitter_card"`
	TwitterTitle       string `json:"twitter_title"`
	TwitterDescription string `json:"twitter_description"`
	TwitterImage       string `json:"twitter_image"`
	TwitterURL         string `json:"twitter_url"`
}

func (h *EventHandlers) GetEventMetadata(ctx context.Context, eventID string, skipCache bool) (*EventMetadata, error) {
	if !skipCache {
		eventMetadata := h.MetadataCache.GetMetdata(ctx, eventID)
		if eventMetadata != nil {
			return eventMetadata, nil
		}
	}

	events, err := h.OracleDB.GetEvents(ctx, []string{eventID}, false)
	if err != nil {
		return nil, err
	}
	if len(events) == 0 {
		return nil, nil
	}
	event := events[0]
	handler, err := h.getHandlerForType(event.Type)
	if err != nil {
		return nil, err
	}
	eventMetadata, err := handler.GetEventMetadata(ctx, event)
	if err != nil {
		return nil, err
	}
	h.MetadataCache.CacheMetadata(ctx, eventMetadata)

	return eventMetadata, nil
}

func (h *EventHandlers) GetEvent(ctx context.Context, eventID string, getDeleted bool, skipCache bool) (TypedEvent, error) {
	if eventID == "" {
		return nil, nil
	}

	events, err := h.GetEvents(ctx, []string{eventID}, getDeleted, skipCache)
	if err != nil || len(events) == 0 {
		return nil, err
	} else if len(events) > 1 {
		return nil, errors.Errorf("GetEvents loaded %d events instead of an event for \"%s\"", len(events), eventID)
	}

	event := events[0]
	if event.GetID() != eventID {
		return nil, errors.Errorf("GetEvents loaded event, \"%s\", instead of the event for \"%s\"", event.GetID(), eventID)
	}

	return event, nil
}

func (h *EventHandlers) GetEvents(ctx context.Context, eventIDs []string, getDeleted bool, skipCache bool) ([]TypedEvent, error) {
	eventIDToEvent := make(map[string]TypedEvent, len(eventIDs))

	eventIDsToGet := make(map[string]struct{}, len(eventIDs))
	for _, eventID := range eventIDs {
		eventIDsToGet[eventID] = struct{}{}
	}
	if !skipCache {
		cachedEvents := h.EventCache.GetEvents(ctx, eventIDs, h.unmarshallEvent)
		for _, event := range cachedEvents {
			eventIDToEvent[event.GetID()] = event
			delete(eventIDsToGet, event.GetID())
		}
	}

	idsToGet := make([]string, 0, len(eventIDsToGet))
	for eventID := range eventIDsToGet {
		idsToGet = append(idsToGet, eventID)
	}

	// Load the remaining events from the DB.
	if len(idsToGet) > 0 {
		var fetchedEvents []TypedEvent
		var err error
		if skipCache {
			fetchedEvents, err = h.loadEventsFromDB(ctx, idsToGet, getDeleted)
		} else {
			fetchedEvents, err = h.loadEventsFromThrottledDB(ctx, idsToGet, getDeleted)
		}
		if err != nil {
			return nil, err
		}

		for _, fetchedEvent := range fetchedEvents {
			eventIDToEvent[fetchedEvent.GetID()] = fetchedEvent
		}
	}

	// Populate events in the order the event IDs were given.
	events := make([]TypedEvent, 0, len(eventIDToEvent))
	for _, eventID := range eventIDs {
		event := eventIDToEvent[eventID]
		if event != nil {
			events = append(events, event)
		}
	}

	return events, nil
}

// loadEventsFromThrottledDB combines identical requests over a second to a single request to the DB.
func (h *EventHandlers) loadEventsFromThrottledDB(ctx context.Context, eventIDs []string, getDeleted bool) ([]TypedEvent, error) {
	loadResult, shouldLoadFromDB := h.LoadEventsThrottler.Get(eventIDs, getDeleted)
	if !shouldLoadFromDB {
		return loadResult.WaitAndGetResult(ctx)
	}

	typedEvents, err := h.loadEventsFromDB(ctx, eventIDs, getDeleted)
	loadResult.SetResult(typedEvents, err)

	return typedEvents, err
}

// loadEventsFromDB loads events from the DB, converts them to TypedEvents and caches the results.
func (h *EventHandlers) loadEventsFromDB(ctx context.Context, eventIDs []string, getDeleted bool) ([]TypedEvent, error) {
	dbEvents, err := h.OracleDB.GetEvents(ctx, eventIDs, getDeleted)
	if err != nil {
		return nil, err
	}

	dbEvents = h.Filter.FilterEvents(ctx, dbEvents)

	// If we're fetching deleted events, don't cache the results.
	shouldCache := !getDeleted
	return h.convertDBEvents(ctx, dbEvents, shouldCache)
}

func (h *EventHandlers) convertDBEvents(ctx context.Context, dbEvents []*db.Event, cacheResults bool) ([]TypedEvent, error) {
	types := map[string][]*db.Event{}
	for _, dbEvent := range dbEvents {
		if _, ok := types[dbEvent.Type]; !ok {
			types[dbEvent.Type] = make([]*db.Event, 0, len(dbEvents))
		}
		types[dbEvent.Type] = append(types[dbEvent.Type], dbEvent)
	}
	res := make([]TypedEvent, 0, len(dbEvents))
	for t, events := range types {
		handler, err := h.getHandlerForType(t)
		if err != nil {
			return nil, err
		}
		evs, err := handler.GetEvents(ctx, events)
		if err != nil {
			// We may want this to return whatever data it can instead of erroring out completely.
			// Let's keep an eye on this and see if it becomes a problem. - Oct 1st, 2017
			return nil, err
		}
		if cacheResults {
			h.EventCache.CacheEvents(ctx, evs, handler.MarshallEvent)
		}
		res = append(res, evs...)
	}
	return res, nil
}

func (h *EventHandlers) unmarshallEvent(data []byte) (TypedEvent, error) {
	for _, handler := range h.Handlers {
		ev := handler.UnmarshallEvent(data)
		if ev != nil {
			return ev, nil
		}
	}
	return nil, errors.New("could not unmarshall event data")
}

func (h *EventHandlers) CreateEvent(ctx context.Context, params *EventUpdateParams) (TypedEvent, error) {
	handler, err := h.getHandlerForType(params.Type)
	if err != nil {
		return nil, err
	}
	return handler.CreateEvent(ctx, params)
}

func (h *EventHandlers) UpdateEvent(ctx context.Context, oldEvent *db.Event, params *EventUpdateParams) (TypedEvent, error) {
	handler, err := h.getHandlerForType(params.Type)
	if err != nil {
		return nil, err
	}
	ev, err := handler.UpdateEvent(ctx, oldEvent, params)
	if err != nil {
		return nil, err
	}
	h.EventCache.InvalidateEvent(ctx, ev.GetID())
	h.MetadataCache.InvalidateMetadata(ctx, ev.GetID())
	return ev, nil
}

func (h *EventHandlers) DeleteEvent(ctx context.Context, event *db.Event, params *EventDeleteParams) (TypedEvent, error) {
	handler, err := h.getHandlerForType(event.Type)
	if err != nil {
		return nil, err
	}
	ev, err := handler.DeleteEvent(ctx, event, params)
	if err != nil {
		return nil, err
	}
	h.EventCache.InvalidateEvent(ctx, ev.GetID())
	h.MetadataCache.InvalidateMetadata(ctx, ev.GetID())
	return ev, nil
}

func (h *EventHandlers) AddLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	handler, err := h.getHandlerForType(dbEvent.Type)
	if err != nil {
		return nil, err
	}
	return handler.AddLocalization(ctx, dbEvent, params)
}

func (h *EventHandlers) UpdateLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	handler, err := h.getHandlerForType(dbEvent.Type)
	if err != nil {
		return nil, err
	}
	return handler.UpdateLocalization(ctx, dbEvent, params)
}

func (h *EventHandlers) RemoveLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	handler, err := h.getHandlerForType(dbEvent.Type)
	if err != nil {
		return nil, err
	}
	return handler.RemoveLocalization(ctx, dbEvent, params)
}

func (h *EventHandlers) GetArchiveVideos(ctx context.Context, eventIDs []string) (map[string][]video.ArchiveVideo, error) {
	videos := h.VideoCache.GetArchives(ctx, eventIDs)
	if videos == nil {
		videos = make(map[string][]video.ArchiveVideo, len(eventIDs))
	}
	uncachedEventIDs := make([]string, 0, len(eventIDs))
	for _, eventID := range eventIDs {
		if _, exists := videos[eventID]; !exists {
			uncachedEventIDs = append(uncachedEventIDs, eventID)
		}
	}

	if len(uncachedEventIDs) > 0 {
		events, err := h.GetEvents(ctx, uncachedEventIDs, false, false)
		if err != nil {
			return nil, err
		}
		// For now this is done serially, but can be done in a loop if necessary
		for _, event := range events {
			handler, err := h.getHandlerForType(event.GetType())
			if err != nil {
				return nil, err
			}
			archives, err := handler.GetArchiveVideos(ctx, event)
			if err != nil {
				h.Log.Log("err", err, "event_id", event.GetID(), "could no get archive videos")
				continue
			}
			videos[event.GetID()] = archives
			h.VideoCache.CacheArchives(ctx, event.GetID(), archives)
		}
	}

	return videos, nil
}

type ArchiveVideosForEvent struct {
	Videos      []video.ArchiveVideo
	HasNextPage bool
}

func (h *EventHandlers) GetArchiveVideosForEvent(ctx context.Context, eventID string) ([]video.ArchiveVideo, error) {
	idVideoMap, err := h.GetArchiveVideos(ctx, []string{eventID})
	if err != nil || idVideoMap == nil {
		return nil, err
	}

	return idVideoMap[eventID], nil
}
