package types

import (
	"context"
	"fmt"
	"sort"
	"strings"
	"sync"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/gea/internal/clock"
)

type LoadEventsThrottlerConfig interface {
	GetThrottleInterval() time.Duration
}

type DistconfLoadEventsThrottlerConfig struct {
	ThrottleInterval *distconf.Duration
}

var _ LoadEventsThrottlerConfig = &DistconfLoadEventsThrottlerConfig{}

func (c *DistconfLoadEventsThrottlerConfig) Load(dconf *distconf.Distconf) error {
	c.ThrottleInterval = dconf.Duration("gea.load_event_throttler.interval", time.Second)
	return nil
}

func (c *DistconfLoadEventsThrottlerConfig) GetThrottleInterval() time.Duration {
	return c.ThrottleInterval.Get()
}

type LoadEventsThrottler struct {
	config   LoadEventsThrottlerConfig
	clock    clock.Clock
	mutex    sync.Mutex
	stateMap map[string]*loadEventsThrottlerState
}

func NewLoadEventsThrottler(config LoadEventsThrottlerConfig, clock clock.Clock) *LoadEventsThrottler {
	return &LoadEventsThrottler{
		config:   config,
		clock:    clock,
		stateMap: make(map[string]*loadEventsThrottlerState),
	}
}

func (e *LoadEventsThrottler) Get(eventIDs []string, getDeleted bool) (*LoadEventsResult, bool) {
	key := e.generateKey(eventIDs, getDeleted)
	now := e.clock.NowUTC()

	e.mutex.Lock()
	defer e.mutex.Unlock()

	e.removeExpired(now)

	state := e.stateMap[key]
	if state != nil {
		return state.result, false
	}

	state = &loadEventsThrottlerState{
		timeStarted: now,
		result:      newLoadEventsResult(),
	}
	e.stateMap[key] = state

	return state.result, true
}

// Generate a key that comprises of the event IDs and whether or not we are fetching deleted events.
func (e *LoadEventsThrottler) generateKey(eventIDs []string, getDeleted bool) string {
	tokens := make([]string, 0, len(eventIDs)+1)
	tokens = append(tokens, eventIDs...)
	sort.Strings(tokens)

	tokens = append(tokens, fmt.Sprintf("%t", getDeleted))
	return strings.Join(tokens, ",")
}

// removeExpired assumes that the caller locks the mutex.
func (e *LoadEventsThrottler) removeExpired(now time.Time) {
	interval := e.config.GetThrottleInterval()

	expiredKeys := make([]string, 0)
	for key, state := range e.stateMap {
		if now.Sub(state.timeStarted) > interval {
			expiredKeys = append(expiredKeys, key)
		}
	}

	for _, expiredKey := range expiredKeys {
		delete(e.stateMap, expiredKey)
	}
}

type loadEventsThrottlerState struct {
	timeStarted time.Time
	result      *LoadEventsResult
}

type LoadEventsResult struct {
	events []TypedEvent
	err    error

	mutex          sync.RWMutex
	resultRecieved chan struct{}
}

func newLoadEventsResult() *LoadEventsResult {
	return &LoadEventsResult{
		resultRecieved: make(chan struct{}),
	}
}

func (l *LoadEventsResult) WaitAndGetResult(ctx context.Context) ([]TypedEvent, error) {
	err := l.waitForResultOrTimeout(ctx)
	if err != nil {
		return nil, err
	}

	l.mutex.RLock()
	defer l.mutex.RUnlock()

	return l.events, l.err
}

func (l *LoadEventsResult) waitForResultOrTimeout(ctx context.Context) error {
	select {
	case <-l.resultRecieved:
		return nil
	case <-ctx.Done():
		return errors.Wrap(ctx.Err(), "stopped waiting on loading event because the context was closed")
	}
}

// SetResult should only be called once.
func (l *LoadEventsResult) SetResult(events []TypedEvent, err error) {
	// Store the result.
	l.mutex.Lock()
	l.events = events
	l.err = nil
	l.mutex.Unlock()

	// Close the resultRecieved to signal that waiting go-routines can get the result.
	close(l.resultRecieved)
}
