package types

import (
	"context"
	"testing"
	"time"

	testutils "code.justin.tv/twitch-events/gea/internal/test-utils"
	. "github.com/smartystreets/goconvey/convey"
)

type staticLoadEventsThrottlerConfig struct {
	interval time.Duration
}

var _ LoadEventsThrottlerConfig = &staticLoadEventsThrottlerConfig{}

func (s *staticLoadEventsThrottlerConfig) GetThrottleInterval() time.Duration {
	return s.interval
}

func TestLoadEventsThrottler(t *testing.T) {
	Convey("LoadEventsThrottler", t, func() {
		interval := time.Second
		config := &staticLoadEventsThrottlerConfig{
			interval: interval,
		}

		Convey("Throttler should group requests with the same params together", func() {
			clock := &testutils.StubClock{
				ControlledNow: time.Now(),
			}
			throttler := NewLoadEventsThrottler(config, clock)

			eventIDs := []string{"eventID1", "eventID2"}

			result, shouldQueryDB := throttler.Get(eventIDs, false)
			So(shouldQueryDB, ShouldBeTrue)
			So(result, ShouldNotBeNil)

			Convey("Second request at the same time should wait on the original request", func() {
				secondResult, shouldQueryDB := throttler.Get(eventIDs, false)
				So(shouldQueryDB, ShouldBeFalse)
				So(secondResult, ShouldEqual, result)
			})

			Convey("Request with event IDs in a different order should wait on the original request", func() {
				secondResult, shouldQueryDB := throttler.Get([]string{"eventID2", "eventID1"}, false)
				So(shouldQueryDB, ShouldBeFalse)
				So(secondResult, ShouldEqual, result)
			})

			Convey("Request with different event IDs should query the db", func() {
				secondResult, shouldQueryDB := throttler.Get([]string{"eventID1"}, false)
				So(shouldQueryDB, ShouldBeTrue)
				So(secondResult, ShouldNotBeNil)
				So(secondResult, ShouldNotEqual, result)
			})

			Convey("Request with different getDeleted should query the db", func() {
				secondResult, shouldQueryDB := throttler.Get(eventIDs, true)
				So(shouldQueryDB, ShouldBeTrue)
				So(secondResult, ShouldNotBeNil)
				So(secondResult, ShouldNotEqual, result)
			})
		})

		Convey("Throttler should group requests within a time interval together", func() {
			now := time.Now()
			clock := &testutils.StubClock{
				ControlledNow: now,
			}
			throttler := NewLoadEventsThrottler(config, clock)
			eventIDs := []string{"eventID1", "eventID2"}

			result, shouldQueryDB := throttler.Get(eventIDs, false)
			So(shouldQueryDB, ShouldBeTrue)
			So(result, ShouldNotBeNil)

			// Requests made WITHIN the time interval should wait on the original request, instead of
			// querying the db.
			now = now.Add(interval)
			clock.SetNow(now)
			secondResult, shouldQueryDB := throttler.Get(eventIDs, false)
			So(shouldQueryDB, ShouldBeFalse)
			So(secondResult, ShouldEqual, result)

			// Requests made AFTER the time interval should query the db.
			now = now.Add(time.Millisecond)
			clock.SetNow(now)
			thirdResult, shouldQueryDB := throttler.Get(eventIDs, false)
			So(shouldQueryDB, ShouldBeTrue)
			So(thirdResult, ShouldNotEqual, result)
		})
	})
}

type eventsErrorPair struct {
	events []TypedEvent
	err    error
}

func TestLoadEventsResult(t *testing.T) {
	Convey("LoadEventsResult", t, func() {
		interval := time.Second
		config := &staticLoadEventsThrottlerConfig{
			interval: interval,
		}
		clock := &testutils.StubClock{
			ControlledNow: time.Now(),
		}
		throttler := NewLoadEventsThrottler(config, clock)

		eventIDs := []string{"eventID1", "eventID2"}

		// A request comes in to load events.  It should be instructed to load them from the db.
		result, shouldQueryDB := throttler.Get(eventIDs, false)
		So(shouldQueryDB, ShouldBeTrue)
		So(result, ShouldNotBeNil)

		Convey("A second request that should wait and timeout if the first request takes too long to set the result", func() {
			// A second request comes in to load the events, and is instructed to wait on the first request.
			secondResult, shouldQueryDB := throttler.Get(eventIDs, false)
			So(shouldQueryDB, ShouldBeFalse)
			So(result, ShouldNotBeNil)

			ctx := context.Background()
			ctx, cancelFunc := context.WithTimeout(ctx, time.Millisecond*100)
			defer cancelFunc()

			// The second request waits on the first, but since we don't have the first request set the result, the
			// second should stop waiting after the context times out.
			events, err := secondResult.WaitAndGetResult(ctx)
			So(err, ShouldNotBeNil)
			So(events, ShouldBeNil)
		})

		Convey("A second request that should wait should be able to get the result from the first request", func() {
			beforeWaitAndGetResultChan := make(chan struct{})
			resultChan := make(chan *eventsErrorPair)
			go func() {
				// A second request comes in to load the events, and is instructed to wait on the first request.
				secondResult, _ := throttler.Get(eventIDs, false)

				ctx := context.Background()
				ctx, cancelFunc := context.WithTimeout(ctx, time.Second)
				defer cancelFunc()

				// Signal that this goroutine is about to start waiting.
				close(beforeWaitAndGetResultChan)

				// The second request waits on the first request.  In this test we will have the first request
				// set the result.  When that happens, we expect that the second request should stop waiting
				// and get the result that the first request set.
				events, err := secondResult.WaitAndGetResult(ctx)

				// Push the result that the second request retrieved in a channel so that the main goroutine
				// can wait for this goroutine to complete, and so it can do assertions on the result.
				resultChan <- &eventsErrorPair{
					events: events,
					err:    err,
				}
			}()

			// Wait for the second request to start waiting on the first request.
			// (Using sleeps for synchronization leads to flaky tests - we should find another way for the
			//	second request's goroutine to communicate to this goroutine that it has begun to wait, but I
			//  don't think that it is possible with go's synchronization tools.)
			<-beforeWaitAndGetResultChan
			time.Sleep(time.Millisecond * 100)

			// The first request loads events from the database, and calls SetResult to communicate those
			// events to other requests that are waiting.
			events := []TypedEvent{
				&SingleEvent{
					ID: "eventID1",
				},
				&SingleEvent{
					ID: "eventID2",
				},
			}
			result.SetResult(events, nil)

			// Wait for the second request to get its result.
			var result *eventsErrorPair
			select {
			case result = <-resultChan:
			case <-time.After(time.Second * 2):
				// If a goroutine never returns, kill the test.
				t.Fatal("TestIntegration_ThrottlingGetEvent timed out waiting for GetEvents to return")
			}

			// Verify that the second request got the results that the first request produced.
			So(result, ShouldNotBeNil)
			So(result.err, ShouldBeNil)
			So(result.events, ShouldHaveLength, len(events))
		})
	})
}
