package types

import (
	"context"
	"time"

	"encoding/json"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/gomemcache/memcache"
	"code.justin.tv/twitch-events/gea/internal/cache"
)

type MetadataCacheConfig struct {
	MetadataCacheDuration *distconf.Duration
}

func (c *MetadataCacheConfig) Load(dconf *distconf.Distconf) error {
	c.MetadataCacheDuration = dconf.Duration("gea.metadata_cache_duration", 5*time.Minute)
	return nil
}

type MetadataCache struct {
	Config *MetadataCacheConfig
	Cache  *cache.Service
	Stats  *service_common.StatSender
	Log    *log.ElevatedLog
}

func (c *MetadataCache) metadataKey(eventID string) string {
	return "gea:md:" + eventID
}

func (c *MetadataCache) GetMetdata(ctx context.Context, eventID string) *EventMetadata {
	startTime := time.Now()
	defer func() {
		c.Stats.TimingDurationC("get.timing", time.Since(startTime), 0.1)
	}()

	key := c.metadataKey(eventID)

	cached, err := c.Cache.Get(ctx, key)
	if err != nil {
		if err == memcache.ErrCacheMiss {
			return nil
		}
		c.Log.Log("err", err, "keys", keys, "could not get metadata from cache")
		return nil
	}
	metadata := EventMetadata{}
	err = json.Unmarshal(cached.Value, &metadata)
	if err != nil {
		c.Log.Log("err", err, "key", key, "could not unmarshall metadata from cache")
		return nil
	}

	return &metadata
}

func (c *MetadataCache) CacheMetadata(ctx context.Context, metadata *EventMetadata) {
	startTime := time.Now()
	defer func() {
		c.Stats.TimingDurationC("set.timing", time.Since(startTime), 0.1)
	}()

	key := c.metadataKey(metadata.ID)
	data, err := json.Marshal(metadata)
	if err != nil {
		c.Log.Log("err", err, "key", key, "could not marshall metadata for cache")
	}
	cacheErr := c.Cache.Set(ctx, &memcache.Item{
		Key:        key,
		Value:      data,
		Expiration: int32(c.Config.MetadataCacheDuration.Get().Seconds()),
	})
	if cacheErr != nil {
		c.Log.Log("err", cacheErr, "key", key, "could not cache metadata")
	}
}

func (c *MetadataCache) InvalidateMetadata(ctx context.Context, eventID string) {
	key := c.metadataKey(eventID)
	err := c.Cache.Delete(ctx, key)
	if err != nil {
		if err != memcache.ErrCacheMiss {
			c.Log.Log("err", err, "key", key, "could not invalidate metadata in cache")
		}
	}
}
