package types

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/images"
	"code.justin.tv/twitch-events/gea/internal/video"
	usersservice "code.justin.tv/web/users-service/client/usersclient_internal"
)

const (
	EventTypePremiere      = "premiere"
	premiereIDAttr         = "premiere_id"
	premiereMarshallPrefix = "premiere:v1:"
)

// The marshall prefix should be changed when fields are added to the struct
type PremiereEvent struct {
	ID      string `json:"id"`
	OwnerID string `json:"owner_id"`
	Type    string `json:"type"`

	CreatedAt time.Time  `json:"created_at"`
	UpdatedAt time.Time  `json:"updated_at"`
	DeletedAt *time.Time `json:"deleted_at,omitempty"`

	StartTime  time.Time `json:"start_time"`
	EndTime    time.Time `json:"end_time"`
	TimeZoneID string    `json:"time_zone_id"`

	ImageID     *images.ImageID `json:"image_id,omitempty"`
	ImageURL    string          `json:"image_url"`
	Language    string          `json:"language"`
	Title       string          `json:"title"`
	Description string          `json:"description"`
	ChannelID   string          `json:"channel_id"`
	GameID      string          `json:"game_id"`

	PremiereID string `json:"premiere_id"`
}

func (e *PremiereEvent) GetType() string {
	return EventTypePremiere
}

func (e *PremiereEvent) GetID() string {
	return e.ID
}

func (e *PremiereEvent) GetChannelIDs() []string {
	return []string{e.ChannelID}
}

func (e *PremiereEvent) GetTitle() string {
	return e.Title
}

func (e *PremiereEvent) GetParentID() string {
	return ""
}

func (e *PremiereEvent) GetOwnerID() string {
	return e.OwnerID
}

type PremiereEventHandler struct {
	BaseEventHandler
	ImageURLClient *images.ImageURLClient
	UserService    usersservice.InternalClient
}

func (h *PremiereEventHandler) Handles(eventType string) bool {
	return eventType == EventTypePremiere
}

func (h *PremiereEventHandler) CreateEvent(ctx context.Context, params *EventUpdateParams) (TypedEvent, error) {
	err := h.validateEventUpdate(ctx, nil, params)
	if err != nil {
		return nil, err
	}

	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbEvent, err := h.OracleDB.CreateEvent(txCtx, toCreateDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error creating event")
	}
	attr := map[string]string{
		premiereIDAttr: *params.PremiereID,
	}
	err = h.OracleDB.SetEventAttributes(txCtx, dbEvent.ID, attr)
	if err != nil {
		return nil, errors.Wrap(err, "error creating event")
	}

	event := h.fromDBEvent(dbEvent, attr)

	_, err = h.OracleDB.CreateBroadcast(txCtx, &db.CreateDBBroadcastParams{
		EventID:   event.ID,
		Language:  event.Language,
		ChannelID: event.ChannelID,
		GameID:    event.GameID,
		StartTime: event.StartTime,
		EndTime:   event.EndTime,
	})
	if err != nil {
		return nil, errors.Wrap(err, "error creating broadcast for event")
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}

	return event, nil
}

func (h *PremiereEventHandler) validateEventUpdate(ctx context.Context, eventID *string, params *EventUpdateParams) error {
	if params.OwnerID == "" {
		return invalid("owner_id", "should have a owner")
	}
	if params.ParentID != nil {
		return invalid("parent_id", "should not have parent")
	}
	if params.StartTime == nil || params.StartTime.IsZero() {
		return invalid("start_time", "should have a start time")
	}
	if params.EndTime == nil || params.EndTime.IsZero() {
		return invalid("end_time", "should have an end time")
	}
	if !params.StartTime.Before(*params.EndTime) {
		return invalid("end_time", "should be after start time")
	}

	if params.TimeZoneID == nil || *params.TimeZoneID == "" {
		return invalid("time_zone_id", "should have a time zone")
	} else if !isTimeZoneValid(*params.TimeZoneID) {
		return invalid("time_zone_id", "expected a tz database time zone (e.g. America/New_York)")
	}

	if params.Language == nil || *params.Language == "" {
		return invalid("language", "should have a language")
	}
	if params.Title == nil || *params.Title == "" {
		return invalid("title", "should have a title")
	}
	if params.Description == nil || *params.Description == "" {
		return invalid("description", "should have a description")
	}
	if params.ChannelID == nil || *params.ChannelID == "" {
		return invalid("channel_id", "should have a channel")
	}
	if params.GameID == nil || *params.GameID == "" {
		return invalid("game_id", "should have a game")
	}
	if params.PremiereID == nil || *params.PremiereID == "" {
		return invalid("premiere_id", "should have a premiere")
	}
	return h.validateEventDoesNotOverlap(ctx, eventID, *params.ChannelID, *params.StartTime, *params.EndTime)
}

func (h *PremiereEventHandler) validateEventDoesNotOverlap(ctx context.Context, eventID *string, channelID string, startTime time.Time, endTime time.Time) error {
	// Is the start time within another event?
	eventsAtStartTime, err := h.OracleDB.GetEventIDsSortedByID(ctx, &db.BroadcastFilter{
		Types:           []string{EventTypePremiere},
		ChannelIDs:      []string{channelID},
		StartTimeWindow: &db.TimeWindow{End: &startTime},
		EndTimeWindow:   &db.TimeWindow{Start: &startTime},
	}, "", 10)
	if err != nil {
		return err
	}
	for _, id := range eventsAtStartTime.EventIDs {
		if eventID == nil || *eventID != id {
			return invalid("start_time", "a premiere already exists in that time")
		}
	}
	// Is the end time within another event?
	eventsAtEndTime, err := h.OracleDB.GetEventIDsSortedByID(ctx, &db.BroadcastFilter{
		Types:           []string{EventTypePremiere},
		ChannelIDs:      []string{channelID},
		StartTimeWindow: &db.TimeWindow{End: &endTime},
		EndTimeWindow:   &db.TimeWindow{Start: &endTime},
	}, "", 10)
	if err != nil {
		return err
	}
	for _, id := range eventsAtEndTime.EventIDs {
		if eventID == nil || *eventID != id {
			return invalid("end_time", "a premiere already exists in that time")
		}
	}
	// Does the event encompass another event?
	eventsOverTimePreiod, err := h.OracleDB.GetEventIDsSortedByID(ctx, &db.BroadcastFilter{
		Types:           []string{EventTypePremiere},
		ChannelIDs:      []string{channelID},
		StartTimeWindow: &db.TimeWindow{Start: &startTime},
		EndTimeWindow:   &db.TimeWindow{End: &endTime},
	}, "", 10)
	if err != nil {
		return err
	}
	for _, id := range eventsOverTimePreiod.EventIDs {
		if eventID == nil || *eventID != id {
			return invalid("start_time", "a premiere already exists in that time")
		}
	}
	return nil
}

func (h *PremiereEventHandler) fromDBEvent(dbEvent *db.Event, attr map[string]string) *PremiereEvent {
	// Time zone ID is now required for Premiere events, however it is possible that there are Premiere events
	// in the database that were created without time zone IDs.  We'll default to an empty string for now until
	// we can fill in the NULL time zone IDs.
	timeZoneID := ""
	if dbEvent.TimeZoneID != nil {
		timeZoneID = *dbEvent.TimeZoneID
	}

	return &PremiereEvent{
		ID:      dbEvent.ID,
		OwnerID: dbEvent.OwnerID,
		Type:    EventTypePremiere,

		CreatedAt:  dbEvent.CreatedAt,
		DeletedAt:  dbEvent.DeletedAt,
		TimeZoneID: timeZoneID,

		ImageID:     dbEvent.CoverImageID,
		ImageURL:    h.ImageURLClient.GetImageURL(dbEvent.CoverImageID),
		Language:    *dbEvent.Language,
		Title:       *dbEvent.Title,
		Description: *dbEvent.Description,
		ChannelID:   *dbEvent.ChannelID,
		GameID:      *dbEvent.GameID,

		StartTime: *dbEvent.StartTime,
		EndTime:   *dbEvent.EndTime,

		PremiereID: attr[premiereIDAttr],
	}
}

func (h *PremiereEventHandler) GetEvents(ctx context.Context, dbEvents []*db.Event) ([]TypedEvent, error) {
	eventIDs := make([]string, 0, len(dbEvents))
	for _, dbEvent := range dbEvents {
		eventIDs = append(eventIDs, dbEvent.ID)
	}
	attrs, err := h.OracleDB.GetEventAttributesForEventIDs(ctx, eventIDs, []string{premiereIDAttr})
	if err != nil {
		return nil, errors.Wrap(err, "could not get attributes for events")
	}

	events := make([]TypedEvent, 0, len(dbEvents))
	for _, dbEvent := range dbEvents {
		events = append(events, h.fromDBEvent(dbEvent, attrs[dbEvent.ID]))
	}
	return events, nil
}

func (h *PremiereEventHandler) GetEventMetadata(ctx context.Context, dbEvent *db.Event) (*EventMetadata, error) {
	attrs, err := h.OracleDB.GetEventAttributesForEventIDs(ctx, []string{dbEvent.ID}, []string{premiereIDAttr})
	if err != nil {
		return nil, err
	}
	if len(attrs) == 0 || attrs[dbEvent.ID] == nil {
		return nil, errors.New("Attributes not found")
	}
	ev := h.fromDBEvent(dbEvent, attrs[dbEvent.ID])

	owner, err := h.UserService.GetUserByID(ctx, ev.OwnerID, nil)
	if err != nil {
		return nil, err
	}

	timeLocation := h.LoadTimeLocation(ctx, ev.TimeZoneID)
	description := formatTimeRange(ev.StartTime, ev.EndTime, timeLocation)
	ownerName := owner.Displayname
	title := fmt.Sprintf("%s: %s", *ownerName, ev.Title)
	url := fmt.Sprintf("https://www.twitch.tv/events/%s", ev.ID)
	OGImageURL := strings.Replace(ev.ImageURL, "{width}", OGImageWidth, 1)
	OGImageURL = strings.Replace(OGImageURL, "{height}", OGImageHeight, 1)
	twitterImageURL := strings.Replace(ev.ImageURL, "{width}", twitterImageWidth, 1)
	twitterImageURL = strings.Replace(twitterImageURL, "{height}", twitterImageHeight, 1)

	metadata := EventMetadata{
		ID:          ev.ID,
		Title:       title,
		Description: description,

		OGTitle:       title,
		OGDescription: description,
		OGURL:         url,
		OGImage:       OGImageURL,
		OGImageWidth:  OGImageWidth,
		OGImageHeight: OGImageHeight,
		OGType:        openGraphType,

		TwitterCard:        twitterCard,
		TwitterTitle:       title,
		TwitterDescription: description,
		TwitterImage:       twitterImageURL,
		TwitterURL:         url,
	}
	return &metadata, nil
}

func (h *PremiereEventHandler) UpdateEvent(ctx context.Context, oldEvent *db.Event, params *EventUpdateParams) (TypedEvent, error) {
	if err := h.validateEventUpdate(ctx, &oldEvent.ID, params); err != nil {
		return nil, err
	}
	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbEvent, err := h.OracleDB.UpdateEvent(txCtx, oldEvent.ID, toUpdateDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error updating event")
	}
	attr := map[string]string{
		premiereIDAttr: *params.PremiereID,
	}
	err = h.OracleDB.SetEventAttributes(txCtx, dbEvent.ID, attr)
	if err != nil {
		return nil, errors.Wrap(err, "error creating event")
	}

	event := h.fromDBEvent(dbEvent, attr)

	_, err = h.OracleDB.UpdateBroadcast(txCtx, event.ID, *oldEvent.Language, &db.UpdateDBBroadcastParams{
		Language:  event.Language,
		ChannelID: event.ChannelID,
		GameID:    event.GameID,
		StartTime: event.StartTime,
		EndTime:   event.EndTime,
	})
	if err != nil {
		return nil, errors.Wrap(err, "error updating broadcast for event")
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}
	return event, nil
}

func (h *PremiereEventHandler) DeleteEvent(ctx context.Context, oldEvent *db.Event, params *EventDeleteParams) (TypedEvent, error) {
	eventID := oldEvent.ID

	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	attrs, err := h.OracleDB.GetEventAttributesForEventIDs(txCtx, []string{eventID}, []string{premiereIDAttr})
	if err != nil {
		return nil, errors.Wrap(err, "could not get attributes for events")
	}

	dbEvent, err := h.OracleDB.DeleteEvent(txCtx, eventID, toDeleteDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error deleting event")
	}
	event := h.fromDBEvent(dbEvent, attrs[eventID])

	_, err = h.OracleDB.DeleteBroadcastsByEventID(txCtx, event.ID)
	if err != nil {
		return nil, errors.Wrap(err, "error updating broadcast for event")
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}
	return event, nil
}

func (h *PremiereEventHandler) MarshallEvent(event TypedEvent) ([]byte, error) {
	data, err := json.Marshal(event)
	if err != nil {
		return nil, errors.Wrap(err, "could not marshall event")
	}
	return addPrefix(premiereMarshallPrefix, data), nil
}

func (h *PremiereEventHandler) UnmarshallEvent(data []byte) TypedEvent {
	d, ok := removePrefix(premiereMarshallPrefix, data)
	if !ok {
		return nil
	}
	var event PremiereEvent
	err := json.Unmarshal(d, &event)
	if err != nil {
		h.Log.Log("err", err, "could not unmarshall event data")
	}
	return &event
}

func (h *PremiereEventHandler) AddLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	return nil, errors.New("localization not supported for premieres")
}

func (h *PremiereEventHandler) UpdateLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	return nil, errors.New("localization not supported for premieres")
}

func (h *PremiereEventHandler) RemoveLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	return nil, errors.New("localization not supported for premieres")
}

func (h *PremiereEventHandler) GetArchiveVideos(ctx context.Context, typedEvent TypedEvent) ([]video.ArchiveVideo, error) {
	// TODO Replace with a lookup to Gala
	_, ok := typedEvent.(*PremiereEvent)
	if !ok {
		return nil, errors.Errorf("wrong handler for type %s for event id %s", typedEvent.GetType(), typedEvent.GetID())
	}
	return make([]video.ArchiveVideo, 0), nil
}
