package types

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/images"
	"code.justin.tv/twitch-events/gea/internal/video"
	usersservice "code.justin.tv/web/users-service/client/usersclient_internal"
)

const (
	EventTypeSegment      = "segment"
	segmentMarshallPrefix = "segment:v1:"
)

// The cache key version number should be bumped when fields are added to the struct
type SegmentEvent struct {
	ID       string `json:"id"`
	OwnerID  string `json:"owner_id"`
	Type     string `json:"type"`
	ParentID string `json:"parent_id"`

	CreatedAt time.Time  `json:"created_at"`
	UpdatedAt time.Time  `json:"updated_at"`
	DeletedAt *time.Time `json:"deleted_at,omitempty"`

	StartTime  time.Time `json:"start_time"`
	EndTime    time.Time `json:"end_time"`
	TimeZoneID string    `json:"time_zone_id"`

	ImageID     *images.ImageID `json:"image_id"`
	ImageURL    string          `json:"image_url"`
	Language    string          `json:"language"`
	Title       string          `json:"title"`
	Description string          `json:"description"`
	ChannelID   string          `json:"channel_id"`
	GameID      string          `json:"game_id"`
}

func (e *SegmentEvent) GetType() string {
	return EventTypeSegment
}

func (e *SegmentEvent) GetID() string {
	return e.ID
}

func (e *SegmentEvent) GetChannelIDs() []string {
	return []string{e.ChannelID}
}

func (e *SegmentEvent) GetTitle() string {
	return e.Title
}

func (e *SegmentEvent) GetParentID() string {
	return e.ParentID
}

func (e *SegmentEvent) GetOwnerID() string {
	return e.OwnerID
}

type SegmentEventHandler struct {
	BaseEventHandler
	EventCache       *EventCache
	MetadataCache    *MetadataCache
	ImageURLClient   *images.ImageURLClient
	VideoFinder      *video.Finder
	TimetableHandler *TimetableEventHandler
	UserService      usersservice.InternalClient
}

func (h *SegmentEventHandler) Handles(eventType string) bool {
	return eventType == EventTypeSegment
}

func (h *SegmentEventHandler) CreateEvent(ctx context.Context, params *EventUpdateParams) (TypedEvent, error) {
	err := h.validateEventUpdate(ctx, params)
	if err != nil {
		return nil, err
	}

	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbEvent, err := h.OracleDB.CreateEvent(txCtx, toCreateDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error creating event")
	}
	events, err := h.fromDBEvents(ctx, []*db.Event{dbEvent})
	if err != nil {
		return nil, err
	}
	event := events[0]

	_, err = h.OracleDB.CreateBroadcast(txCtx, &db.CreateDBBroadcastParams{
		EventID:   event.ID,
		Language:  event.Language,
		ChannelID: event.ChannelID,
		GameID:    event.GameID,
		StartTime: event.StartTime,
		EndTime:   event.EndTime,
	})
	if err != nil {
		return nil, errors.Wrap(err, "error creating broadcast for event")
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}

	// Also need to invalidate the parent event
	h.EventCache.InvalidateEvent(ctx, event.ParentID)
	h.MetadataCache.InvalidateMetadata(ctx, event.ParentID)
	return event, nil
}

func (h *SegmentEventHandler) validateEventUpdate(ctx context.Context, params *EventUpdateParams) error {
	if params.ParentID == nil || *params.ParentID == "" {
		return invalid("parent_id", "should have a parent")
	}

	parent, err := h.OracleDB.GetEvent(ctx, *params.ParentID, false)
	if err != nil {
		return err
	}
	if parent == nil {
		return invalid("parent_id", "parent event does not exist")
	}
	if parent.Type != EventTypeTimetable {
		return invalid("parent_id", "parent event must be a timetable")
	}

	if params.OwnerID == "" {
		return invalid("owner_id", "should have a owner")
	}
	if params.OwnerID != parent.OwnerID {
		return invalid("owner_id", "should be the same as parent event")
	}
	if params.StartTime == nil || params.StartTime.IsZero() {
		return invalid("start_time", "should have a start time")
	}
	if params.EndTime == nil || params.EndTime.IsZero() {
		return invalid("end_time", "should have an end time")
	}
	if !params.StartTime.Before(*params.EndTime) {
		return invalid("end_time", "should be after start time")
	}
	if params.TimeZoneID != nil {
		return invalid("time_zone_id", "should not have a time zone")
	}
	if params.Language != nil {
		return invalid("language", "should not have a language")
	}
	if params.Title == nil || *params.Title == "" {
		return invalid("title", "should have a title")
	}
	if params.ChannelID == nil || *params.ChannelID == "" {
		return invalid("channel_id", "should have a channel")
	}
	if params.GameID == nil || *params.GameID == "" {
		return invalid("game_id", "should have a game")
	}
	return nil
}

func (h *SegmentEventHandler) fromDBEvents(ctx context.Context, dbEvents []*db.Event) ([]*SegmentEvent, error) {
	parentIDs := make(map[string]struct{})
	for _, dbEvent := range dbEvents {
		parentIDs[*dbEvent.ParentID] = struct{}{}
	}
	parents, err := h.TimetableHandler.GetInheritedFields(ctx, keys(parentIDs))
	if err != nil {
		return nil, errors.Wrap(err, "error looking up parents")
	}

	events := make([]*SegmentEvent, 0, len(dbEvents))
	for _, dbEvent := range dbEvents {
		parent := parents[*dbEvent.ParentID]
		event := &SegmentEvent{
			ID:       dbEvent.ID,
			OwnerID:  dbEvent.OwnerID,
			Type:     EventTypeSegment,
			ParentID: *dbEvent.ParentID,

			CreatedAt: dbEvent.CreatedAt,
			UpdatedAt: dbEvent.UpdatedAt,
			DeletedAt: dbEvent.DeletedAt,

			StartTime:  *dbEvent.StartTime,
			EndTime:    *dbEvent.EndTime,
			TimeZoneID: parent.TimeZoneID,

			ImageID:     dbEvent.CoverImageID,
			ImageURL:    h.ImageURLClient.GetImageURL(dbEvent.CoverImageID),
			Language:    parent.Language,
			Title:       *dbEvent.Title,
			Description: coalesce(dbEvent.Description, ""),
			ChannelID:   *dbEvent.ChannelID,
			GameID:      *dbEvent.GameID,
		}
		events = append(events, event)
	}
	return events, nil
}

func (h *SegmentEventHandler) GetEvents(ctx context.Context, dbEvents []*db.Event) ([]TypedEvent, error) {
	events, err := h.fromDBEvents(ctx, dbEvents)
	if err != nil {
		return nil, err
	}
	typedEvents := make([]TypedEvent, 0, len(events))
	for _, event := range events {
		typedEvents = append(typedEvents, event)
	}
	return typedEvents, nil
}

func (h *SegmentEventHandler) GetEventMetadata(ctx context.Context, dbEvent *db.Event) (*EventMetadata, error) {
	events, err := h.fromDBEvents(ctx, []*db.Event{dbEvent})
	if err != nil {
		return nil, err
	}
	if len(events) == 0 {
		return nil, errors.New("Event not found")
	}
	ev := events[0]

	eventTimetables, err := h.OracleDB.GetEvents(ctx, []string{ev.ParentID}, false)
	if err != nil {
		return nil, err
	}
	if len(eventTimetables) == 0 {
		return nil, errors.New("Timetable not found")
	}
	timetableName := eventTimetables[0].Title

	timeLocation := h.LoadTimeLocation(ctx, ev.TimeZoneID)
	description := formatTimeRange(ev.StartTime, ev.EndTime, timeLocation)
	title := fmt.Sprintf("%s: %s", *timetableName, ev.Title)
	url := fmt.Sprintf("https://www.twitch.tv/events/%s", ev.ID)
	OGImageURL := strings.Replace(ev.ImageURL, "{width}", OGImageWidth, 1)
	OGImageURL = strings.Replace(OGImageURL, "{height}", OGImageHeight, 1)
	twitterImageURL := strings.Replace(ev.ImageURL, "{width}", twitterImageWidth, 1)
	twitterImageURL = strings.Replace(twitterImageURL, "{height}", twitterImageHeight, 1)

	metadata := EventMetadata{
		ID:          ev.ID,
		Title:       title,
		Description: description,

		OGTitle:       title,
		OGDescription: description,
		OGURL:         url,
		OGImage:       OGImageURL,
		OGImageWidth:  OGImageWidth,
		OGImageHeight: OGImageHeight,
		OGType:        openGraphType,

		TwitterCard:        twitterCard,
		TwitterTitle:       title,
		TwitterDescription: description,
		TwitterImage:       twitterImageURL,
		TwitterURL:         url,
	}
	return &metadata, nil
}

type SegmentsSummaryFields struct {
	StartTime  *time.Time
	EndTime    *time.Time
	ChannelIDs []string
	GameIDs    []string
}

func (h *SegmentEventHandler) GetSummaryFields(ctx context.Context, parentIDs []string) (map[string]*SegmentsSummaryFields, error) {
	ret := make(map[string]*SegmentsSummaryFields, len(parentIDs))
	for _, parentID := range parentIDs {
		ret[parentID] = &SegmentsSummaryFields{}
	}
	events, err := h.OracleDB.GetEventsByParentIDs(ctx, parentIDs, false)
	if err != nil {
		return nil, err
	}
	for _, event := range events {
		if event.Type == EventTypeSegment {
			summary := ret[*event.ParentID]
			if summary.StartTime == nil || event.StartTime.Before(*summary.StartTime) {
				summary.StartTime = event.StartTime
			}
			if summary.EndTime == nil || event.EndTime.After(*summary.EndTime) {
				summary.EndTime = event.EndTime
			}
			if !in(*event.ChannelID, summary.ChannelIDs) {
				summary.ChannelIDs = append(summary.ChannelIDs, *event.ChannelID)
			}
			if !in(*event.GameID, summary.GameIDs) {
				summary.GameIDs = append(summary.GameIDs, *event.GameID)
			}
		}
	}
	return ret, nil
}

func (h *SegmentEventHandler) UpdateEvent(ctx context.Context, oldEvent *db.Event, params *EventUpdateParams) (TypedEvent, error) {
	if err := h.validateEventUpdate(ctx, params); err != nil {
		return nil, err
	}
	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbEvent, err := h.OracleDB.UpdateEvent(txCtx, oldEvent.ID, toUpdateDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error updating event")
	}
	events, err := h.fromDBEvents(ctx, []*db.Event{dbEvent})
	if err != nil {
		return nil, err
	}
	event := events[0]

	language := ""
	if oldEvent.Language != nil {
		language = *oldEvent.Language
	} else if oldEvent.ParentID != nil {
		parent, innerErr := h.OracleDB.GetEvent(ctx, *oldEvent.ParentID, false)
		if innerErr != nil {
			return nil, innerErr
		} else if parent == nil || parent.Language == nil {
			return nil, errors.Errorf("could not load segment event's language, id: %s", oldEvent.ID)
		}
		language = *parent.Language
	} else {
		return nil, errors.Errorf("could not load segment event's language, id: %s", oldEvent.ID)
	}

	_, err = h.OracleDB.UpdateBroadcast(txCtx, event.ID, language, &db.UpdateDBBroadcastParams{
		Language:  event.Language,
		ChannelID: event.ChannelID,
		GameID:    event.GameID,
		StartTime: event.StartTime,
		EndTime:   event.EndTime,
	})
	if err != nil {
		return nil, errors.Wrap(err, "error updating broadcast for event")
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}

	// Also need to invalidate the parent event
	h.EventCache.InvalidateEvent(ctx, event.ParentID)
	return event, nil
}

func (h *SegmentEventHandler) DeleteEvent(ctx context.Context, oldEvent *db.Event, params *EventDeleteParams) (TypedEvent, error) {
	eventID := oldEvent.ID

	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbEvent, err := h.OracleDB.DeleteEvent(txCtx, eventID, toDeleteDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error deleting event")
	}
	events, err := h.fromDBEvents(ctx, []*db.Event{dbEvent})
	if err != nil {
		return nil, err
	}
	event := events[0]

	_, err = h.OracleDB.DeleteBroadcastsByEventID(txCtx, event.ID)
	if err != nil {
		return nil, errors.Wrap(err, "error updating broadcast for event")
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}

	// Also need to invalidate the parent event
	h.EventCache.InvalidateEvent(ctx, event.ParentID)
	return event, nil
}

func (h *SegmentEventHandler) DeleteSegmentsByParentID(ctx context.Context, parentEventID string, params *EventDeleteParams) error {
	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	segments, err := h.OracleDB.GetEventsByParentIDs(txCtx, []string{parentEventID}, false)
	if err != nil {
		return errors.Wrap(err, "error deleting segments")
	}

	if len(segments) > 0 {
		_, err = h.deleteSegments(txCtx, segments, params)
		if err != nil {
			return err
		}
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return errors.Wrap(err, "error deleting segments")
	}

	// Also need to invalidate the events
	for _, segment := range segments {
		h.EventCache.InvalidateEvent(ctx, segment.ID)
	}
	return err
}

func (h *SegmentEventHandler) deleteSegments(ctx context.Context, segments []*db.Event, params *EventDeleteParams) ([]*db.Event, error) {
	segmentIDs := make([]string, len(segments))
	for _, segment := range segments {
		segmentIDs = append(segmentIDs, segment.ID)
	}

	deletedEvents, err := h.OracleDB.DeleteEvents(ctx, segmentIDs, toDeleteDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "could not delete events")
	}

	_, err = h.OracleDB.DeleteBroadcastsByEventIDs(ctx, segmentIDs)
	if err != nil {
		return nil, errors.Wrap(err, "could not delete events")
	}

	return deletedEvents, nil
}

func (h *SegmentEventHandler) MarshallEvent(event TypedEvent) ([]byte, error) {
	data, err := json.Marshal(event)
	if err != nil {
		return nil, errors.Wrap(err, "could not marshall event")
	}
	return addPrefix(segmentMarshallPrefix, data), nil
}

func (h *SegmentEventHandler) UnmarshallEvent(data []byte) TypedEvent {
	d, ok := removePrefix(segmentMarshallPrefix, data)
	if !ok {
		return nil
	}
	var event SegmentEvent
	err := json.Unmarshal(d, &event)
	if err != nil {
		h.Log.Log("err", err, "could not unmarshall event data")
	}
	return &event
}

func (h *SegmentEventHandler) AddLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	return nil, errors.New("localization not supported for segments")
}

func (h *SegmentEventHandler) UpdateLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	return nil, errors.New("localization not supported for segments")
}

func (h *SegmentEventHandler) RemoveLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	return nil, errors.New("localization not supported for segments")
}

func (h *SegmentEventHandler) GetArchiveVideos(ctx context.Context, typedEvent TypedEvent) ([]video.ArchiveVideo, error) {
	event, ok := typedEvent.(*SegmentEvent)
	if !ok {
		return nil, errors.Errorf("wrong handler for type %s for event id %s", typedEvent.GetType(), typedEvent.GetID())
	}
	return h.VideoFinder.FindArchiveVideos(ctx, event.ChannelID, event.StartTime, event.EndTime)
}
