package types

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/images"
	"code.justin.tv/twitch-events/gea/internal/video"
	usersservice "code.justin.tv/web/users-service/client/usersclient_internal"
)

const (
	EventTypeSingle      = "single"
	singleMarshallPrefix = "single:v1:"
)

// The cache key version number should be bumped when fields are added to the struct
type SingleEvent struct {
	ID      string `json:"id"`
	OwnerID string `json:"owner_id"`
	Type    string `json:"type"`

	CreatedAt time.Time  `json:"created_at"`
	UpdatedAt time.Time  `json:"updated_at"`
	DeletedAt *time.Time `json:"deleted_at,omitempty"`

	StartTime  time.Time `json:"start_time"`
	EndTime    time.Time `json:"end_time"`
	TimeZoneID string    `json:"time_zone_id"`

	ImageID     *images.ImageID `json:"image_id,omitempty"`
	ImageURL    string          `json:"image_url"`
	Language    string          `json:"language"`
	Title       string          `json:"title"`
	Description string          `json:"description"`
	ChannelID   string          `json:"channel_id"`
	GameID      string          `json:"game_id"`
}

func (e *SingleEvent) GetType() string {
	return EventTypeSingle
}

func (e *SingleEvent) GetID() string {
	return e.ID
}

func (e *SingleEvent) GetChannelIDs() []string {
	return []string{e.ChannelID}
}

func (e *SingleEvent) GetTitle() string {
	return e.Title
}

func (e *SingleEvent) GetParentID() string {
	return ""
}

func (e *SingleEvent) GetOwnerID() string {
	return e.OwnerID
}

type SingleEventHandler struct {
	BaseEventHandler
	ImageURLClient *images.ImageURLClient
	VideoFinder    *video.Finder
	UserService    usersservice.InternalClient
}

func (h *SingleEventHandler) Handles(eventType string) bool {
	return eventType == EventTypeSingle
}

func (h *SingleEventHandler) CreateEvent(ctx context.Context, params *EventUpdateParams) (TypedEvent, error) {
	err := h.validateEventUpdate(ctx, params)
	if err != nil {
		return nil, err
	}

	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbEvent, err := h.OracleDB.CreateEvent(txCtx, toCreateDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error creating event")
	}
	event := h.fromDBEvent(dbEvent)

	_, err = h.OracleDB.CreateBroadcast(txCtx, &db.CreateDBBroadcastParams{
		EventID:   event.ID,
		Language:  event.Language,
		ChannelID: event.ChannelID,
		GameID:    event.GameID,
		StartTime: event.StartTime,
		EndTime:   event.EndTime,
	})
	if err != nil {
		return nil, errors.Wrap(err, "error creating broadcast for event")
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}

	return event, nil
}

func (h *SingleEventHandler) validateEventUpdate(ctx context.Context, params *EventUpdateParams) error {
	if params.OwnerID == "" {
		return invalid("owner_id", "should have a owner")
	}
	if params.ParentID != nil {
		return invalid("parent_id", "should not have parent")
	}
	if params.StartTime == nil || params.StartTime.IsZero() {
		return invalid("start_time", "should have a start time")
	}
	if params.EndTime == nil || params.EndTime.IsZero() {
		return invalid("end_time", "should have an end time")
	}
	if !params.StartTime.Before(*params.EndTime) {
		return invalid("end_time", "should be after start time")
	}
	if params.Language == nil || *params.Language == "" {
		return invalid("language", "should have a language")
	}
	if params.Title == nil || *params.Title == "" {
		return invalid("title", "should have a title")
	}
	if params.ChannelID == nil || *params.ChannelID == "" {
		return invalid("channel_id", "should have a channel")
	}
	if params.GameID == nil || *params.GameID == "" {
		return invalid("game_id", "should have a game")
	}

	// We allow empty time zones for single events because we inherited events from oracle that don't have time zones.
	// After we give these events a time zone in the database, we can make this check more strict.
	if params.TimeZoneID != nil && *params.TimeZoneID != "" {
		if !isTimeZoneValid(*params.TimeZoneID) {
			return invalid("time_zone_id", "expected a tz database time zone (e.g. America/New_York)")
		}
	}

	return nil
}

func (h *SingleEventHandler) fromDBEvent(dbEvent *db.Event) *SingleEvent {
	desc := ""
	if dbEvent.Description != nil {
		desc = *dbEvent.Description
	}

	// Time zone ID is now required for Single events, however there are Single events in the database that were
	// created without time zone IDs.  We'll default to an empty string for now until we can fill in the NULL time
	// zone IDs.
	timeZoneID := ""
	if dbEvent.TimeZoneID != nil {
		timeZoneID = *dbEvent.TimeZoneID
	}

	return &SingleEvent{
		ID:      dbEvent.ID,
		OwnerID: dbEvent.OwnerID,
		Type:    EventTypeSingle,

		CreatedAt: dbEvent.CreatedAt,
		UpdatedAt: dbEvent.UpdatedAt,
		DeletedAt: dbEvent.DeletedAt,

		ImageID:     dbEvent.CoverImageID,
		ImageURL:    h.ImageURLClient.GetImageURL(dbEvent.CoverImageID),
		Language:    *dbEvent.Language,
		Title:       *dbEvent.Title,
		Description: desc,
		ChannelID:   *dbEvent.ChannelID,
		GameID:      *dbEvent.GameID,

		StartTime:  *dbEvent.StartTime,
		EndTime:    *dbEvent.EndTime,
		TimeZoneID: timeZoneID,
	}
}

func (h *SingleEventHandler) GetEvents(ctx context.Context, dbEvents []*db.Event) ([]TypedEvent, error) {
	events := make([]TypedEvent, 0, len(dbEvents))
	for _, dbEvent := range dbEvents {
		events = append(events, h.fromDBEvent(dbEvent))
	}
	return events, nil
}

func (h *SingleEventHandler) GetEventMetadata(ctx context.Context, dbEvent *db.Event) (*EventMetadata, error) {
	ev := h.fromDBEvent(dbEvent)

	owner, err := h.UserService.GetUserByID(ctx, ev.OwnerID, nil)
	if err != nil {
		return nil, err
	}

	timeLocation := h.LoadTimeLocation(ctx, ev.TimeZoneID)
	description := formatTimeRange(ev.StartTime, ev.EndTime, timeLocation)
	ownerName := owner.Displayname
	title := fmt.Sprintf("%s: %s", *ownerName, ev.Title)
	url := fmt.Sprintf("https://www.twitch.tv/events/%s", ev.ID)
	OGImageURL := strings.Replace(ev.ImageURL, "{width}", OGImageWidth, 1)
	OGImageURL = strings.Replace(OGImageURL, "{height}", OGImageHeight, 1)
	twitterImageURL := strings.Replace(ev.ImageURL, "{width}", twitterImageWidth, 1)
	twitterImageURL = strings.Replace(twitterImageURL, "{height}", twitterImageHeight, 1)

	metadata := EventMetadata{
		ID:          ev.ID,
		Title:       title,
		Description: description,

		OGTitle:       title,
		OGDescription: description,
		OGURL:         url,
		OGImage:       OGImageURL,
		OGImageWidth:  OGImageWidth,
		OGImageHeight: OGImageHeight,
		OGType:        openGraphType,

		TwitterCard:        twitterCard,
		TwitterTitle:       title,
		TwitterDescription: description,
		TwitterImage:       twitterImageURL,
		TwitterURL:         url,
	}
	return &metadata, nil
}

func (h *SingleEventHandler) UpdateEvent(ctx context.Context, oldEvent *db.Event, params *EventUpdateParams) (TypedEvent, error) {
	if err := h.validateEventUpdate(ctx, params); err != nil {
		return nil, err
	}
	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbEvent, err := h.OracleDB.UpdateEvent(txCtx, oldEvent.ID, toUpdateDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error updating event")
	}
	event := h.fromDBEvent(dbEvent)

	_, err = h.OracleDB.UpdateBroadcast(txCtx, event.ID, *oldEvent.Language, &db.UpdateDBBroadcastParams{
		Language:  event.Language,
		ChannelID: event.ChannelID,
		GameID:    event.GameID,
		StartTime: event.StartTime,
		EndTime:   event.EndTime,
	})
	if err != nil {
		return nil, errors.Wrap(err, "error updating broadcast for event")
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}

	return event, nil
}

func (h *SingleEventHandler) DeleteEvent(ctx context.Context, oldEvent *db.Event, params *EventDeleteParams) (TypedEvent, error) {
	eventID := oldEvent.ID

	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbEvent, err := h.OracleDB.DeleteEvent(txCtx, eventID, toDeleteDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error deleting event")
	}
	event := h.fromDBEvent(dbEvent)

	_, err = h.OracleDB.DeleteBroadcastsByEventID(txCtx, event.ID)
	if err != nil {
		return nil, errors.Wrap(err, "error updating broadcast for event")
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}

	return event, nil
}

func (h *SingleEventHandler) MarshallEvent(event TypedEvent) ([]byte, error) {
	data, err := json.Marshal(event)
	if err != nil {
		return nil, errors.Wrap(err, "could not marshall event")
	}
	return addPrefix(singleMarshallPrefix, data), nil
}

func (h *SingleEventHandler) UnmarshallEvent(data []byte) TypedEvent {
	d, ok := removePrefix(singleMarshallPrefix, data)
	if !ok {
		return nil
	}
	var event SingleEvent
	err := json.Unmarshal(d, &event)
	if err != nil {
		h.Log.Log("err", err, "could not unmarshall event data")
	}
	return &event
}

func (h *SingleEventHandler) AddLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	event := h.fromDBEvent(dbEvent)

	ctx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(ctx, createdTx)

	_, err = h.validateLocalizationUpdate(ctx, dbEvent, params, true)
	if err != nil {
		return nil, err
	}

	local, err := h.AddLocalizationCommon(ctx, dbEvent, params)
	if err != nil {
		return nil, errors.Wrap(err, "error adding localization")
	}

	if local.ChannelID != nil {
		_, err = h.OracleDB.CreateBroadcast(ctx, &db.CreateDBBroadcastParams{
			EventID:   event.ID,
			Language:  local.Language,
			ChannelID: *local.ChannelID,
			GameID:    event.GameID,
			StartTime: event.StartTime,
			EndTime:   event.EndTime,
		})
	}
	if err != nil {
		return nil, errors.Wrap(err, "error creating broadcast for localization")
	}

	err = h.OracleDB.CommitTx(ctx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving localization")
	}

	return local, nil
}

func (h *SingleEventHandler) UpdateLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	event := h.fromDBEvent(dbEvent)

	ctx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(ctx, createdTx)

	old, err := h.validateLocalizationUpdate(ctx, dbEvent, params, false)
	if err != nil {
		return nil, err
	}

	new, err := h.UpdateLocalizationCommon(ctx, dbEvent, params)
	if err != nil {
		return nil, errors.Wrap(err, "error updating localization")
	}

	if new.ChannelID != nil && old.ChannelID == nil {
		// Added a channel to the existing localization
		_, err = h.OracleDB.CreateBroadcast(ctx, &db.CreateDBBroadcastParams{
			EventID:   event.ID,
			Language:  new.Language,
			ChannelID: *new.ChannelID,
			GameID:    event.GameID,
			StartTime: event.StartTime,
			EndTime:   event.EndTime,
		})
	} else if new.ChannelID == nil && old.ChannelID != nil {
		// Removed a channel from the existing localization
		_, err = h.OracleDB.DeleteBroadcastByEventIDAndLanguage(ctx, event.ID, new.Language)
	} else if new.ChannelID != nil && old.ChannelID != nil && *new.ChannelID != *old.ChannelID {
		// Changed the channel on the existing localization
		_, err = h.OracleDB.UpdateBroadcast(ctx, event.ID, old.Language, &db.UpdateDBBroadcastParams{
			Language:  new.Language,
			ChannelID: *new.ChannelID,
			GameID:    event.GameID,
			StartTime: event.StartTime,
			EndTime:   event.EndTime,
		})
	}
	if err != nil {
		return nil, errors.Wrap(err, "error updating broadcast for localization")
	}

	err = h.OracleDB.CommitTx(ctx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving localization")
	}

	return new, nil
}

func (h *SingleEventHandler) RemoveLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	event := h.fromDBEvent(dbEvent)

	ctx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(ctx, createdTx)

	res, err := h.RemoveLocalizationCommon(ctx, event.ID, params.Language)
	if err != nil {
		return nil, errors.Wrap(err, "error deleting localization")
	}

	_, err = h.OracleDB.DeleteBroadcastByEventIDAndLanguage(ctx, event.ID, params.Language)
	if err != nil {
		return nil, errors.Wrap(err, "error deleting broadcast for localization")
	}

	err = h.OracleDB.CommitTx(ctx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving localization")
	}

	return res, nil
}

func (h *SingleEventHandler) GetArchiveVideos(ctx context.Context, typedEvent TypedEvent) ([]video.ArchiveVideo, error) {
	event, ok := typedEvent.(*SingleEvent)
	if !ok {
		return nil, errors.Errorf("wrong handler for type %s for event id %s", typedEvent.GetType(), typedEvent.GetID())
	}
	return h.VideoFinder.FindArchiveVideos(ctx, event.ChannelID, event.StartTime, event.EndTime)
}
