package types

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/images"
	"code.justin.tv/twitch-events/gea/internal/video"
	usersservice "code.justin.tv/web/users-service/client/usersclient_internal"
)

const (
	EventTypeTimetable      = "timetable"
	timetableMarshallPrefix = "timetable:v1:"
)

// The cache key version number should be bumped when fields are added to the struct
type TimetableEvent struct {
	ID      string `json:"id"`
	OwnerID string `json:"owner_id"`
	Type    string `json:"type"`

	CreatedAt time.Time  `json:"created_at"`
	UpdatedAt time.Time  `json:"updated_at"`
	DeletedAt *time.Time `json:"deleted_at,omitempty"`

	StartTime  *time.Time `json:"start_time,omitempty"`
	EndTime    *time.Time `json:"end_time,omitempty"`
	TimeZoneID string     `json:"time_zone_id"`

	ImageID     *images.ImageID `json:"image_id"`
	ImageURL    string          `json:"image_url"`
	Language    string          `json:"language"`
	Title       string          `json:"title"`
	Description string          `json:"description"`
	ChannelIDs  []string        `json:"channel_ids"`
	GameIDs     []string        `json:"game_ids"`
}

func (e *TimetableEvent) GetType() string {
	return EventTypeTimetable
}

func (e *TimetableEvent) GetID() string {
	return e.ID
}

func (e *TimetableEvent) GetChannelIDs() []string {
	return e.ChannelIDs
}

func (e *TimetableEvent) GetTitle() string {
	return e.Title
}

func (e *TimetableEvent) GetParentID() string {
	return ""
}

func (e *TimetableEvent) GetOwnerID() string {
	return e.OwnerID
}

type TimetableEventHandler struct {
	BaseEventHandler
	ImageURLClient *images.ImageURLClient
	SegmentHandler *SegmentEventHandler
	UserService    usersservice.InternalClient
}

func (h *TimetableEventHandler) Handles(eventType string) bool {
	return eventType == EventTypeTimetable
}

func (h *TimetableEventHandler) CreateEvent(ctx context.Context, params *EventUpdateParams) (TypedEvent, error) {
	err := h.validateEventUpdate(ctx, params)
	if err != nil {
		return nil, err
	}

	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbEvent, err := h.OracleDB.CreateEvent(txCtx, toCreateDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error creating event")
	}
	events, err := h.fromDBEvents(ctx, []*db.Event{dbEvent})
	if err != nil {
		return nil, err
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}

	return events[0], nil
}

func (h *TimetableEventHandler) validateEventUpdate(ctx context.Context, params *EventUpdateParams) error {
	if params.OwnerID == "" {
		return invalid("owner_id", "should have a owner")
	}
	if params.ParentID != nil {
		return invalid("parent_id", "should not have parent")
	}
	if params.StartTime != nil {
		return invalid("start_time", "should not have a start time")
	}
	if params.EndTime != nil {
		return invalid("end_time", "should not have an end time")
	}

	if params.TimeZoneID == nil || *params.TimeZoneID == "" {
		return invalid("time_zone_id", "should have a time zone")
	} else if !isTimeZoneValid(*params.TimeZoneID) {
		return invalid("time_zone_id", "expected a tz database time zone (e.g. America/New_York)")
	}

	if params.Language == nil || *params.Language == "" {
		return invalid("language", "should have a language")
	}
	if params.Title == nil || *params.Title == "" {
		return invalid("title", "should have a title")
	}
	if params.ChannelID != nil {
		return invalid("channel_id", "should not have a channel")
	}
	if params.GameID != nil {
		return invalid("game_id", "should not have a game")
	}
	return nil
}

func (h *TimetableEventHandler) fromDBEvents(ctx context.Context, dbEvents []*db.Event) ([]*TimetableEvent, error) {
	ids := make(map[string]struct{})
	for _, dbEvent := range dbEvents {
		ids[dbEvent.ID] = struct{}{}
	}
	summaries, err := h.SegmentHandler.GetSummaryFields(ctx, keys(ids))
	if err != nil {
		return nil, errors.Wrap(err, "error looking up segment summaries")
	}

	events := make([]*TimetableEvent, 0, len(dbEvents))
	for _, dbEvent := range dbEvents {
		summary := summaries[dbEvent.ID]
		event := &TimetableEvent{
			ID:      dbEvent.ID,
			OwnerID: dbEvent.OwnerID,
			Type:    EventTypeTimetable,

			CreatedAt: dbEvent.CreatedAt,
			UpdatedAt: dbEvent.UpdatedAt,
			DeletedAt: dbEvent.DeletedAt,

			StartTime:  summary.StartTime,
			EndTime:    summary.EndTime,
			TimeZoneID: *dbEvent.TimeZoneID,

			ImageID:     dbEvent.CoverImageID,
			ImageURL:    h.ImageURLClient.GetImageURL(dbEvent.CoverImageID),
			Language:    *dbEvent.Language,
			Title:       *dbEvent.Title,
			Description: coalesce(dbEvent.Description, ""),
			ChannelIDs:  summary.ChannelIDs,
			GameIDs:     summary.GameIDs,
		}
		// Do this because json will marshall nil to null, but an empty array to []
		if event.ChannelIDs == nil {
			event.ChannelIDs = []string{}
		}
		if event.GameIDs == nil {
			event.GameIDs = []string{}
		}
		events = append(events, event)
	}

	return events, nil
}

func (h *TimetableEventHandler) GetEvents(ctx context.Context, dbEvents []*db.Event) ([]TypedEvent, error) {
	events, err := h.fromDBEvents(ctx, dbEvents)
	if err != nil {
		return nil, err
	}
	typedEvents := make([]TypedEvent, 0, len(events))
	for _, event := range events {
		typedEvents = append(typedEvents, event)
	}
	return typedEvents, nil
}

func (h *TimetableEventHandler) GetEventMetadata(ctx context.Context, dbEvent *db.Event) (*EventMetadata, error) {
	events, err := h.fromDBEvents(ctx, []*db.Event{dbEvent})
	if err != nil {
		return nil, err
	}
	if len(events) == 0 {
		return nil, errors.New("Event not found")
	}
	ev := events[0]

	owner, err := h.UserService.GetUserByID(ctx, ev.OwnerID, nil)
	if err != nil {
		return nil, err
	}

	description := ""
	if ev.StartTime != nil && ev.EndTime != nil {
		timeLocation := h.LoadTimeLocation(ctx, ev.TimeZoneID)
		description = formatTimeRange(*ev.StartTime, *ev.EndTime, timeLocation)
	}
	ownerName := owner.Displayname
	title := fmt.Sprintf("%s: %s", *ownerName, ev.Title)
	url := fmt.Sprintf("https://www.twitch.tv/events/%s", ev.ID)
	OGImageURL := strings.Replace(ev.ImageURL, "{width}", OGImageWidth, 1)
	OGImageURL = strings.Replace(OGImageURL, "{height}", OGImageHeight, 1)
	twitterImageURL := strings.Replace(ev.ImageURL, "{width}", twitterImageWidth, 1)
	twitterImageURL = strings.Replace(twitterImageURL, "{height}", twitterImageHeight, 1)

	metadata := EventMetadata{
		ID:          ev.ID,
		Title:       title,
		Description: description,

		OGTitle:       title,
		OGDescription: description,
		OGURL:         url,
		OGImage:       OGImageURL,
		OGImageWidth:  OGImageWidth,
		OGImageHeight: OGImageHeight,
		OGType:        openGraphType,

		TwitterCard:        twitterCard,
		TwitterTitle:       title,
		TwitterDescription: description,
		TwitterImage:       twitterImageURL,
		TwitterURL:         url,
	}
	return &metadata, nil
}

type TimetableInheritedFields struct {
	TimeZoneID string
	Language   string
}

func (h *TimetableEventHandler) GetInheritedFields(ctx context.Context, eventIDs []string) (map[string]*TimetableInheritedFields, error) {
	events, err := h.OracleDB.GetEvents(ctx, eventIDs, false)
	if err != nil {
		return nil, err
	}
	ret := make(map[string]*TimetableInheritedFields, len(events))
	for _, event := range events {
		if event.Type == EventTypeTimetable {
			ret[event.ID] = &TimetableInheritedFields{
				TimeZoneID: *event.TimeZoneID,
				Language:   *event.Language,
			}
		}
	}
	for _, eventID := range eventIDs {
		if _, exists := ret[eventID]; !exists {
			return nil, errors.Errorf("event %s not found", eventID)
		}
	}
	return ret, nil
}

func (h *TimetableEventHandler) UpdateEvent(ctx context.Context, oldEvent *db.Event, params *EventUpdateParams) (TypedEvent, error) {
	if err := h.validateEventUpdate(ctx, params); err != nil {
		return nil, err
	}
	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbEvent, err := h.OracleDB.UpdateEvent(txCtx, oldEvent.ID, toUpdateDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error updating event")
	}
	events, err := h.fromDBEvents(ctx, []*db.Event{dbEvent})
	if err != nil {
		return nil, err
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}

	return events[0], nil
}

func (h *TimetableEventHandler) DeleteEvent(ctx context.Context, oldEvent *db.Event, params *EventDeleteParams) (TypedEvent, error) {
	eventID := oldEvent.ID

	txCtx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(txCtx, createdTx)

	dbEvent, err := h.OracleDB.DeleteEvent(txCtx, eventID, toDeleteDBEventParams(params))
	if err != nil {
		return nil, errors.Wrap(err, "error deleting event")
	}
	events, err := h.fromDBEvents(ctx, []*db.Event{dbEvent})
	if err != nil {
		return nil, err
	}

	err = h.SegmentHandler.DeleteSegmentsByParentID(txCtx, eventID, params)
	if err != nil {
		return nil, errors.Wrap(err, "error deleting segments for event")
	}

	err = h.OracleDB.CommitTx(txCtx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving event")
	}

	return events[0], nil
}

func (h *TimetableEventHandler) MarshallEvent(event TypedEvent) ([]byte, error) {
	data, err := json.Marshal(event)
	if err != nil {
		return nil, errors.Wrap(err, "could not marshall event")
	}
	return addPrefix(timetableMarshallPrefix, data), nil
}

func (h *TimetableEventHandler) UnmarshallEvent(data []byte) TypedEvent {
	d, ok := removePrefix(timetableMarshallPrefix, data)
	if !ok {
		return nil
	}
	var event TimetableEvent
	err := json.Unmarshal(d, &event)
	if err != nil {
		h.Log.Log("err", err, "could not unmarshall event data")
	}
	return &event
}

func (h *TimetableEventHandler) validateLocalizationUpdate(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams, isCreate bool) (*Localization, error) {
	if params.ChannelID != nil {
		return nil, invalid("channel_id", "should not have a channel")
	}
	return h.BaseEventHandler.validateLocalizationUpdate(ctx, dbEvent, params, isCreate)
}

func (h *TimetableEventHandler) AddLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	ctx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(ctx, createdTx)

	_, err = h.validateLocalizationUpdate(ctx, dbEvent, params, true)
	if err != nil {
		return nil, err
	}

	res, err := h.AddLocalizationCommon(ctx, dbEvent, params)
	if err != nil {
		return nil, errors.Wrap(err, "error adding localization")
	}

	err = h.OracleDB.CommitTx(ctx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving localization")
	}

	return res, nil
}

func (h *TimetableEventHandler) UpdateLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	ctx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(ctx, createdTx)

	_, err = h.validateLocalizationUpdate(ctx, dbEvent, params, false)
	if err != nil {
		return nil, err
	}

	new, err := h.UpdateLocalizationCommon(ctx, dbEvent, params)
	if err != nil {
		return nil, errors.Wrap(err, "error updating localization")
	}

	err = h.OracleDB.CommitTx(ctx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving localization")
	}

	return new, nil
}

func (h *TimetableEventHandler) RemoveLocalization(ctx context.Context, dbEvent *db.Event, params *LocalizationUpdateParams) (*Localization, error) {
	ctx, createdTx, err := h.OracleDB.StartOrJoinTx(ctx, nil)
	if err != nil {
		return nil, errors.Wrap(err, "could not start transaction")
	}
	defer h.OracleDB.RollbackTxIfNotCommitted(ctx, createdTx)

	res, err := h.RemoveLocalizationCommon(ctx, dbEvent.ID, params.Language)
	if err != nil {
		return nil, errors.Wrap(err, "error deleting localization")
	}

	err = h.OracleDB.CommitTx(ctx, createdTx)
	if err != nil {
		return nil, errors.Wrap(err, "error saving localization")
	}

	return res, nil
}

func (h *TimetableEventHandler) GetArchiveVideos(ctx context.Context, typedEvent TypedEvent) ([]video.ArchiveVideo, error) {
	// TODO Implement
	_, ok := typedEvent.(*TimetableEvent)
	if !ok {
		return nil, errors.Errorf("wrong handler for type %s for event id %s", typedEvent.GetType(), typedEvent.GetID())
	}
	return make([]video.ArchiveVideo, 0), nil
}
